/**
 * @copyright 2020 Tridium, Inc. All Rights Reserved.
 * @author JJ Frankovich
 */

/**
 * API Status: **Private**
 * @module nmodule/bajaui/rc/fe/CollectionView
 */
define(['baja!', 'log!nmodule.bajaui.rc.fe.CollectionView', 'jquery', 'Promise', 'underscore', 'niagaraSystemProperties', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/wb/table/Table', 'nmodule/webEditors/rc/wb/table/model/TableModel', 'nmodule/webEditors/rc/wb/table/model/columns/PropertyColumn', 'nmodule/webEditors/rc/wb/PropertySheet', 'nmodule/webEditors/rc/fe/BaseWidget', 'nmodule/webEditors/rc/wb/commands/SetDisplayNameCommand', 'nmodule/webEditors/rc/wb/table/menu/DefaultTableContextMenu', 'nmodule/webEditors/rc/fe/feDialogs', 'lex!bajaui', 'nmodule/webEditors/rc/util/htmlUtils', 'css!nmodule/bajaui/rc/bajaui'], function (baja, log, $, Promise, _, niagaraSystemProperties, fe, Table, TableModel, PropertyColumn, PropertySheet, BaseWidget, SetDisplayNameCommand, DefaultTableContextMenu, feDialogs, lexs, htmlUtils) {
  'use strict';

  var lex = lexs[0],
    logSevere = log.severe.bind(log);

  /**
   * CollectionView widget for displaying BITables (usually histories or tabular results from bql queries)
   *
   * @class
   * @alias module:nmodule/bql/rc/fe/CollectionView
   * @extends module:nmodule/webEditors/rc/fe/BaseWidget
   */
  var CollectionView = function CollectionView(moduleName, keyName, formFactor) {
    BaseWidget.call(this, {
      moduleName: moduleName,
      keyName: keyName,
      formFactor: formFactor
    });
    this.$lex = lex;
    this.$maxRows = 1000;
  };
  CollectionView.prototype = Object.create(BaseWidget.prototype);
  CollectionView.prototype.constructor = CollectionView;

  /**
   * @private
   * @return {nmodule/webEditors/rc/wb/table/Table}
   */
  CollectionView.prototype.$getTable = function () {
    return this.$table;
  };

  /**
   * @private
   * @returns {number} the maximum number of rows that should be displayed
   */
  CollectionView.prototype.$getConfiguredLimit = function () {
    return this.$maxRows;
  };

  /**
   *
   * Retrieves the warning text from the lexicon.
   * @returns {string}
   */
  CollectionView.prototype.$getWarningText = function (rows) {
    return lex.get('CollectionView.tableTruncated', rows);
  };

  /**
   * Sets the Display names for the Columns For the PropertySheet in the Dialog.
   * @private
   */
  CollectionView.prototype.$provideDisplayNames = function (subject) {
    this.$columns.map(function (column) {
      SetDisplayNameCommand.setDisplayName(subject, baja.SlotPath.escape(column.getName()), column.getDisplayName());
    });
  };
  CollectionView.prototype.doInitialize = function (dom) {
    var that = this,
      maxRows;
    maxRows = parseInt(niagaraSystemProperties["niagara.webTableMaxLength"]);
    if (maxRows > 0) {
      that.$maxRows = maxRows;
    }
    dom.addClass('CollectionView').html("<div class='CollectionViewHeader'>           \n           <div class='showHideMenu'></div>\n           <div class='CollectionView-warning' style='display: none;'></div>           \n         </div>\n         <div class='CollectionView-tableContainer'>\n           <div class='tableview'>          \n           </div>\n         </div>");
    return fe.makeFor({
      type: Table
    }).then(function (table) {
      that.$table = table;
      that.$defaultTableMenu = new DefaultTableContextMenu(that.$table);
      that.$defaultTableMenu.arm(that.jq(), ".showHideMenu");
      that.$addDialogHandlers();
      return table.initialize(that.jq().find('.tableview'));
    });
  };

  /**
   * Loads data into the view, can be called multiple times.
   * @param {baja.coll.Table} value
   * @returns {Promise}
   */
  CollectionView.prototype.doLoad = function (value) {
    var that = this,
      warning = $(that.jq()).find('.CollectionView-warning');
    warning.empty();
    warning.hide();
    return that.$getRows(value).then(function (rows) {
      that.$columns = value.getColumns();
      return that.$makeModel(rows, that.$columns);
    }).then(function () {
      return that.$table.load(that.$model);
    });
  };

  /**
   * @private
   * @param {baja.coll.Table} table
   * @returns {Promise} which resolves to an array of rows from the table
   */
  CollectionView.prototype.$getRows = function (table) {
    var that = this,
      warning = that.jq().find('.CollectionView-warning'),
      rows = [];
    warning.empty();
    return table.cursor({
      limit: that.$getConfiguredLimit() + 1
    }).then(function (cursor) {
      cursor.each(function (item) {
        if (rows.length >= that.$getConfiguredLimit()) {
          warning.show();
        } else {
          rows.push(item);
        }
      });
      warning.text(that.$getWarningText(rows.length));
      return rows;
    });
  };

  /**
   * adds a model to the widget from the rows and columns
   * @private
   */
  CollectionView.prototype.$makeModel = function (rows, columns) {
    var _this = this;
    return Promise.all(_.map(columns, function (column) {
      var c = new PropertyColumn(baja.SlotPath.escape(column.getName()), {
        displayName: column.getDisplayName(),
        type: column.getType()
      });

      //Since results might be limited, don't allowing sorting since the sort doesn't work with a limit
      c.setSortable(false);

      //ensure the proper TimeZone is loaded from the Database
      var context = column.getFacets().toObject();
      if (context.TimeZone) {
        return baja.TimeZoneDatabase.get().then(function (db) {
          context.TimeZone = db.getTimeZone(context.TimeZone.getId());
          c.data('context', context);
          return c;
        });
      }
      c.data('context', context);
      return c;
    })).then(function (columns) {
      _this.$model = new TableModel({
        columns: columns,
        rows: rows
      });
    });
  };

  /**
   * handler for show details command that displays a details dialog box
   * @private
   * @returns {Promise} a promise that resolves with an instance of the editor in the dialog.
   */
  CollectionView.prototype.$showDetails = function () {
    var selected = this.$getTable().getSelectedRows()[0];
    if (selected) {
      var subject = selected.getSubject().newCopy();
      this.$provideDisplayNames(subject);
      return feDialogs.showFor({
        type: PropertySheet,
        properties: {
          showHeader: false,
          displayOnly: true
        },
        value: subject,
        formFactor: 'mini',
        readonly: true
      });
    } else {
      return Promise.reject(new Error('No selected columns'));
    }
  };

  /**
   * arms the rows of the table with the details dialog
   * @private
   */
  CollectionView.prototype.$addDialogHandlers = function () {
    var that = this;
    that.jq().on('dblclick', 'tr', function () {
      that.$showDetails()["catch"](logSevere);
    });
    htmlUtils.contextMenuOnLongPress(that.jq(), {
      fireClick: true
    });
    that.jq().on('contextmenu', 'tr', function () {
      that.$showDetails()["catch"](logSevere);
      return false;
    });
  };

  /**
   * Destroy all child editors
   *
   * @returns {Promise}
   */
  CollectionView.prototype.doDestroy = function () {
    return this.getChildWidgets().destroyAll();
  };

  /**
   * Set all child editors enabled/disabled.
   *
   * @param {Boolean} enabled
   * @returns {Promise}
   */
  CollectionView.prototype.doEnabled = function (enabled) {
    return this.getChildWidgets().setAllEnabled(enabled);
  };

  /**
   * Set all child editors readonly/writable.
   *
   * @param {Boolean} readonly
   * @returns {Promise}
   */
  CollectionView.prototype.doReadonly = function (readonly) {
    return this.getChildWidgets().setAllReadonly(readonly);
  };
  return CollectionView;
});
