/**
 * @copyright 2019 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * @private
 * @module bajaux/widgets/ToStringWidget
 */
define(['bajaux/Widget', 'Promise', 'underscore'], function (Widget, Promise, _) {
  'use strict';

  var escape = _.escape;

  /**
   * Widget whose only job is to render a string representation of the loaded
   * value.
   *
   * @private
   * @class
   * @alias module:bajaux/widgets/ToStringWidget
   * @extends module:bajaux/Widget
   */
  var ToStringWidget = function ToStringWidget() {
    Widget.apply(this, arguments);
  };
  ToStringWidget.prototype = Object.create(Widget.prototype);
  ToStringWidget.prototype.constructor = ToStringWidget;

  /**
   * Make an editor display-only. `doInitialize()`, `doLoad()`, and `doSave()`
   * will always be overridden with display-only methods. If `updateDisplay()`
   * or `valueToString()` are implemented by the editor, they will be
   * preserved.
   *
   * @private
   * @param {module:bajaux/Widget} ed
   */
  ToStringWidget.$mixin = function (ed) {
    var prot = ToStringWidget.prototype;
    ed.doInitialize = prot.doInitialize;
    ed.doLoad = prot.doLoad;
    ed.doRead = prot.doRead;
    ed.doSave = prot.doSave;
    ed.doDestroy = prot.doDestroy;
    ed.updateDisplay = ed.updateDisplay || prot.updateDisplay;
    ed.valueToString = ed.valueToString || prot.valueToString;
  };

  /**
   * Converts the value to a string using the standard `String()` function.
   * Override to add special to-string logic.
   *
   * @param {*} value
   * @returns {String|Promise<String>}
   */
  ToStringWidget.prototype.valueToString = function (value) {
    return String(value);
  };

  /**
   * Updates the current display. Useful to update the current display
   * whenever the loaded value changes.
   *
   * @param {*} value
   * @returns {Promise|undefined}
   */
  ToStringWidget.prototype.updateDisplay = function (value) {
    var _this = this;
    if (typeof value === 'undefined') {
      value = this.value();
    }
    return Promise.resolve(this.valueToString(value)).then(function (str) {
      return _this.jq().html(toDisplayHtml(str));
    });
  };

  /**
   * Converts the value to a string and sets the element's text value.
   * @param {*} value
   */
  ToStringWidget.prototype.doLoad = function (value) {
    return this.updateDisplay(value);
  };
  function toDisplayHtml(str) {
    return escape(str).replace(/\n/g, '<br>');
  }
  return ToStringWidget;
});
