function _typeof(o) { "@babel/helpers - typeof"; return _typeof = "function" == typeof Symbol && "symbol" == typeof Symbol.iterator ? function (o) { return typeof o; } : function (o) { return o && "function" == typeof Symbol && o.constructor === Symbol && o !== Symbol.prototype ? "symbol" : typeof o; }, _typeof(o); }
/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * Utility methods used for managing bajaux containers.
 * 
 * @module bajaux/container/util
 * @private
 */
define(["Promise", "bajaux/Widget", "bajaux/util/CommandButtonGroup"], function (Promise, Widget, CommandButtonGroup) {
  "use strict";

  var exports = {},
    hideCommandBar = "hideCommandBar",
    baja;

  /**
   * Decode a value that's been passed in as a parameter.
   * Please note, this will only work if this Properties instance contains
   * the Property of the specified name and the Property has the 'parameter' attribute
   * set to true. If these conditions aren't met, the return value will be null.
   *
   * @param {module:bajaux/Properties} properties
   * @param {String} name The name of the Property being used to decode.
   * @param {String} value The value currently being decoded.
   * @returns {number|boolean|string|null} The decoded Property value or null.
   */
  exports.decodeValueFromParameter = function (properties, name, value) {
    var to;
    if (properties.has(name) && properties.get(name, "parameter", false)) {
      // Decode the parameter value (it's easier doing this in JavaScript).
      value = decodeURIComponent(value);

      // Check the value of the current Property as we need to try and match it.
      to = _typeof(properties.getValue(name));
      if (to === "number") {
        value = parseFloat(value);
      } else if (to === "boolean") {
        if (typeof value === "string") {
          value = value === "true";
        } else {
          value = !!value;
        }
      }
    } else {
      value = null;
    }
    return value;
  };

  /**
   * The hide command bar property name.
   * 
   * @type {String}
   */
  exports.hideCommandBar = hideCommandBar;

  /**
   * Toggle the Command Bar's visibility on/off.
   * 
   * @param  widget The Widget instance.
   * @param  {JQuery} toolbarJq The jQuery wrapped toolbar DOM.
   */
  exports.toggleCommandBar = function (widget, toolbarJq) {
    var hide = widget.properties().getValue(hideCommandBar);

    // If a property hasn't been specified then base the command bar showing
    // on whether there are any commands in the widget.
    if (typeof hide !== "boolean") {
      hide = widget.getCommandGroup().isEmpty();
    }
    toolbarJq.toggle(!hide).toggleClass('hidden', hide);
  };
  function $initCommandBar(widget, containerJq) {
    var headerJq = containerJq.children(".bajaux-toolbar-outer"),
      toolbarJq = headerJq.children(".bajaux-toolbar"),
      contentJq = containerJq.children(".bajaux-widget-container"),
      cg = toolbarJq.data("widget");
    return Promise.resolve(cg && typeof cg.destroy === "function" && cg.destroy()).then(function () {
      exports.toggleCommandBar(widget, toolbarJq);
      var group = new CommandButtonGroup({
        properties: {
          toolbar: true
        }
      });
      return group.initialize(toolbarJq).then(function () {
        return group.load(widget.getCommandGroup());
      }).then(function () {
        function layoutContent() {
          contentJq.css('margin-top', headerJq.height());
        }
        if (containerJq.is('.ux-fullscreen-support, .ux-fullscreen-support .ux-fullscreen') && !widget.properties().getValue(hideCommandBar, false)) {
          if (!widget.$layoutContent) {
            var layout = widget.layout;
            widget.layout = widget.$layoutContent = function () {
              // Layout the content first and then perform the widget layout.
              layoutContent();
              return layout.apply(this, arguments);
            };
          }
        }
        layoutContent();
      });
    });
  }

  /**
   * Initializes the Command Bar.
   *
   * Please note, this will destroy any old Command Bar it finds before
   * creating the new one.
   * 
   * @param  widget The Widget instance.
   * @param  {JQuery} toolbarJq The jQuery wrapped toolbar DOM.
   * @returns {Promise} The promise that's resolved once the 
   * Command Bar has been initialized.
   */
  exports.initCommandBar = function (widget, toolbarJq) {
    var promise = toolbarJq.data("initCommandBarPromise") || Promise.resolve(),
      initFunc = function initFunc() {
        return $initCommandBar(widget, toolbarJq);
      };

    // Only allow the command bar to initialize one at a time.    
    promise = promise.then(initFunc, initFunc);
    toolbarJq.data("initCommandBarPromise", promise);
    return promise;
  };

  /**
   * Pre-destroy timeout. This has been exposed for unit testing.
   * 
   * @type {Number}
   */
  exports.$preDestroyTimeout = 10000;

  /**
   * Called just before a widget is destroyed. This method prevents 'unsubscribe'
   * network calls from happening when a widget is destroyed. Instead it delays the
   * unsubscription network calls until a later time. Providing the widget in question
   * is using its subscriber for all subscriptions, this should speed up destroying
   * this widget.
   * 
   * @param {module:bajaux/Widget} widget A widget that's about to be destroyed.
   */
  exports.preDestroy = function (widget) {
    var sub;
    baja = baja || require.defined("baja!") ? require("baja!") : null;
    if (widget.hasMixIn("subscriber") && baja && typeof widget.getSubscriber === "function" && typeof widget.getSubscriber().getComponents === "function") {
      sub = new baja.Subscriber();
      sub.subscribe(widget.getSubscriber().getComponents());
      baja.clock.schedule(function () {
        sub.unsubscribeAll();
      }, exports.$preDestroyTimeout);
    }
  };
  return exports;
});
