/**
 * @license Copyright 2012, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview MobileGridPane class
 * 
 * @author Gareth Johnson
 * @version 0.0.2.0
 */
define(['baja!', 'jquery', 'jquerymobile', 'mobile/px/util.px', 'mobile/px/widgets/Widget'], function (baja, $, jqm, pxUtil, Widget) {
  "use strict";

  var widgetType = pxUtil.getWidgetType(),
    callSuper = baja.callSuper;

  /**
   * @class MobileGridPane.
   * 
   * The mobile pane Widget that represents Niagara's 'mobile:MobileGridPane' Type.
   * 
   * @name MobileGridPane
   * @extends Widget
   */
  function MobileGridPane() {
    callSuper(MobileGridPane, this, arguments);
    this.$kids = null;
    this.$tableDom = null;
  }
  baja.subclass(MobileGridPane, Widget);

  /**
   * Append the Widget's DOM content onto the specified DOM element.
   *
   * @param {Object} dom DOM element to append content too.
   */
  MobileGridPane.prototype.load = function (dom) {
    var colCount = this.getColumnCount(),
      colPercent = 100 / colCount,
      count = 0,
      html = "<div data-role='none' class='mobileGridPane'><table data-role='none' class='mobileGridPaneTable' cellpadding='0' cellspacing='0'><tbody>",
      kids = this.$kids = [],
      tableDiv;

    // Create html for table...
    this.getSlots().properties().is(widgetType).eachValue(function (widget) {
      kids.push(widget);
      if (count === 0) {
        html += "<tr>";
      }
      html += "<td class='mobileGridPaneData' style='width: " + colPercent + "%;'></td>";
      if (++count === colCount) {
        html += "</tr>";
        count = 0;
      }
    });
    if (count > 0 && kids.length > 0) {
      html += "</tr>";
    }
    html += "</tbody></table></div>";

    // Create and attach to the DOM
    tableDiv = $(html);
    tableDiv.appendTo(dom);
    this.$dom = tableDiv;
    this.$tableDom = tableDiv.children(".mobileGridPaneTable");
  };

  /**
   * Append the Widget's DOM content onto the specified DOM element.
   */
  MobileGridPane.prototype.loadChildren = function () {
    // Build up the HTML for the jQuery Mobile column grid
    var kids = this.$kids,
      i = 0;
    this.$tableDom.find("td.mobileGridPaneData").each(function () {
      kids[i].load($(this));
      kids[i++].loadChildren();
    });

    // Listen for any updates to any vertical alignment changes... 
    baja.iterate(kids, function (k) {
      if (k.getDomElement() && k.has("valign")) {
        var update = k.update;
        k.update = function (hasUpdate) {
          // First call the original update
          return update.apply(k, arguments).then(function () {
            if (hasUpdate("valign")) {
              // If the vertical alignment is being updated then
              var kd = k.getDomElement(),
                td = kd.parent();
              td.css("vertical-align", kd.css("vertical-align"));
            }
          });
        };
      }
    });
  };

  /**
   * Update the DOM associated with this Widget.
   *
   * @param {Function} hasUpdate called to test whether a 
   *                               given Property can update the DOM.
   */
  MobileGridPane.prototype.doUpdate = function (hasUpdate) {
    var dom = this.$dom,
      tableDom = this.$tableDom,
      rows,
      gap;

    // Update Row Gap
    if (hasUpdate("rowGap")) {
      gap = this.getRowGap() / 2 + "px";
      rows = tableDom.children("tbody").children("tr");

      // Apply top and bottom margin to all data
      rows.children("td").css({
        "padding-top": gap,
        "padding-bottom": gap
      });

      // No top margin on first row
      rows.first().children("td").css("padding-top", "0px");

      // No bottom margin on last row
      rows.last().children("td").css("padding-bottom", "0px");
    }

    // Update Column Gap
    if (hasUpdate("columnGap")) {
      gap = this.getColumnGap() / 2 + "px";
      rows = tableDom.children("tbody").children("tr");

      // Apply top and bottom margin to all data
      rows.children("td").css({
        "padding-left": gap,
        "padding-right": gap
      });

      // No left margin on first td in a row
      rows.children("td:first-child").css("padding-left", "0px");

      // No right margin on last td in a row
      rows.children("td:last-child").css("padding-right", "0px");
    }

    // Padding
    if (hasUpdate("padding")) {
      this.applyCssFromSlot("padding", dom);
    }

    // Rounded corners
    if (hasUpdate("roundedCorners")) {
      if (this.getRoundedCorners()) {
        dom.css({
          "-moz-border-radius": ".6em",
          "border-radius": ".6em"
        });
      } else {
        dom.css({
          "-moz-border-radius": "",
          "border-radius": ""
        });
      }
    }

    // Shadow
    if (hasUpdate("shadow")) {
      if (this.getShadow()) {
        dom.css({
          "-moz-box-shadow": "0 0 5px #888",
          "-webkit-box-shadow": "0 0 5px #888",
          "box-shadow": "0 0 5px #888"
        });
      } else {
        dom.css({
          "-moz-box-shadow": "",
          "-webkit-box-shadow": "",
          "box-shadow": ""
        });
      }
    }

    // Background (will override theme's background color)
    if (hasUpdate("background")) {
      this.applyCssFromSlot("background", {
        dom: dom,
        colorSelector: "background",
        imageSelector: "background"
      });
    }

    // Foreground (will color the border)
    if (hasUpdate("foreground")) {
      this.applyCssFromSlot("foreground", {
        dom: dom,
        colorSelector: "border",
        prefix: "1px solid "
      });
    }
  };
  return MobileGridPane;
});
