/**
 * @license Copyright 2012, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview Brush class
 * 
 * @author Gareth Johnson
 * @version 0.0.2.0
 */
define(['baja!', 'mobile/px/gx/Color', 'mobile/px/gx/gxStyles', 'mobile/util/gradient'], function (baja, Color, gxStyles, gradientUtil) {
  "use strict";

  var linearGradientSelections = ["-webkit-linear-gradient", "-moz-linear-gradient", "-o-linear-gradient", "-ms-linear-gradient"
    //"linear-gradient" leave out and rely on
    // vendor specific gradients
    // for now as this causes some issues with the latest versions
    // of Firefox.
    ],
    radialGradientSelections = ["-webkit-radial-gradient", "-moz-radial-gradient", "-o-radial-gradient", "-ms-radial-gradient"
    //"radial-gradient" leave out and rely on
    // vendor specific gradients
    // for now as this causes some issues with the latest versions
    // of Firefox.
    ];

  // Solid color brush
  function Solid(color) {
    // Update the DOM element
    this.update = function (obj) {
      color.update(obj);
    };
    this.encodeToString = function () {
      return color.encodeToString();
    };
  }

  // Image Brush
  function Image(encoding) {
    var str = encoding,
      res,
      regex = /([a-zA-Z]*)\(([^()]*)\)/g,
      params = {};

    // Remove image(...)        
    str = str.substring(6, str.length - 1);
    res = regex.exec(str);
    while (res) {
      params[res[1]] = res[2];
      res = regex.exec(str);
    }

    // Ensure all image variables are initialized
    params.tile = params.tile || "false";
    params.halign = params.halign || "center";
    params.valign = params.valign || "center";

    // Update the DOM element with the image
    this.update = function (obj) {
      if (obj.imageSelector && obj.imageSelector === "background") {
        var css = "url(/ord?" + params.source + ") ";
        if (params.tile === "true") {
          css += "repeat ";
        } else if (params.tile === "false") {
          css += "no-repeat ";
        } else if (params.tile === "x") {
          css += "repeat-x ";
        } else if (params.tile === "y") {
          css += "repeat-y ";
        }
        css += params.halign + " " + params.valign;
        obj.dom.css(obj.imageSelector, css);
      }
    };
    this.encodeToString = function () {
      return encoding;
    };
  }
  function parseStop(res) {
    // Parse out 'offset color'
    var stopSplitStr = res[2].split(" "),
      colorCss = Color.DEFAULT.decodeFromString(stopSplitStr[1]).getCss(),
      offset = stopSplitStr[0];
    return {
      color: colorCss,
      offset: offset,
      toString: function toString() {
        return colorCss + " " + offset;
      }
    };
  }

  /**
   * @private
   * @inner
   * @ignore
   * @param {String} encoding a String encoding of a BBrush.LinearGradient
   * @returns {Object} an object with <code>angle<code> and <code>stops</code>
   * properties. The stops are a two-dimensional array in the form
   * <code>[["0%", "color1"], ["100%", "color2"]]<code>
   */
  function parseAngleAndStops(encoding) {
    var str = encoding,
      regex = /([a-zA-Z]*)\(([^()]*)\)/g,
      stops = [],
      stop,
      angle = 0,
      res;

    // Remove linearGradient(...)        
    str = str.substring(15, str.length - 1);
    res = regex.exec(str);
    while (res) {
      // We need to decode the stops in order
      if (res[1] === "stop") {
        // Parse out offset color
        stop = parseStop(res);
        stops.push([stop.offset, stop.color]);
      } else if (res[1] === "angle") {
        angle = parseInt(res[2], 10);
      }
      res = regex.exec(str);
    }

    // Build base CSS
    return {
      angle: angle,
      stops: stops
    };
  }

  // Linear Gradient Brush
  function LinearGradient(encoding) {
    var as = parseAngleAndStops(encoding),
      angle = as.angle,
      stops = as.stops;
    this.update = function (obj) {
      var css = gradientUtil.createLinearGradientCss(angle, stops),
        selector = obj.imageSelector,
        prefix = obj.prefix || "",
        dom = obj.dom,
        i;
      if (css !== null) {
        // Apply the first color for older browser support
        dom.css(selector, prefix + css.background);

        // Add limited IE8/9 Support for background color only
        if (selector === "background") {
          dom.css("filter", css.filter);
        }
        dom.css(selector, "-webkit-gradient" + css.webkitGradient);
        for (i = 0; i < linearGradientSelections.length; ++i) {
          dom.css(selector, linearGradientSelections[i] + css.linearGradientNonStandard);
        }
        dom.css(selector, "linear-gradient" + css.linearGradient);
      }
    };
    this.encodeToString = function () {
      return encoding;
    };
  }

  // Radial Gradient Brush
  function RadialGradient(encoding) {
    var str = encoding,
      regex = /([a-zA-Z]*)\(([^()]*)\)/g,
      stops = [],
      center = "0% 0%",
      css,
      firstColor,
      res;

    // Remove radialGradient(...)        
    str = str.substring(15, str.length - 1);
    res = regex.exec(str);
    while (res) {
      // We need to decode the stops in order
      if (res[1] === "stop") {
        // Parse out offset color
        stops.push(parseStop(res));
      } else if (res[1] === "c") {
        center = res[2];
      }
      res = regex.exec(str);
    }

    // Build base CSS
    css = "(" + center + ", circle, " + stops.join(", ") + ")";
    firstColor = stops.length > 0 ? stops[0].color : "#FFFFFF";
    this.update = function (obj) {
      // Apply the first color for older browser support (< IE10)
      obj.dom.css(obj.imageSelector, firstColor);
      var i;
      for (i = 0; i < radialGradientSelections.length; ++i) {
        obj.dom.css(obj.imageSelector, radialGradientSelections[i] + css);
      }
    };
    this.encodeToString = function () {
      return encoding;
    };
  }

  // Parse a String and return a paint Object
  function parseBrush(encoding) {
    if (encoding.match(/^image\(/)) {
      return new Image(encoding);
    } else if (encoding.match(/^linearGradient\(/)) {
      return new LinearGradient(encoding);
    } else if (encoding.match(/^radialGradient\(/)) {
      return new RadialGradient(encoding);
    }
    return new Solid(Color.DEFAULT.decodeFromString(encoding));
  }

  // TODO: We need to support images here.

  /**
   * @class Brush.
   * 
   * This is currently a partial implementation of Niagara's 'gx:Brush' Type.
   * 
   * Please note, this isn't a complete implementation but will do for the Px App.
   * When creating a Simple, always use the 'make' method instead of creating a new Object.
   *
   * @name Brush
   * @extends baja.Simple
   */
  function Brush(paint) {
    baja.callSuper(Brush, this, arguments);
    this.$paint = paint;
  }
  baja.subclass(Brush, baja.Simple);

  /**
   * Make a Brush
   *
   * @param {Object} obj Object Literal for the method's arguments.
   * @param {String} [obj.str] the String encoding for a Brush.
   * @param {Color} [obj.color] the solid color for the brush.
   * @return {Brush}
   */
  Brush.make = function (obj) {
    obj = baja.objectify(obj, "str");
    if (obj.str === "" || obj.str === "null") {
      return Brush.NULL;
    }
    var paint;
    if (obj.str) {
      paint = parseBrush(obj.str);
    } else if (obj.color) {
      paint = new Solid(obj.color);
    }
    return new Brush(paint);
  };

  /**
   * Make a Brush
   *
   * @param {Object} obj Object Literal for the method's arguments.
   * @param {String} [obj.str] the String encoding for a Brush.
   * @param {Color} [obj.color] the solid color for the brush.
   * @return {Brush}
   */
  Brush.prototype.make = function (obj) {
    return Brush.make.apply(Brush, arguments);
  };

  /**
   * Decode a Brush from a String
   * 
   * @name Brush#decodeFromString
   * @function
   *
   * @param {String} str
   * @return {Brush}
   */
  Brush.prototype.decodeFromString = function (str) {
    return Brush.make(str);
  };

  /**
   * Encode the Brush to a String
   *
   * @return {String}
   */
  Brush.prototype.encodeToString = function () {
    return this.$paint.encodeToString();
  };

  /**
   * Default Brush instance
   */
  Brush.DEFAULT = new Brush(new Solid(Color.DEFAULT));

  /**
   * Null Brush instance
   */
  Brush.NULL = new Brush(new Solid(Color.NULL));

  /**
   * Update an HTML DOM element.
   *
   * @param {Object} obj contains method arguments.
   */
  Brush.prototype.update = function (obj) {
    return gxStyles.applyBrush.apply(gxStyles, [this].concat(Array.prototype.slice.call(arguments)));
  };
  return Brush;
});
