/**
 * @copyright 2019 Tridium, Inc. All Rights Reserved.
 */

/**
 * @private
 * @module nmodule/jsonToolkit/rc/fe/QueryPicker
 */
define([
    'baja!',
    'baja!jsonToolkit:JsonSchema',
    'log!nmodule.jsonToolkit.rc.fe.QueryPicker',
    'Promise',
    'underscore',
    'bajaux/mixin/subscriberMixIn',
    'nmodule/webEditors/rc/fe/baja/util/DepthSubscriber',
    'nmodule/webEditors/rc/fe/baja/util/compUtils',
    'nmodule/webEditors/rc/fe/baja/BaseEditor',
    'css!nmodule/jsonToolkit/rc/jsonToolkit'
  ], function (
    baja,
    types,
    log,
    Promise,
    _,
    subscriberMixIn,
    DepthSubscriber,
    compUtils,
    BaseEditor
  ) {

  'use strict';

  var logFine = log.fine.bind(log),
     closest = compUtils.closest;


  /**
   * A field editor for working with the 'query' ('baja:String') slot on a
   * 'jsonToolkit:JsonSchemaBoundQueryResult' component
   *
   *
   * @private
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/jsonToolkit/rc/fe/QueryPicker
   */
  var QueryPicker = function QueryPicker(params) {
    BaseEditor.call(this, _.extend({
      keyName: 'QueryPicker'
    }, params));
    this.$subscriber = new DepthSubscriber(5);

    subscriberMixIn(this);
  };
  QueryPicker.prototype = Object.create(BaseEditor.prototype);
  QueryPicker.prototype.constructor = QueryPicker;

  /**
   * Initializes this editor.
   *
   * @param {JQuery} dom
   */
  QueryPicker.prototype.doInitialize = function (dom) {
    var that = this;

    dom.addClass('QueryPicker');

    dom.html(
      '<span><select class="query-selection" /></span>'
    );

    dom.on('change', '.query-selection', function (e) {
      that.setModified(true);
      e.preventDefault();
      return false;
    });
  };

  /**
   * loads the queryName into the editor
   *
   * @param {String} queryName
   * @returns {Promise}
   */
  QueryPicker.prototype.doLoad = function (queryName) {
    return this.$populateSelectOptions();
  };


  /**
   * populates the dropdown
   *
   * @private
   * @returns {Promise}
   */
  QueryPicker.prototype.$populateSelectOptions = function () {
    var that = this,
      selectElement = that.$getSelect();

    // clear out the dropdown options
    that.$getSelect().empty();

    // add a blank element to use if the current value no longer exists,
    // this element will be removed it the current value is an option.
    selectElement.append(
      '<option class="ux-option blank-option" value=""></option>'
    );
    // and make it the selected one.
    selectElement.val('');


    return that.$getQueryOptions().then(function (queryOptions) {
      queryOptions.forEach(function (queryName) {
        selectElement.append(
          '<option class="ux-option" value="' + _.escape(queryName) + '">' + _.escape(baja.SlotPath.unescape(queryName)) + '</option>'
        );
        if (queryName === that.value()) {
          // if this matches the current value, set it as the selected option
          selectElement.val(queryName);
          // remove the blank option
          that.jq().find('.blank-option').remove();
        }
      });
    });
  };

  /**
   * construct a list of available query names
   *
   * @private
   * @returns {Promise.<Array.<String>>}
   */
  QueryPicker.prototype.$getQueryOptions = function () {
    var queryOptions = [],
      // get the parent jsonSchema
      parentSchema = this.$getParentSchema(),
      sub = this.getSubscriber();


    if (!parentSchema) { return Promise.resolve(queryOptions); }

    sub.subscribe(parentSchema);

    // There appears to be a scenario where the navOrd is not populated.
    // A screen refresh appears to sort it out. The same thing happens in workbench
    // so we will just catch the 'error'
    if (!parentSchema.getQueries().getNavOrd()) {
      logFine('NavOrd not populated');
      return Promise.resolve(queryOptions);
    }

    return parentSchema.getQueries().getNavOrd().get({ subscriber: sub })
      .then(function (leasedComp) {
        leasedComp.getSlots().properties().is('jsonToolkit:JsonSchemaQuery').each(function (prop) {
          queryOptions.push(prop.getName());
        });
        return queryOptions;
      });
  };

  /**
   * returns the JsonSchema parent for this complex
   *
   * @private
   * @returns {baja.Component}
   */
  QueryPicker.prototype.$getParentSchema = function () {
    return closest(this.getComplex(), 'jsonToolkit:JsonSchema');
  };

  /**
   * Reads the select element and return the currently selected value
   *
   * @returns {Promise.<String>}
   */
  QueryPicker.prototype.doRead = function () {
    return this.$getSelect().val();
  };

  /**
   * Get the select element.
   *
   * @private
   * @returns {JQuery}
   */
  QueryPicker.prototype.$getSelect = function () {
    return this.jq().find('.query-selection');
  };

  /**
   * Enables or disables the select dropdown.
   *
   * @param {Boolean} enabled
   */
  QueryPicker.prototype.doEnabled = function (enabled) {
    this.$getSelect().prop('disabled', this.isReadonly() || !enabled);
  };

  /**
   * Disables or enables the select dropdown.
   *
   * @param {Boolean} readonly
   */
  QueryPicker.prototype.doReadonly = function (readonly) {
    this.$getSelect().prop('disabled', !this.isEnabled() || readonly);
  };

  /**
   * Clean up after this editor
   * @returns {Promise}
   */
  QueryPicker.prototype.doDestroy = function () {
    this.jq().removeClass('QueryPicker');
  };

  return (QueryPicker);
});
