/* global requirejs, window, niagara, setTimeout */
/**
 * @copyright 2019 Tridium, Inc. All Rights Reserved.
 */

/**
 * @private
 * @module nmodule/jsonToolkit/rc/fe/JsonOutputEditor
 */
requirejs.config({
  paths: {
    // additional resources, not in std js-ux ace implementation
    'ace/theme': '/module/jsonToolkit/lib/ace/theme',
    'ace/mode': '/module/jsonToolkit/lib/ace/mode',
    'ace/requirejs': '/module/jsonToolkit/lib/ace/requirejs'
  }
});

define([
    'baja!',
    'baja!jsonToolkit:JsonSchema',
    'log!nmodule.jsonToolkit.rc.fe.JsonOutputEditor',
    'jquery',
    'underscore',
    'Promise',
    'dialogs',
    'bajaux/mixin/subscriberMixIn',
    'nmodule/webEditors/rc/fe/baja/util/DepthSubscriber',
    'nmodule/js/rc/asyncUtils/asyncUtils',
    'nmodule/webEditors/rc/fe/baja/BaseEditor',
    'bajaux/commands/Command',
    'bajaux/commands/CommandGroup',
    'bajaux/util/CommandButtonGroup',
    'ace/ace',
    'lex!jsonToolkit',
    'hbs!nmodule/jsonToolkit/rc/fe/template/loader', // a copy of hbs!nmodule/js/rc/dialogs/loader
    'css!nmodule/jsonToolkit/rc/jsonToolkit'
  ], function (
    baja,
    types,
    log,
    $,
    _,
    Promise,
    dialogs,
    subscriberMixIn,
    DepthSubscriber,
    asyncUtils,
    BaseEditor,
    Command,
    CommandGroup,
    CommandButtonGroup,
    ace,
    lexs,
    loadingImg
  ) {

  'use strict';

  var jsonToolkitLex = lexs[0],
     logError = log.severe.bind(log),
     logFine = log.fine.bind(log),
     waitForTrue =  asyncUtils.waitForTrue,
     INITIAL_WAIT_MS = 25000,
     FINAL_WAIT_MS = 5000;

  var SINGLE_LINE_ICON = jsonToolkitLex.get('viewToggleBtn.icon.singleLine'),
    MULTI_LINE_ICON = jsonToolkitLex.get('viewToggleBtn.icon'),
    SINGLE_LINE_NAME = jsonToolkitLex.get('viewToggleBtn.label.raw'),
    MULTI_LINE_NAME = jsonToolkitLex.get('viewToggleBtn.displayName'),
    DEFAULT_ROWS = 12,
    MAX_ROWS = 25;



  /**
   * A field editor for working with the 'query' ('baja:String') slot on a
   * 'jsonToolkit:JsonSchemaBoundQueryResult' component
   *
   *
   * @private
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/jsonToolkit/rc/fe/JsonOutputEditor
   */
  var JsonOutputEditor = function JsonOutputEditor(params) {
    BaseEditor.call(this, _.extend({
      keyName: 'JsonOutputEditor'
    }, params));

    this.$subscriber = new DepthSubscriber(5);

    subscriberMixIn(this);
  };

  JsonOutputEditor.prototype = Object.create(BaseEditor.prototype);
  JsonOutputEditor.prototype.constructor = JsonOutputEditor;

  /**
   * Initializes this editor.
   *
   * @param {JQuery} dom
   */
  JsonOutputEditor.prototype.doInitialize = function (dom) {
    var that = this;

    dom.addClass('JsonOutputEditor');

    // default to multiline format
    this.$setMultiLineFormat(true);

    // Add an ace editor to the dom
    that.$editor = ace.edit(
      $('<div class="aceEditor" />').appendTo(dom).get(0)
    );

    that.$setUpEditor();

    return that.$makeCommandGroup(dom)
      .then(function (value) {
        that.$enableCommands(that.$hasOpInvoke());
      });
  };

  /**
   * creates a command group, populates it with buttons and adds it to the dom
   *
   * @private
   * @returns {Promise}
   */
  JsonOutputEditor.prototype.$makeCommandGroup = function (dom) {
    var that = this;

    // add a column for command buttons
    var commandsCol = $('<div class="commandsCol"/>').appendTo(dom);

    var generateJsonCmd = new Command({
      module: 'jsonToolkit',
      lex: 'generateJsonBtn',
      func: _.bind(that.$forceGenerateJson, that)
    });

    var clearBtn = new Command({
      module: 'jsonToolkit',
      lex: 'clearBtn',
      func: _.bind(that.$clearOutput, that)
    });

    var metricsBtn = new Command({
      module: 'jsonToolkit',
      lex: 'metricsBtn',
      func: _.bind(that.$openMetrics, that)
    });

    var viewToggleBtn = new Command({
      module: 'jsonToolkit',
      lex: 'viewToggleBtn',
      func: _.bind(that.$toggleView, that)
    });

    that.$commandButtonGroup = new CommandButtonGroup();

    return that.$commandButtonGroup.initialize(commandsCol)
      .then(function (value) {
        return that.$commandButtonGroup.load(
          new CommandGroup({
            commands: [
              generateJsonCmd,
              clearBtn,
              metricsBtn,
              viewToggleBtn
            ]
          })
        );
      });
  };

  /**
   * whether operator invoke permission is available on the this editor's complex
   *
   * @private
   */
  JsonOutputEditor.prototype.$hasOpInvoke = function () {
    return this.getComplex().getPermissions().hasOperatorInvoke();
  };


  /**
   * sets up the ace editor
   *
   * @private
   */
  JsonOutputEditor.prototype.$setUpEditor = function () {
    var editor = this.$getEditor();

    editor.setReadOnly(true);

    // stops line highlighting
    editor.setHighlightActiveLine(false);
    editor.setHighlightGutterLine(false);

    // makes the cursor invisible
    editor.renderer.$cursorLayer.element.style.opacity = 0;
    // prevents text selection
    //editor.container.style.pointerEvents = "none";

    // removes the gutter, and therefore the line numbers
    editor.renderer.setShowGutter(false);
    // removes the line numbers
    //editor.renderer.setOption('showLineNumbers', false);

    editor.renderer.setOptions({
      maxLines: MAX_ROWS,
      minLines: DEFAULT_ROWS
    });

    editor.setTheme('ace/theme/monokai'); // dark background theme

    editor.session.setMode('ace/mode/json'); // adds the syntax highlighting

    /*   While running the tests this warning appears:
     *   WARN: 'Automatically scrolling cursor into view after selection change',
     *       'this will be disabled in the next version',
     *       'set editor.$blockScrolling = Infinity to disable this message'
     *   Contrary to the warning this does not disable the messages.
     */
    // editor.$blockScrolling = Infinity;
  };

  /**
   * return the ace editor
   *
   * @private
   * @returns {ace.Editor}
   */
  JsonOutputEditor.prototype.$getEditor = function () {
    return this.$editor;
  };

  /**
   * regenerates the json.
   *
   * @private
   */
  JsonOutputEditor.prototype.$forceGenerateJson = function () {

    var that = this;
    var initialSchemaGenerations = -1;

    dialogs.show({
      title: jsonToolkitLex.get('generating'),
      content: function (dlg, jq) {
        var parentSchema = that.$getParentSchema(),
          dialogJq = jq,
          dlgHeader = dialogJq.parent().parent().find('.js-dialog-header');

        dialogJq.html("<img class='js-dialog-loading-img' src='" + loadingImg() + "'/>");
        dlgHeader.addClass('dialog-generateJson');

        function waitInterval(ms) {
          return new Promise(function (resolve) { setTimeout(resolve, ms); });
        }

        that.$getInitialSchemaGenerations()
          .then(function (initial) {
            initialSchemaGenerations = initial;

            // forceGenerateJson refreshes the json on the component, this will
            // then increment its schemaGenerations, and that will in turn fire
            // the changed call back set up in #$getInitialSchemaGenerations.
            return Promise.all([
              parentSchema && parentSchema.invoke({ slot: 'forceGenerateJson' }),
              // make sure the dialog appears for half a second, so it doesn't
              // 'flash' on and off
              waitInterval(500)
            ]);
          })
          .then(function () {
            //return waitInterval(5000).then(function () { // force a pause for test purposes
            return that.$waitForSchemaIncremented(initialSchemaGenerations, INITIAL_WAIT_MS)
             .catch(function () {
                dlgHeader.text(jsonToolkitLex.get('generateJsonBtn.still.waiting'));
              //   return waitInterval(5000).catch(function (ignore) {}); // force another pause for test purposes
              // })
              // .then(function () {
                // try again for the final few seconds
                return that.$waitForSchemaIncremented(initialSchemaGenerations, FINAL_WAIT_MS);
              });
          })
          .finally(function () {
            dlg.close();
          })
          .catch(function (err) {
            baja.error(err);
            dialogs.showOk({
              title: jsonToolkitLex.get('generateJsonBtn.error'),
              content: err
            });
          });
      }
    });
  };

  /**
   * resolves when the underlying value for schemaGenerations has changed,
   * rejects if this doesn't happen within the maxWait time
   *
   * @private
   * @param {baja.Long} initialValue the value to test against.
   * @param {Number} maxWait the number of milliseconds to wait.
   * @returns {Promise}
   */
  JsonOutputEditor.prototype.$waitForSchemaIncremented = function (initialValue, maxWait) {
    var that = this;
    return waitForTrue(function () {
      return initialValue !== that.$schemaGenerations;
    }, maxWait);
  };


  /**
   * @private
   * @param {baja.Property} prop the property that has changed.
   */
  JsonOutputEditor.prototype.$changedHandler = function (prop) {
    var that = this;
    if (prop.getName() === 'schemaGenerations') {
      that.$schemaGenerations = that.$getParentSchema().getConfig().getDebug().getMetrics().get(prop);
    }
  };

  /**
   * initializes the schemaGenerations and subscribes to any changes in its value
   *
   * @private
   * @returns {Promise}
   */
  JsonOutputEditor.prototype.$getInitialSchemaGenerations = function () {
    var that = this,
      sub = that.getSubscriber();

    sub.subscribe(that.$getParentSchema());
    sub.attach('changed',
      _.bind(that.$changedHandler, that)
    );

    return baja.Ord.make({
      base: that.$getParentSchema().getNavOrd(),
      child: 'slot:config/debug/metrics'
    })
      .get({ subscriber: sub })
      .then(function (metrics) {
        that.$schemaGenerations = metrics.getSchemaGenerations();
        return that.$schemaGenerations;
      });
  };

  /**
   * clears the schema count
   *
   * @private
   * @returns {Promise}
   */
  JsonOutputEditor.prototype.$clearOutput = function () {
    var parentSchema = this.$getParentSchema();

    return Promise.resolve(parentSchema && parentSchema.clearOutput());
  };

  /**
   * opens the metrics slot in a new tab.
   *
   * @private
   */
  JsonOutputEditor.prototype.$openMetrics = function () {
    var parentSchema = this.$getParentSchema();

    if (parentSchema) {
      var metricsOrd = baja.Ord.make({
          base: parentSchema.getNavOrd().relativizeToSession(),
          child: 'slot:config/debug/metrics'
        }),
        hyperlinkOrd = metricsOrd;

      if (window.niagara.env.type === 'wb') {
        // An adjustment to the ord is needed in workbench, this adds 'fox' or 'foxs'
        hyperlinkOrd = baja.Ord.make({
          base: niagara.env.getBaseOrd(),
          child: metricsOrd
        }).normalize();
      }

      niagara.env.toHyperlink(hyperlinkOrd)
        .then(function (hyperlink) {
          window.open(hyperlink, '_blank');
        }).catch(logError);
    }
  };

  /**
   * toggles the textArea between single and multi-line display
   *
   * @private
   * @returns {Promise}
   */
  JsonOutputEditor.prototype.$toggleView = function () {
    this.$setMultiLineFormat(!(this.$isMultiLineFormat()));

    if (this.$isMultiLineFormat()) {
      this.$getToggleCommand().setIcon(MULTI_LINE_ICON);
      this.$getToggleCommand().setDisplayNameFormat(MULTI_LINE_NAME);

      var formattedString = this.$formatForEditor(this.value()),
        count = (formattedString.match(/[\n]/g) || []).length,
        minRows = Math.max(DEFAULT_ROWS, count + 1);

      this.$getEditor().renderer.setOption('maxLines', Math.min(MAX_ROWS, minRows));
    } else {
      this.$getToggleCommand().setIcon(SINGLE_LINE_ICON);
      this.$getToggleCommand().setDisplayNameFormat(SINGLE_LINE_NAME);

      this.$getEditor().renderer.setOption('maxLines', DEFAULT_ROWS);
    }

    return this.$populateTextArea();
  };

  /**
   * returns whether the editor is in multiLineFormat mode
   *
   * @private
   * @returns {Boolean}
   */
  JsonOutputEditor.prototype.$isMultiLineFormat = function () {
    return this.$multiLineFormat;
  };

  /**
   * set whether the editor is in multiLineFormat mode
   *
   * @private
   * @param {Boolean} isMultiLineFormat
   */
  JsonOutputEditor.prototype.$setMultiLineFormat = function (multiLineFormat) {
    this.$multiLineFormat = multiLineFormat;
  };

  /**
   * returns the JsonSchema parent
   *
   * @private
   * @returns {baja.Component}
   */
  JsonOutputEditor.prototype.$getParentSchema = function () {
    return this.getComplex();
  };

  /**
   * loads the queryName into the editor
   *
   * @param {String} outputText
   * @returns {Promise}
   */
  JsonOutputEditor.prototype.doLoad = function (outputText) {
    return this.$populateTextArea();
  };

  /**
   * populates the textArea element
   *
   * @returns {Promise}
   */
  JsonOutputEditor.prototype.$populateTextArea = function () {
    var that = this;

    return Promise.resolve(
      this.$getEditor().setValue(this.$formatForEditor(this.value()))
    ).then(function () {
      that.$getEditor().clearSelection();
      that.$getEditor().navigateFileStart();
    });
  };

  /**
   * returns the passed string in indented format
   *
   * @private
   * @param {String} stringToFormat
   * @returns {String}
   */
  JsonOutputEditor.prototype.$formatForEditor = function (stringToFormat) {
    if (!stringToFormat || !this.$isMultiLineFormat()) { return stringToFormat; }

    try {
      return JSON.stringify(JSON.parse(stringToFormat), null, 2);
    } catch (err) {
      logFine('Error converting to indented format: ' + err);
      return stringToFormat;
    }
  };

  /**
   * Enables or disables the Generate, Clear Output & Stats buttons.
   * (Indented Display is always enabled)
   *
   * @param {Boolean} enabled
   */
  JsonOutputEditor.prototype.$enableCommands = function (enabled) {
    this.$getGenerateCommand().setEnabled(enabled);
    this.$getClearCommand().setEnabled(enabled);
    this.$getMetricsCommand().setEnabled(enabled);
  };

  /**
   * returns the Generate Command
   *
   * @private
   * @returns {module:bajaux/commands/Command}
   */
  JsonOutputEditor.prototype.$getGenerateCommand = function () {
    return this.$commandButtonGroup.value().getChildren()[0];
  };


  /**
   * returns the Clear Command
   *
   * @private
   * @returns {module:bajaux/commands/Command}
   */
  JsonOutputEditor.prototype.$getClearCommand = function () {
    return this.$commandButtonGroup.value().getChildren()[1];
  };

  /**
   * returns the Metrics Command
   *
   * @private
   * @returns {module:bajaux/commands/Command}
   */
  JsonOutputEditor.prototype.$getMetricsCommand = function () {
    return this.$commandButtonGroup.value().getChildren()[2];
  };

  /**
   * returns the Toggle Command
   *
   * @private
   * @returns {module:bajaux/commands/Command}
   */
  JsonOutputEditor.prototype.$getToggleCommand = function () {
    return this.$commandButtonGroup.value().getChildren()[3];
  };

  /**
   * Clean up after this editor
   * @returns {Promise}
   */
  JsonOutputEditor.prototype.doDestroy = function () {
    this.jq().removeClass('JsonOutputEditor');

    return this.$getEditor().destroy();
  };

  return (JsonOutputEditor);
});
