/**
 * @Copyright 2019 Tridium, Inc. All Rights Reserved.
 * @author h129264
 */

define(['baja!'], function (baja) {
  'use strict';

  /**
   * API Status: **Private**
   * @exports nmodule/videoDriver/rc/rpc/rpc
   */
  var exports = {};

  /**
   * Connected cameras will be returned from the cache.
   *
   * @returns {Promise.<string[]>}
   */
  exports.getAllCameras = function () {
    return callVideoDriverRpc('getCameraList');
  };

  /**
   * @typedef {Object} module:nmodule/videoDriver/rc/rpc/rpc~CameraInfo
   * @property {string} displayName camera display name
   * @property {boolean} focusEnabled whether the camera has focus enabled
   * @property {number} focusInterval how often to fire focus events
   * @property {boolean} irisEnabled whether the camera has iris enabled
   * @property {number} irisInterval how often to fire iris events
   * @property {boolean} isTimestampPreferred whether the video should show the current timestamp
   * @property {String}  moveToPresetAction the name of the action to fire to load a pan/zoom preset
   * @property {boolean} moveToPresetEnabled whether the camera can load a pan/zoom preset
   * @property {boolean} panTiltEnabled whether the camera has pan and tilt enabled
   * @property {String} presetEnum string encoding of a DynamicEnum with the available presets
   * @property {String} storePresetAction the name of the action to fire to store a pan/zoom preset
   * @property {boolean} storePresetEnabled whether the camera can store current pan/zoom settings to a preset
   * @property {boolean} zoomEnabled whether the camera has zoom enabled
   * @property {number} zoomInterval how often to fire zoom events
   */

  /**
  * @param {String|baja.Ord} ord ORD to a `BIVideoCamera` instance
  * @returns {Promise.<module:nmodule/videoDriver/rc/rpc/rpc~CameraInfo>}
  */
  exports.getCameraInfo = function (ord) {
    return callVideoDriverRpc('getCameraInfo', [String(ord)]);
  };

  /**
   * @param {String|baja.Ord} [ord]
   * @returns {Promise.<boolean>}
   */
  exports.isFoxVideoStreamPreferred = function (ord) {
    //fox streaming is disabled in web until NCCB-43905 is in progress
    return Promise.resolve(false);
    // if (!ord) {
    //   return Promise.resolve(false);
    // } else {
    //   return callVideoDriverRpc('isFoxVideoStreamPreferred', [ String(ord) ]);
    // }
  };

  /**
   * @param {String|baja.Ord} ord
   * @returns {Promise.<{ streamUrl: string }>}
   */
  exports.startHlsStream = function (ord) {
    return callVideoDriverRpc('startHlsStream', [String(ord)]);
  };

  /**
   * @param {String|baja.Ord} ord
   * @returns {Promise}
   */
  exports.stopHlsStream = function (ord) {
    return callVideoDriverRpc('stopHlsStream', [String(ord)]);
  };

  //TODO: move this into xprotect
  /**
   * Fetches the Server Connection details - url, username, password, port
   * @param {nvideo:VideoCamera} value - video camera component
   * @returns {Promise}
   */
  exports.getConnectionParams = function (value) {
    return baja.rpc({
      ord: value.getNavOrd(),
      methodName: 'getServerConnectionParamsWithRpc'
    });
  };

  /**
   * RPC to perform a focus operation on a camera.
   *
   * @param {String|baja.Ord} ord ord to a `videoDriver:IVideoFocusControl`
   * @param {String} focusOption tag of a `videoDriver:VideoFocusEnum`
   */
  exports.focus = function (ord, focusOption) {
    return callVideoDriverRpc('focusControl', [String(ord), focusOption]);
  };

  /**
   * RPC to perform a focus operation on a camera.
   *
   * @param {String|baja.Ord} ord ord to a `videoDriver:IVideoIrisControl`
   * @param {String} irisOption tag of a `videoDriver:VideoIrisEnum`
   */
  exports.iris = function (ord, irisOption) {
    return callVideoDriverRpc('irisControl', [String(ord), irisOption]);
  };

  /**
   * RPC to perform pan/tilt controls on the camera
   *
   * @param {String|baja.Ord} ord ord to a `videoDriver:IVideoPanTiltControl`
   * @param {String} panTiltOption tag of a `videoDriver:VideoPanTiltEnum`
   */
  exports.move = function (ord, panTiltOption) {
    return callVideoDriverRpc('move', [String(ord), panTiltOption]);
  };

  /**
   * RPC to perform zoom controls on the camera
   *
   * @param {String|baja.Ord} ord ord to a `videoDriver:IVideoZoomControl`
   * @param {String} zoomOption tag of a `videoDriver:VideoZoomEnum`
   * @returns {Promise}
   */
  exports.zoom = function (ord, zoomOption) {
    return callVideoDriverRpc('zoom', [String(ord), zoomOption]);
  };

  /**
   * @typedef {Object} module:nmodule/videoDriver/rc/rpc/rpc~CameraEventJSONObject
   * @property {baja.AbsTime} timeIndex
   * @property {string} description
   */

  /**
   * RPC to get a list of events from a camera based upon given time
   * @param {string|baja.Ord} ord the ord for the
   * @param {baja.AbsTime} time the time to get the records from
   * @param {number} fetchVector the number and direction of records to get
   * @returns {Array.<module:nmodule/videoDriver/rc/rpc/rpc~CameraEventJSONObject>} the return json values
   */
  exports.getCameraPlaybackEvents = function (ord, time, fetchVector) {
    return callVideoDriverRpc('getCameraPlaybackEvents', [String(ord), time.encodeToString(), fetchVector]).then(function (eventArray) {
      for (var i = 0; i < eventArray.length; i++) {
        eventArray[i].timeIndex = baja.AbsTime.make().decodeFromString(eventArray[i].timeIndex);
      }
      return eventArray;
    });
  };
  function callVideoDriverRpc(methodName, args) {
    return baja.rpc({
      typeSpec: 'videoDriver:VideoDriverRpc',
      methodName: methodName,
      args: args
    });
  }
  return exports;
});
