/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 */

/**
 * API Status: **Private**
 * @module nmodule/history/rc/fe/CapacityEditor
 */
define(['baja!',
        'lex!history',
        'bajaux/events',
        'jquery',
        'Promise',
        'underscore',
        'nmodule/history/rc/baja/Capacity',
        'nmodule/webEditors/rc/fe/fe',
        'nmodule/webEditors/rc/fe/baja/BaseEditor',
        'nmodule/webEditors/rc/fe/baja/OrderedMapEditor',
        'css!nmodule/history/rc/fe/historyEditors'], function (
         baja,
         lexs,
         events,
         $,
         Promise,
         _,
         Capacity,
         fe,
         BaseEditor,
         OrderedMapEditor) {

  'use strict';

  var historyLex = lexs[0],

      DESTROY_EVENT = events.DESTROY_EVENT,
      DISABLE_EVENT = events.DISABLE_EVENT,
      ENABLE_EVENT = events.ENABLE_EVENT,
      INITIALIZE_EVENT = events.INITIALIZE_EVENT,
      LOAD_EVENT = events.LOAD_EVENT,
      MODIFY_EVENT = events.MODIFY_EVENT,
      READONLY_EVENT = events.READONLY_EVENT,
      WRITABLE_EVENT = events.WRITABLE_EVENT;

  var TYPES = (function () {
    var map = new baja.OrderedMap();
    map.put(historyLex.get('unlimited'), Capacity.RESTRICT_NONE);
    map.put(historyLex.get('recordCount'), Capacity.RESTRICT_RECORD_COUNT);
    return map;
  }());

  /**
   * Editor for working with a `history:Capacity` value.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/history/rc/fe/CapacityEditor
   */
  var CapacityEditor = function CapacityEditor(params) {
    BaseEditor.call(this, _.extend({ formFactor: 'mini' }, params));
  };
  CapacityEditor.prototype = Object.create(BaseEditor.prototype);
  CapacityEditor.prototype.constructor = CapacityEditor;

  CapacityEditor.DEFAULT_MAX_RECORDS = 500;

  /**
   * Get the numeric editor for max records.
   *
   * @private
   * @returns {module:nmodule/webEditors/rc/fe/baja/IntegerEditor}
   */
  CapacityEditor.prototype.$getMaxRecordsEditor = function () {
    return this.jq().children('.maxRecords').data('widget');
  };

  /**
   * Get the dropdown editor for capacity type.
   *
   * @private
   * @returns {module:nmodule/webEditors/rc/fe/baja/OrderedMapEditor}
   */
  CapacityEditor.prototype.$getTypesEditor = function () {
    return this.jq().children('.capacityType').data('widget');
  };

  /**
   * Called whenever a new capacity type is selected, hiding/showing the max
   * records editor as appropriate.
   *
   * @private
   */
  CapacityEditor.prototype.$capacityTypeChanged = function() {
    var mapEd = this.$getTypesEditor(),
        maxEd = this.$getMaxRecordsEditor();
    switch (mapEd.getSelectedValue()) {
      case Capacity.RESTRICT_NONE:
        maxEd.jq().css('display', 'none'); break;
      case Capacity.RESTRICT_RECORD_COUNT:
        maxEd.jq().css('display', ''); break;
    }
  };

  /**
   * Initialize editors for capacity type and max records.
   *
   * @param {jQuery} dom
   * @returns {Promise}
   */
  CapacityEditor.prototype.doInitialize = function (dom) {
    var that = this;

    dom.on([
      DESTROY_EVENT,
      DISABLE_EVENT, ENABLE_EVENT,
      INITIALIZE_EVENT, LOAD_EVENT,
      READONLY_EVENT, WRITABLE_EVENT
    ].join(' '), '.editor', false);

    dom.on(MODIFY_EVENT, '.capacityType', function () {
      that.$capacityTypeChanged();
      that.setModified(true);
      return false;
    });

    dom.on(MODIFY_EVENT, '.maxRecords', function () {
      that.setModified(true);
      return false;
    });

    return Promise.join(
      fe.buildFor({
        dom: $('<div class="capacityType"/>').appendTo(dom),
        formFactor: 'mini',
        type: OrderedMapEditor,
        value: TYPES
      }),
      fe.buildFor({
        dom: $('<div class="maxRecords"/>').appendTo(dom),
        properties: { min: 0, postlabel: historyLex.get('records') },
        formFactor: 'mini',
        value: baja.Integer.DEFAULT
      })
    );
  };

  /**
   * Load the type and max records of the `Capacity` into the appropriate
   * sub-editors.
   *
   * @param {module:nmodule/history/rc/baja/Capacity} capacity a `history:Capacity`
   * @returns {Promise}
   */
  CapacityEditor.prototype.doLoad = function (capacity) {
    var that = this,
        byRecords = capacity.isByRecordCount(),
        capacityType = byRecords ?
          Capacity.RESTRICT_RECORD_COUNT :
          Capacity.RESTRICT_NONE,
        maxRecords = byRecords ?
          capacity.getMaxRecords() :
          CapacityEditor.DEFAULT_MAX_RECORDS;

    return Promise.join(
      that.$getMaxRecordsEditor().load(maxRecords),
      that.$getTypesEditor().setSelectedValue(capacityType, true)
    )
      .then(function() {
        return that.$capacityTypeChanged();
      });
  };

  /**
   * Resolve a new `history:Capacity` instance constructed from the selected
   * capacity type and max records entered.
   */
  CapacityEditor.prototype.doRead = function () {
    return this.getChildEditors().readAll().spread(Capacity.make);
  };

  /**
   * Destroy all child editors.
   *
   * @returns {Promise}
   */
  CapacityEditor.prototype.doDestroy = function () {
    return this.getChildEditors().destroyAll();
  };

  /**
   * Set all child editors readonly/writable.
   *
   * @param {Boolean} readonly
   * @returns {Promise}
   */
  CapacityEditor.prototype.doReadonly = function (readonly) {
    return this.getChildEditors().setAllReadonly(readonly);
  };

  /**
   * Set all child editors enabled/disabled.
   *
   * @param {Boolean} enabled
   * @returns {Promise}
   */
  CapacityEditor.prototype.doEnabled = function (enabled) {
    return this.getChildEditors().setAllEnabled(enabled);
  };

  return CapacityEditor;
});
