/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Aaron Laniewicz
 */

/**
 * API Status: **Private**
 * @module nmodule/history/rc/baja/HistoryId
 */
define(['baja!',
        'bajaScript/baja/obj/objUtil'], function (baja, objUtil) {

  'use strict';

  var Simple = baja.Simple,
      cacheDecode = objUtil.cacheDecode,
      cacheEncode = objUtil.cacheEncode,

      // Good format 1:
      //   group[2]: ^ or @
      //   group[3]: historyName
      // Good format 2:
      //   group[4]: deviceName
      //   group[5]: historyName
      // Bad format:
      //   group[1]: empty
      //    -or-
      //   group[6]: not empty
      //               (                 1                     )
      //                (  2  )(   3   )   (   4   )  (   5   ) (6 )
      DECODE_REGEX = /^((\^|\@)([^\/.]*)|\/([^\/.]*)\/([^\/.]*))(.*)/;

  /**
   * BajaScript representation of a `history:HistoryId`.
   *
   * @class
   * @extends baja.Simple
   * @alias nmodule/history/rc/baja/HistoryId
   */
  var HistoryId = function HistoryId() {
    Simple.apply(this, arguments);
  };
  HistoryId.prototype = Object.create(Simple.prototype);
  HistoryId.prototype.constructor = HistoryId;


  /**
   * Make a new `HistoryId` instance.
   *
   * @param {String} deviceName
   * @param {String} historyName
   * @returns {module:nmodule/history/rc/baja/HistoryId}
   */
  HistoryId.make = function (deviceName, historyName) {
    var hid = new HistoryId();
    hid.$deviceName = deviceName || '';
    hid.$historyName = historyName || '';
    return hid;
  };

  /**
   * Get the device name.
   *
   * @returns {String}
   */
  HistoryId.prototype.getDeviceName = function () { return this.$deviceName; };

  /**
   * Get the history name.
   *
   * @returns {String}
   */
  HistoryId.prototype.getHistoryName = function () { return this.$historyName; };

  /**
   * Make a new `HistoryId` instance.
   * @see module:nmodule/history/rc/baja/HistoryId.make
   */
  HistoryId.prototype.make = function () {
    return HistoryId.make.apply(HistoryId, arguments);
  };


  /**
   * Decode a `HistoryId` from a `String` in one of the following formats:
   * 
   * - /deviceName/historyName
   * - @historyName
   * - ^historyName
   *
   * @param {String} str String encoding of a HistoryId
   * @returns {module:nmodule/history/rc/baja/HistoryId}
   */
  HistoryId.prototype.decodeFromString = cacheDecode(function (str) {
    if (!str) { return HistoryId.make(); }

    var result = DECODE_REGEX.exec(str);
    if (!result || result[6]) {
      throw new Error('Invalid history:HistoryId encoding: ' + str);
    } else if (result[2]) {
      return HistoryId.make(result[2], result[3]);
    } else {
      return HistoryId.make(result[4], result[5]);
    }
  });

  /**
   * Encode a `HistoryId` to a `String`.
   *
   * @returns {String}
   */
  HistoryId.prototype.encodeToString = cacheEncode(function () {
    var deviceName = this.$deviceName,
        historyName = this.$historyName;

    if (!deviceName && !historyName) {
      return '';
    }
    if (deviceName === HistoryId.DEVICE_SHORTHAND ||
        deviceName === HistoryId.NIAGARA_STATION_SHORTHAND) {
      return deviceName + historyName;
    }
    return '/' + deviceName + '/' + historyName;
  });

  /**
   * Shorthand conversion for station.
   * @type {String}
   */
  HistoryId.DEVICE_SHORTHAND = '^';

  /**
   * Shorthand conversion for parent station.
   * @type {String}
   */
  HistoryId.NIAGARA_STATION_SHORTHAND = '@';

  HistoryId.DEFAULT = HistoryId.make('','');

  baja.registerType('history:HistoryId', function () { return HistoryId; });

  return HistoryId;
});
