/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 */

/**
 * API Status: **Private**
 * @module nmodule/history/rc/baja/Capacity
 */
define(['baja!',
        'bajaScript/baja/obj/objUtil'], function (baja, objUtil) {

  'use strict';

  var Simple = baja.Simple,
      cacheDecode = objUtil.cacheDecode,
      cacheEncode = objUtil.cacheEncode,

      STR_REGEX = /(\d):(\d+)/;

  /**
   * BajaScript representation of a `history:Capacity`.
   *
   * @class
   * @extends baja.Simple
   * @alias nmodule/history/rc/baja/Capacity
   */
  var Capacity = function Capacity() {
    Simple.apply(this, arguments);
  };
  Capacity.prototype = Object.create(Simple.prototype);
  Capacity.prototype.constructor = Capacity;

  /**
   * Represent an unlimited Capacity.
   * @type {number}
   */
  Capacity.RESTRICT_NONE = 0;

  /**
   * Represent a Capacity restricted by number of records.
   * @type {number}
   */
  Capacity.RESTRICT_RECORD_COUNT = 1;

  /**
   * Represent a Capacity restricted by storage size.
   * @type {number}
   */
  Capacity.RESTRICT_STORAGE_SIZE = 2;

  /**
   * Make a new `Capacity` instance.
   *
   * @param {Number} restrictBy number indicating the capacity type; use a
   * constant like `RESTRICT_NONE`
   * @param {Number} max max number of records (only used for
   * `RESTRICT_RECORD_COUNT`; otherwise unused)
   * @returns {module:nmodule/history/rc/baja/Capacity}
   */
  Capacity.make = function (restrictBy, max) {
    switch(restrictBy) {
      case Capacity.RESTRICT_NONE:
      case Capacity.RESTRICT_RECORD_COUNT:
      case Capacity.RESTRICT_STORAGE_SIZE:
        var cap = new Capacity();
        cap.$restrictBy = restrictBy;
        cap.$max = max;
        return cap;
      default:
        throw new Error('unknown restrictBy: ' + restrictBy);
    }
  };

  /**
   * Make a new `Capacity` instance.
   * @see module:nmodule/history/rc/baja/Capacity.make
   */
  Capacity.prototype.make = function () {
    return Capacity.make.apply(Capacity, arguments);
  };

  Capacity.prototype.decodeFromString = cacheDecode(function (str) {
    var result = STR_REGEX.exec(str);
    if (!result) {
      throw new Error('invalid history:Capacity encoding: ' + str);
    }
    return Capacity.make(parseInt(result[1], 10), parseInt(result[2], 10));
  });

  Capacity.prototype.encodeToString = cacheEncode(function () {
    return this.$restrictBy + ':' + this.$max;
  });

  /**
   * Get the max number of records denoted by this Capacity. Returns -1 for
   * unrestricted and throws for other capacity types.
   * @returns {Number}
   * @throws {Error} if this capacity cannot be described in terms of number
   * of records
   */
  Capacity.prototype.getMaxRecords = function () {
    switch (this.$restrictBy) {
      case Capacity.RESTRICT_NONE:
        return -1;
      case Capacity.RESTRICT_RECORD_COUNT:
        var max = this.$max;
        return max > baja.Integer.MAX_VALUE ? baja.Integer.MAX_VALUE.valueOf() : max;
      default:
        throw new Error('Capacity not restricted by record count.');
    }
  };

  /**
   * Return true if this capacity is measured in number of records.
   * @returns {boolean}
   */
  Capacity.prototype.isByRecordCount = function () {
    return this.$restrictBy === Capacity.RESTRICT_RECORD_COUNT;
  };

  /**
   * Return true if this capacity is measured in storage size.
   * @returns {boolean}
   */
  Capacity.prototype.isByStorageSize = function () {
    return this.$restrictBy === Capacity.RESTRICT_STORAGE_SIZE;
  };

  /**
   * Return true if this capacity is unrestricted.
   * @returns {boolean}
   */
  Capacity.prototype.isUnlimited = function () {
    return this.$restrictBy === Capacity.RESTRICT_NONE;
  };

  Capacity.DEFAULT = Capacity.UNLIMITED =
    Capacity.make(Capacity.RESTRICT_NONE, 0);

  baja.registerType('history:Capacity', function () { return Capacity; });

  return Capacity;
});
