/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 */

/**
 * API Status: **Private**
 * @module nmodule/history/rc/HistoryGroupManager
 */
define([
  'baja!',
  'baja!history:HistoryGroupings,history:HistoryGroup',
  'jquery',
  'Promise',
  'underscore',
  'nmodule/webEditors/rc/wb/mgr/mgrUtils',
  'nmodule/webEditors/rc/wb/mgr/Manager',
  'nmodule/webEditors/rc/wb/mgr/model/MgrModel',
  'nmodule/webEditors/rc/wb/mgr/MgrTypeInfo',
  'nmodule/webEditors/rc/wb/mgr/model/columns/NameMgrColumn',
  'nmodule/webEditors/rc/wb/mgr/model/columns/PropertyMgrColumn',
  'nmodule/webEditors/rc/wb/table/model/Column',
  'nmodule/webEditors/rc/fe/baja/util/DepthSubscriber',
  'nmodule/webEditors/rc/wb/mgr/commands/NewCommand',
  'nmodule/webEditors/rc/wb/mgr/commands/EditCommand',
  'nmodule/webEditors/rc/wb/mgr/commands/MgrDuplicateCommand',
  'nmodule/webEditors/rc/wb/mgr/commands/RemoveCommand',
  'nmodule/webEditors/rc/wb/commands/DeleteCommand',
  'nmodule/webEditors/rc/wb/mixin/ContextMenuSupport'
], function (baja, types, $, Promise, _, mgrUtils,
             Manager, MgrModel, MgrTypeInfo, NameMgrColumn, PropertyMgrColumn,
             Column, DepthSubscriber, NewCommand, EditCommand, DuplicateCommand,
             RemoveCommand, DeleteCommand, addContextMenuSupport) {

  'use strict';

  var HISTORY_GROUP_TYPE = baja.lt('history:HistoryGroup'),
      COLUMNS            = getColumns();

  function getColumns() {
    var historyGroupProps   = baja.$('history:HistoryGroup').getSlots().properties().toArray(),
        historyGroupColumns = _.map(historyGroupProps, function (prop) {
          return new PropertyMgrColumn(prop, {
            flags: Column.flags.EDITABLE,
            type: HISTORY_GROUP_TYPE
          });
        });

    return [ new NameMgrColumn({ flags: Column.flags.EDITABLE }) ].concat(historyGroupColumns);
  }

  /**
   * @class
   * @extends module:nmodule/webEditors/rc/wb/mgr/Manager
   * @alias module:nmodule/history/rc/HistoryGroupManager
   */
  var HistoryGroupManager = function HistoryGroupManager(params) {

    Manager.call(this, _.extend({
      moduleName: 'history',
      keyName: 'HistoryGroupUxManager'
    }, params));


    this.getCommandGroup().add(
      new NewCommand(this),
      new EditCommand(this),
      new DuplicateCommand(this),
      new RemoveCommand(this)
    );

    this.$sub = new DepthSubscriber(2);

    addContextMenuSupport(this);
  };

  //extend and set up prototype chain
  HistoryGroupManager.prototype = Object.create(Manager.prototype);
  HistoryGroupManager.prototype.constructor = HistoryGroupManager;

  /**
   * Returns a reference to the Edit command
   * @private
   * @returns {Command}
   */
  HistoryGroupManager.prototype.$getEditCommand = function () {
    return this.getCommandGroup().get(1);
  };

  /**
   * Returns a reference to the Duplicate command
   * @private
   * @returns {Command}
   */
  HistoryGroupManager.prototype.$getDuplicateCommand = function () {
    return this.getCommandGroup().get(2);
  };

  /**
   * Returns a reference to the Remove command
   * @private
   * @returns {Command}
   */
  HistoryGroupManager.prototype.$getRemoveCommand = function () {
    return this.getCommandGroup().get(3);
  };


  /**
   * Initial DOM setup
   *
   * @param {jQuery} dom the DOM element into which to load this View
   * @return {Promise}
   */
  HistoryGroupManager.prototype.doInitialize = function (dom) {
    var that = this;

    dom.on('dblclick', '.mainTable tr', function (e) {
      that.$getEditCommand().invoke();
    });

    return Manager.prototype.doInitialize.call(this, dom);
  };

  /**
   * Create a MgrModel instance from a history:HistoryGroupings component source
   * @private
   * @param historyGroupings
   */
  HistoryGroupManager.prototype.makeModel = function (historyGroupings) {
    return MgrTypeInfo.make([ 'history:HistoryGroup' ])
      .then(function (newTypes) {
        return new MgrModel({
          componentSource: historyGroupings,
          newTypes: newTypes,
          columns: COLUMNS
        });
      });
  };

  /**
   * Override doLoad() to subscribe to the history groupings component.
   * @param historyGroupings
   * @returns {Promise}
   */
  HistoryGroupManager.prototype.doLoad = function (historyGroupings) {
    var that = this;

    return this.$sub.subscribe(historyGroupings)
      .then(function () {
        return Manager.prototype.doLoad.call(that, historyGroupings);
      }).then(function () {
        var selection = mgrUtils.getMainTableSelection(that),
            editCmd   = that.$getEditCommand(),
            dupCmd    = that.$getDuplicateCommand();

        dupCmd.setComponent(historyGroupings);

        selection.on('changed', function () {
          editCmd.setEnabled(!selection.isEmpty());
          dupCmd.setEnabled(!selection.isEmpty());
        });
      });
  };

  /**
   * Destroy the main view table / unsubscribe
   * @returns {Promise}
   */
  HistoryGroupManager.prototype.doDestroy = function () {
    return Promise.join(this.$sub.unsubscribeAll(), Manager.prototype.doDestroy.apply(this, arguments));
  };

  ////////////////////////////////////////////////////////////
  //Context Menu Support
  ////////////////////////////////////////////////////////////

  /**
   * Callback from CommandGroupContextMenu. Trigger context menus specifically on table row right clicks
   * @override
   * @returns {string}
   */
  HistoryGroupManager.prototype.getContextMenuSelector = function () {
    return 'tr';
  };


  /**
   * Callback from ContextMenuSupport to provide a chance to pre-process
   * the right click menu's CommandGroup prior to displaying it
   *
   * @param group default CommandGroup consisting of subject's menu agents
   * @returns CommandGroup updated CommandGroup
   */
  HistoryGroupManager.prototype.updateMenuCommandGroup = function (group) {
    if (group) {
      group.add(this.$getDuplicateCommand(), this.$getEditCommand());
    }

    return group;
  };

  /////////////////////////////////////////////////////////////////

  return HistoryGroupManager;
});
