/*
 * Copyright 2012, Tridium Inc, All Rights Rervered.
 */
package javax.baja.web.js;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.baja.naming.BOrd;
import javax.baja.sys.BajaRuntimeException;
import javax.baja.sys.Type;

/**
 * A data structure for holding JavaScript information.
 * 
 * @see BIJavaScript
 * 
 * @author   Gareth Johnson on 17 Oct 2012
 * @since    Niagara 4.0
 */
public final class JsInfo
{
  private JsInfo(BOrd js, String buildId)
  {
    if (js == null)
    {
      throw new IllegalArgumentException("js required");
    }
    
    this.js = js;
    this.buildId = buildId;
  }
  
////////////////////////////////////////////////////////////////
// Factory
////////////////////////////////////////////////////////////////  

  /**
   * @param js ORD to the JS file this JsInfo represents
   * @param buildId ID for the JsBuild in which this JS file resides. If null,
   *                it is not considered part of any JsBuild and therefore 
   *                will always be downloaded unminified.
   * @return new JsInfo
   */
  public static JsInfo make(BOrd js, String buildId)
  {
    return new JsInfo(js, buildId);
  }

  /**
   * Create a JsInfo using the given BJsBuild Type's ID as the build ID.
   * 
   * @param js ORD to the JS file this JsInfo represents
   * @param buildType BJsBuild subtype whose ID to use
   * @return new JsInfo
   */
  public static JsInfo make(BOrd js, Type buildType)
  {
    if (buildType == null ||
      buildType.isAbstract() ||
      !buildType.is(BJsBuild.TYPE))
    {
      throw new IllegalArgumentException("BJsBuild subtype required");
    }
    
    BJsBuild build = (BJsBuild) buildType.getInstance();
    
    return new JsInfo(js, build.getId());
  }

  /**
   * Create JsInfo with no build ID - this file will always be downloaded
   * unminified.
   * 
   * @param js ORD to the JS file this JsInfo represents
   * @return new JsInfo
   */
  public static JsInfo make(BOrd js)
  {
    return new JsInfo(js, null);
  }
  
////////////////////////////////////////////////////////////////
// Access
////////////////////////////////////////////////////////////////

  /**
   * @return an ORD to the non-minified JavaScript file. 
   */
  public BOrd getJs()
  {
    return js;
  }
  
  /**
   * @return the RequireJS Id for the JavaScript resource.
   */
  public String getJsId()
  {
    return js.isNull() ? "" : toRequireJsId(js); 
  }

  /**
   * @return ID for the JsBuild in which this JS file resides
   */
  public String getBuildId()
  {
    return buildId;
  }
  
  /**
   * Return the RequireJS IDs necessary to preload before loading this
   * JavaScript resource. Each dependent BJsBuild will have its webdev
   * status checked; its module IDs will be included in the resultant list
   * only if webdev is disabled.
   * 
   * @return all RequireJS IDs
   */
  public String[] getBuiltJsIds()
  {
    List<BJsBuild> builds = getBuilds();
    List<String> ids = new ArrayList<>();

    for (BJsBuild build : builds)
    {
      for (BOrd ord : build.getBuiltFiles())
      {
        ids.add(toRequireJsId(ord));
      }
    }

    return ids.toArray(STRINGS);
  }

  private List<BJsBuild> getBuilds()
  {
    List<BJsBuild> list = new ArrayList<>();
    Optional<BJsBuild> build = BJsBuild.forId(buildId);
    if (build.isPresent()) {
      addBuilds(list, build.get());
    } //else logger.warn
    return list;
  }

  private static void addBuilds(List<BJsBuild> list, BJsBuild build)
  {
    if (!build.isWebDevEnabled())
    {
      list.add(build);
    }

    for (BJsBuild dep : build.getDependentBuilds())
    {
      addBuilds(list, dep);
    }
  }
  
  /**
   * @return true if built JavaScript information is available.
   */
  public boolean hasBuiltJs()
  {
    return !getBuilds().isEmpty();
  }
  
  /**
   * @param ord an ORD pointing to a JS file
   * @return a RequireJS AMD ID representation of that ORD
   */
  public static String toRequireJsId(BOrd ord)
  {
    Matcher matcher = moduleIdPattern.matcher(ord.toString());
    
    if (!matcher.find())
    {
      throw new BajaRuntimeException(invalidErr + ord);
    }
    
    String path = matcher.group(1); 
    
    if (path == null)
    {
      throw new BajaRuntimeException(invalidErr + ord);
    }
    
    return "nmodule/" + path;
  }
   
////////////////////////////////////////////////////////////////
// Attributes
////////////////////////////////////////////////////////////////
  
  private final BOrd js;
  private final String buildId;
  public static final String[] STRINGS = new String[0];
  
  private static final String invalidErr = "Invalid JavaScript ORD: ";
  
  private static final Pattern moduleIdPattern = 
      Pattern.compile("^module://(.+)\\.[jJ][sS]$");
}
