/*
 * Copyright 2015 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.web.js;

import java.util.Arrays;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.baja.naming.BOrd;
import javax.baja.registry.TypeInfo;
import javax.baja.sys.BSingleton;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;
import javax.baja.web.WebDev;

/**
 * This class declares one particular bundle of JavaScript, optimized by the
 * r.js optimizer (and probably minified as well).
 * 
 * If webdev is turned off for this build's ID, then any built files it declares
 * should be downloaded first, before any individual modules it contains are
 * loaded. If webdev is turned on, the built files should be skipped so that
 * the raw, unminified modules are loaded.
 * 
 * Most commonly, there will be one {@code BJsBuild} per Niagara module,
 * declaring a single builtfile containing all the JS in that module. Consider
 * using {@code grunt-niagara} to easily create that builtfile as part of the
 * module build process.
 * 
 * @author Logan Byam
 * @since Niagara 4.0
 * @see <a href="https://github.com/tridium/grunt-niagara">grunt-niagara</a>
 * @see <a href="https://github.com/tridium/grunt-init-niagara">grunt-init-niagara</a>
 */
public abstract class BJsBuild extends BSingleton
{
  public static final Type TYPE = Sys.loadType(BJsBuild.class);
  
  @Override
  public Type getType() { return TYPE; }
  
  private static final String[] EMPTY = new String[0];
  private final String id;
  private final BOrd[] builtFiles;
  private final String[] dependentBuilds;
  private static boolean isAlreadyLogged = false;

  /**
   * @param id the ID to search for
   * @return the BJsBuild instance with the given ID, or empty if not found
   */
  public static Optional<BJsBuild> forId(String id) {
    if (id == null)
    {
      return Optional.empty();
    }

    return BY_ID.computeIfAbsent(id, BJsBuild::queryRegistryByIdSlow);
  }

  private static Optional<BJsBuild> queryRegistryByIdSlow(String id) {
    BJsBuild instance = null;
    
    for (TypeInfo typeInfo : Sys.getRegistry().getTypes(TYPE.getTypeInfo()))
    {
      if (!typeInfo.isAbstract())
      {
        try
        {
          instance = (BJsBuild) typeInfo.getInstance();
        }
        catch (RuntimeException e)
        {
          final String message = "Unable to load BuildJS type -> " + typeInfo.toString();
          if(!isAlreadyLogged)
          {
            Logger.getLogger("web.jsbuild").log(Level.SEVERE, message, e);
            isAlreadyLogged = true;
          }
          else
          {
            Logger.getLogger("web.jsbuild").log(Level.FINE, message, e);
          }
        }

        if (null != instance && id.equals(instance.getId()))
        {
          return Optional.of(instance);
        }
      }
    }
    
    return Optional.empty();
  }

  /**
   * Get an array of all RequireJS module dependencies from the given types.
   * @param types array of types that extend BJsBuild
   * @return array of all module dependencies necessary to instantiate the
   * JS constructors for these types
   */
  private static String[] typesToDependencies(Type[] types) {
    if (types == null)
    {
      throw new IllegalArgumentException("array of BJsBuild types required");
    }

    return Arrays.stream(types)
      .map(type -> {
        if (type == null || type.isAbstract() || !type.is(TYPE))
        {
          throw new IllegalArgumentException("array of BJsBuild types required");
        }
        return ((BJsBuild) type.getInstance()).getId();
      })
      .toArray(size -> new String[size]);
  }

  /**
   * Create an instance with a string ID and array of built/minified Javascript
   * files. This instance will not register dependencies on any other builds,
   * so avoid this constructor unless you are certain your build has no
   * dependencies on any other JS builds.
   *
   * @param id string to uniquely identify this build
   * @param builtFiles array of built/minified files
   * @throws IllegalArgumentException if id or file array is missing
   */
  protected BJsBuild(String id, BOrd[] builtFiles)
  {
    this(id, builtFiles, EMPTY);
  }

  /**
   * Create an instance with a string ID and a single built/minified Javascript
   * file. This instance will not register dependencies on any other builds,
   * so avoid this constructor unless you are certain your build has no
   * dependencies on any other JS builds.
   *
   * @param id string to uniquely identify this build
   * @param builtFile built/minified file
   * @throws IllegalArgumentException if id or file array is missing
   * @since Niagara 4.8
   */
  protected BJsBuild(String id, BOrd builtFile)
  {
    this(id, new BOrd[] { builtFile }, EMPTY);
  }

  /**
   * Create an instance with a string ID and a single built/minified Javascript
   * file.
   *
   * @param id string to uniquely identify this build
   * @param builtFile built/minified file
   * @param dependentBuilds array of IDs of builds this build depends on
   * @throws IllegalArgumentException if id or file array is missing
   * @since Niagara 4.8
   */
  protected BJsBuild(String id, BOrd builtFile, String... dependentBuilds)
  {
    this(id, new BOrd[] { builtFile }, dependentBuilds);
  }

  /**
   * Create an instance with a string ID and a single built/minified Javascript
   * file.
   *
   * @param id string to uniquely identify this build
   * @param builtFile built/minified file
   * @param dependentBuilds array of Types of builds this build depends on
   * @throws IllegalArgumentException if id or file array is missing
   * @since Niagara 4.8
   */
  protected BJsBuild(String id, BOrd builtFile, Type... dependentBuilds)
  {
    this(id, new BOrd[] { builtFile }, dependentBuilds);
  }

  /**
   * Create an instance with a string ID and array of built/minified Javascript
   * files.
   * 
   * @param id string to uniquely identify this build
   * @param builtFiles array of built/minified files
   * @param dependentBuilds array of ID of builds this build depends on
   * @throws IllegalArgumentException if id or file array is missing
   */
  protected BJsBuild(String id, BOrd[] builtFiles, String[] dependentBuilds)
  {
    if (id == null)
    {
      throw new IllegalArgumentException("id required");
    }

    if (!isArrayOfOrds(builtFiles))
    {
      throw new IllegalArgumentException("built files required");
    }
    
    this.id = id;
    this.builtFiles = builtFiles.clone();
    this.dependentBuilds = dependentBuilds.clone();
  }
  
  /**
   * @param id string to uniquely identify this build
   * @param builtFiles array of built/minified files
   * @param dependentTypes array of Types of builds this build depends on
   * @throws IllegalArgumentException if id or built files are missing, or
   * if a non-BJsBuild subtype is given
   */
  protected BJsBuild(String id, BOrd[] builtFiles, Type[] dependentTypes)
  {
    this(id, builtFiles, typesToDependencies(dependentTypes));
  }

  /**
   * Get a string to uniquely identify this build file.
   * 
   * @return this build's ID
   */
  public String getId()
  {
    return id;
  }

  /**
   * Get the array of built/minified files represented by this build.
   * 
   * @return array of built/minified files
   */
  public BOrd[] getBuiltFiles()
  {
    return builtFiles.clone();
  }

  /**
   * Get the other builds this build directly depends on.
   * 
   * @return array of BJsBuilds
   */
  public BJsBuild[] getDependentBuilds()
  {
    return Arrays.stream(dependentBuilds)
      .map(b -> forId(b))
      .filter(Optional::isPresent)
      .map(Optional::get)
      .toArray(size -> new BJsBuild[size]);
  }

  /**
   * Check to see if webdev is enabled for this build. If it is disabled,
   * ensure that RequireJS loads the built files before the actual modules;
   * otherwise, download the raw/unminified modules.
   * 
   * @return true if webdev is enabled for this build
   */
  public boolean isWebDevEnabled()
  {
    return WebDev.get(getId()).isEnabled();
  }

  private static boolean isArrayOfOrds(BOrd[] ords)
  {
    if (ords == null || ords.length == 0) { return false; }
    for (BOrd ord : ords) { if (ord == null) { return false; } }
    return true;
  }

  private static final Map<String, Optional<BJsBuild>> BY_ID = new ConcurrentHashMap<>();
}
