/*
 * Copyright (c) 2014. Tridium, Inc. All Rights Reserved.
 */

package javax.baja.web;

import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateNotYetValidException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.logging.Logger;
import java.util.stream.Collectors;

import javax.baja.firewall.BServerPort;
import javax.baja.license.Feature;
import javax.baja.naming.BOrd;
import javax.baja.nre.annotations.Facet;
import javax.baja.nre.annotations.NiagaraAction;
import javax.baja.nre.annotations.NiagaraProperty;
import javax.baja.nre.annotations.NiagaraType;
import javax.baja.security.crypto.BSslTlsEnum;
import javax.baja.security.crypto.BTlsCipherSuiteGroup;
import javax.baja.security.crypto.CertManagerFactory;
import javax.baja.security.dashboard.BISecurityDashboardProvider;
import javax.baja.security.dashboard.BSecurityItemStatus;
import javax.baja.security.dashboard.LexiconFormatInfo;
import javax.baja.security.dashboard.SecurityDashboardItem;
import javax.baja.sys.Action;
import javax.baja.sys.BAbsTime;
import javax.baja.sys.BAbstractService;
import javax.baja.sys.BBoolean;
import javax.baja.sys.BComponent;
import javax.baja.sys.BFacets;
import javax.baja.sys.BIBoolean;
import javax.baja.sys.BIcon;
import javax.baja.sys.BInteger;
import javax.baja.sys.BRelTime;
import javax.baja.sys.BString;
import javax.baja.sys.BValue;
import javax.baja.sys.Context;
import javax.baja.sys.Flags;
import javax.baja.sys.IPropertyValidator;
import javax.baja.sys.LocalizableRuntimeException;
import javax.baja.sys.Property;
import javax.baja.sys.ServiceNotFoundException;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;
import javax.baja.user.BUserService;
import javax.baja.util.BFormat;
import javax.baja.util.BIRestrictedComponent;
import javax.baja.util.BTypeSpec;
import javax.baja.web.http.BHttpHeaderProviders;

import com.tridium.nre.firewall.IpProtocol;
import com.tridium.nre.util.IPAddressUtil;
import com.tridium.security.BISecurityInfoSource;
import com.tridium.security.BSecurityInfo;
import com.tridium.sys.Nre;
import com.tridium.sys.schema.Fw;
import com.tridium.util.ArrayUtil;
import com.tridium.util.CertAliasCasePropertyValidator;
import com.tridium.web.BHostHeaderValidationSettings;
import com.tridium.web.BHostnameRedirectSettings;
import com.tridium.web.BSameSiteEnum;
import com.tridium.web.WebUtil;
import com.tridium.web.rpc.BUserDataConfig;
import com.tridium.web.servlets.UnauthenticatedCache;
import com.tridium.web.servlets.WebStartServlet;
import com.tridium.web.warmup.BWebWarmupConfig;

/**
 * BWebService encapsulates access to the HTTP {@link BWebServer web server}.
 * <p/>
 * Beginning in Niagara4, the web service is simply a general configuration container. The actual
 * web server implementation is now pluggable. See {@link BWebServer} for details.
 *
 * @author <a href="mailto:mgiannini@tridium.com">Matthew Giannini</a>
 */
@NiagaraType
/** Unsecured HTTP port. */
@NiagaraProperty(
  name = "httpPort",
  type = "BServerPort",
  defaultValue = "new BServerPort(80, IpProtocol.TCP)")
/** Enable unsecured connections to the web server on the http port? */
@NiagaraProperty(
  name = "httpEnabled",
  type = "boolean",
  defaultValue = "true",
  facets = @Facet("BFacets.make(BFacets.SECURITY, BBoolean.TRUE)"))
/** Secured HTTPS port. */
@NiagaraProperty(
  name = "httpsPort",
  type = "BServerPort",
  defaultValue = "new BServerPort(443, IpProtocol.TCP)")
/** Enable secure connections to the web server on the https port? */
@NiagaraProperty(
  name = "httpsEnabled",
  type = "boolean",
  defaultValue = "false",
  facets = @Facet("BFacets.make(BFacets.SECURITY, BBoolean.TRUE)"))
/** Redirect all http requests to the secure https port? */
@NiagaraProperty(
  name = "httpsOnly",
  type = "boolean",
  defaultValue = "false",
  facets = @Facet("BFacets.make(BFacets.SECURITY, BBoolean.TRUE)"))
/** The minimum security protocol to use for SSL. */
@NiagaraProperty(
  name = "httpsMinProtocol",
  type = "BSslTlsEnum",
  defaultValue = "BSslTlsEnum.DEFAULT",
  facets = @Facet("BFacets.make(BFacets.SECURITY, BBoolean.TRUE)"))
/** The cipher suite group to use for the tls connections. */
@NiagaraProperty(
  name = "cipherSuiteGroup",
  type = "BTlsCipherSuiteGroup",
  defaultValue = "BTlsCipherSuiteGroup.recommended",
  facets = @Facet("BFacets.make(BFacets.SECURITY, BBoolean.TRUE)"))
/** The certificate to present to clients for SSL connections. */
@NiagaraProperty(
  name = "httpsCert",
  type = "String",
  defaultValue = "tridium",
  facets = {
    @Facet(name = "BFacets.FIELD_EDITOR", value = "BString.make(\"workbench:CertificateAliasFE\")"),
    @Facet(name = "BFacets.UX_FIELD_EDITOR", value = "BString.make(\"webEditors:CertificateAliasEditor\")"),
    @Facet("BFacets.make(BFacets.SECURITY, BBoolean.TRUE)")
  })
/** Require an HTTPS connection when setting passwords */
@NiagaraProperty(
  name = "requireHttpsForPasswords",
  type = "boolean",
  defaultValue = "true",
  facets = @Facet("BFacets.make(BFacets.SECURITY, BBoolean.TRUE)"))
/** The xframe option for preventing XFS attacks. */
@NiagaraProperty(
  name = "xFrameOptions",
  type = "BXFrameOptionsEnum",
  defaultValue = "BXFrameOptionsEnum.sameorigin",
  flags = Flags.HIDDEN | Flags.READONLY,
  facets = @Facet("BFacets.make(BFacets.SECURITY, BBoolean.TRUE)"))
/** Makes the user id cookie persistent */
@NiagaraProperty(
  name = "rememberUserIdCookie",
  type = "boolean",
  defaultValue = "true")
/** Configure SameSite attribute for cookies */
@NiagaraProperty(
  name = "sameSite",
  type = "BSameSiteEnum",
  defaultValue = "BSameSiteEnum.lax",
  facets = @Facet("BFacets.make(BFacets.SECURITY, BBoolean.TRUE)"))
/** Disables autocomplete on the username field of the login page if false */
@NiagaraProperty(
  name = "allowUsernameAutocomplete",
  type = "boolean",
  defaultValue = "true")
/** The template for the browser login page.  If null, the default template is used. */
@NiagaraProperty(
  name = "loginTemplate",
  type = "BTypeSpec",
  defaultValue = "BTypeSpec.NULL",
  facets = {
    @Facet(name = "BFacets.ALLOW_NULL", value = "BBoolean.TRUE"),
    @Facet(name = "BFacets.TARGET_TYPE", value = "BString.make(\"web:LoginTemplate\")")
  })
/** Enable gzip compression? */
@NiagaraProperty(
  name = "gzipEnabled",
  type = "boolean",
  defaultValue = "false",
  flags = Flags.HIDDEN)
/** If the web server supports request logging, then it should write logs to this directory. */
@NiagaraProperty(
  name = "logFileDirectory",
  type = "BOrd",
  defaultValue = "BOrd.make(\"file:^^webLogs\")",
  flags = Flags.READONLY)
/** Supported client environments. */
@NiagaraProperty(
  name = "clientEnvironments",
  type = "BClientEnvironments",
  defaultValue = "new BClientEnvironments()")
/** Will show exception stack traces in error responses when available if set to true. */
@NiagaraProperty(
  name = "showStackTrace",
  type = "boolean",
  defaultValue = "false")
/** If files aren't available locally, enable jx browser JAR files to be loaded from an external URI. */
@NiagaraProperty(
  name = "appletModuleCachingType",
  type = "web:AppletModuleCachingType",
  defaultValue = "BAppletModuleCachingType.host")
/** Configuration properties for clients using Java Web Start. */
@NiagaraProperty(
  name = "webStartConfig",
  type = "web:WebStartConfig",
  defaultValue = "new BWebStartConfig()")
/** Configuration properties for clients using file resources that want additional caching. */
@NiagaraProperty(
  name = "cacheConfig",
  type = "web:CacheConfig",
  defaultValue = "new BCacheConfig()")
/** Configuration for Web Warmup*/
@NiagaraProperty(
  name = "warmupConfig",
  type = "web:WebWarmupConfig",
  defaultValue = "new BWebWarmupConfig()",
  flags = Flags.NON_CRITICAL)
/** Configuration for hostname redirection */
@NiagaraProperty(
  name = "hostnameRedirectSettings",
  type = "web:HostnameRedirectSettings",
  defaultValue = "new BHostnameRedirectSettings()")
/** HTTP header configuration */
@NiagaraProperty(
  name = "httpHeaderProviders",
  type = "web:HttpHeaderProviders",
  defaultValue = "new BHttpHeaderProviders()")
@NiagaraProperty(
  name = "hostHeaderValidationSettings",
  type = "web:HostHeaderValidationSettings",
  defaultValue = "new BHostHeaderValidationSettings()")
@NiagaraAction(name = "resetAllConnections", flags = Flags.CONFIRM_REQUIRED)
public final class BWebService
  extends BAbstractService
  implements BIRestrictedComponent, BISecurityInfoSource, BISecurityDashboardProvider
{

//region /*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
//@formatter:off
/*@ $javax.baja.web.BWebService(3195288909)1.0$ @*/
/* Generated Mon Mar 01 14:36:25 EST 2021 by Slot-o-Matic (c) Tridium, Inc. 2012-2021 */

////////////////////////////////////////////////////////////////
// Property "httpPort"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the {@code httpPort} property.
   * Unsecured HTTP port.
   * @see #getHttpPort
   * @see #setHttpPort
   */
  public static final Property httpPort = newProperty(0, new BServerPort(80, IpProtocol.TCP), null);
  
  /**
   * Get the {@code httpPort} property.
   * Unsecured HTTP port.
   * @see #httpPort
   */
  public BServerPort getHttpPort() { return (BServerPort)get(httpPort); }
  
  /**
   * Set the {@code httpPort} property.
   * Unsecured HTTP port.
   * @see #httpPort
   */
  public void setHttpPort(BServerPort v) { set(httpPort, v, null); }

////////////////////////////////////////////////////////////////
// Property "httpEnabled"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the {@code httpEnabled} property.
   * Enable unsecured connections to the web server on the http port?
   * @see #getHttpEnabled
   * @see #setHttpEnabled
   */
  public static final Property httpEnabled = newProperty(0, true, BFacets.make(BFacets.SECURITY, BBoolean.TRUE));
  
  /**
   * Get the {@code httpEnabled} property.
   * Enable unsecured connections to the web server on the http port?
   * @see #httpEnabled
   */
  public boolean getHttpEnabled() { return getBoolean(httpEnabled); }
  
  /**
   * Set the {@code httpEnabled} property.
   * Enable unsecured connections to the web server on the http port?
   * @see #httpEnabled
   */
  public void setHttpEnabled(boolean v) { setBoolean(httpEnabled, v, null); }

////////////////////////////////////////////////////////////////
// Property "httpsPort"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the {@code httpsPort} property.
   * Secured HTTPS port.
   * @see #getHttpsPort
   * @see #setHttpsPort
   */
  public static final Property httpsPort = newProperty(0, new BServerPort(443, IpProtocol.TCP), null);
  
  /**
   * Get the {@code httpsPort} property.
   * Secured HTTPS port.
   * @see #httpsPort
   */
  public BServerPort getHttpsPort() { return (BServerPort)get(httpsPort); }
  
  /**
   * Set the {@code httpsPort} property.
   * Secured HTTPS port.
   * @see #httpsPort
   */
  public void setHttpsPort(BServerPort v) { set(httpsPort, v, null); }

////////////////////////////////////////////////////////////////
// Property "httpsEnabled"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the {@code httpsEnabled} property.
   * Enable secure connections to the web server on the https port?
   * @see #getHttpsEnabled
   * @see #setHttpsEnabled
   */
  public static final Property httpsEnabled = newProperty(0, false, BFacets.make(BFacets.SECURITY, BBoolean.TRUE));
  
  /**
   * Get the {@code httpsEnabled} property.
   * Enable secure connections to the web server on the https port?
   * @see #httpsEnabled
   */
  public boolean getHttpsEnabled() { return getBoolean(httpsEnabled); }
  
  /**
   * Set the {@code httpsEnabled} property.
   * Enable secure connections to the web server on the https port?
   * @see #httpsEnabled
   */
  public void setHttpsEnabled(boolean v) { setBoolean(httpsEnabled, v, null); }

////////////////////////////////////////////////////////////////
// Property "httpsOnly"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the {@code httpsOnly} property.
   * Redirect all http requests to the secure https port?
   * @see #getHttpsOnly
   * @see #setHttpsOnly
   */
  public static final Property httpsOnly = newProperty(0, false, BFacets.make(BFacets.SECURITY, BBoolean.TRUE));
  
  /**
   * Get the {@code httpsOnly} property.
   * Redirect all http requests to the secure https port?
   * @see #httpsOnly
   */
  public boolean getHttpsOnly() { return getBoolean(httpsOnly); }
  
  /**
   * Set the {@code httpsOnly} property.
   * Redirect all http requests to the secure https port?
   * @see #httpsOnly
   */
  public void setHttpsOnly(boolean v) { setBoolean(httpsOnly, v, null); }

////////////////////////////////////////////////////////////////
// Property "httpsMinProtocol"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the {@code httpsMinProtocol} property.
   * The minimum security protocol to use for SSL.
   * @see #getHttpsMinProtocol
   * @see #setHttpsMinProtocol
   */
  public static final Property httpsMinProtocol = newProperty(0, BSslTlsEnum.DEFAULT, BFacets.make(BFacets.SECURITY, BBoolean.TRUE));
  
  /**
   * Get the {@code httpsMinProtocol} property.
   * The minimum security protocol to use for SSL.
   * @see #httpsMinProtocol
   */
  public BSslTlsEnum getHttpsMinProtocol() { return (BSslTlsEnum)get(httpsMinProtocol); }
  
  /**
   * Set the {@code httpsMinProtocol} property.
   * The minimum security protocol to use for SSL.
   * @see #httpsMinProtocol
   */
  public void setHttpsMinProtocol(BSslTlsEnum v) { set(httpsMinProtocol, v, null); }

////////////////////////////////////////////////////////////////
// Property "cipherSuiteGroup"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the {@code cipherSuiteGroup} property.
   * The cipher suite group to use for the tls connections.
   * @see #getCipherSuiteGroup
   * @see #setCipherSuiteGroup
   */
  public static final Property cipherSuiteGroup = newProperty(0, BTlsCipherSuiteGroup.recommended, BFacets.make(BFacets.SECURITY, BBoolean.TRUE));
  
  /**
   * Get the {@code cipherSuiteGroup} property.
   * The cipher suite group to use for the tls connections.
   * @see #cipherSuiteGroup
   */
  public BTlsCipherSuiteGroup getCipherSuiteGroup() { return (BTlsCipherSuiteGroup)get(cipherSuiteGroup); }
  
  /**
   * Set the {@code cipherSuiteGroup} property.
   * The cipher suite group to use for the tls connections.
   * @see #cipherSuiteGroup
   */
  public void setCipherSuiteGroup(BTlsCipherSuiteGroup v) { set(cipherSuiteGroup, v, null); }

////////////////////////////////////////////////////////////////
// Property "httpsCert"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the {@code httpsCert} property.
   * The certificate to present to clients for SSL connections.
   * @see #getHttpsCert
   * @see #setHttpsCert
   */
  public static final Property httpsCert = newProperty(0, "tridium", BFacets.make(BFacets.make(BFacets.make(BFacets.FIELD_EDITOR, BString.make("workbench:CertificateAliasFE")), BFacets.make(BFacets.UX_FIELD_EDITOR, BString.make("webEditors:CertificateAliasEditor"))), BFacets.make(BFacets.SECURITY, BBoolean.TRUE)));
  
  /**
   * Get the {@code httpsCert} property.
   * The certificate to present to clients for SSL connections.
   * @see #httpsCert
   */
  public String getHttpsCert() { return getString(httpsCert); }
  
  /**
   * Set the {@code httpsCert} property.
   * The certificate to present to clients for SSL connections.
   * @see #httpsCert
   */
  public void setHttpsCert(String v) { setString(httpsCert, v, null); }

////////////////////////////////////////////////////////////////
// Property "requireHttpsForPasswords"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the {@code requireHttpsForPasswords} property.
   * Require an HTTPS connection when setting passwords
   * @see #getRequireHttpsForPasswords
   * @see #setRequireHttpsForPasswords
   */
  public static final Property requireHttpsForPasswords = newProperty(0, true, BFacets.make(BFacets.SECURITY, BBoolean.TRUE));
  
  /**
   * Get the {@code requireHttpsForPasswords} property.
   * Require an HTTPS connection when setting passwords
   * @see #requireHttpsForPasswords
   */
  public boolean getRequireHttpsForPasswords() { return getBoolean(requireHttpsForPasswords); }
  
  /**
   * Set the {@code requireHttpsForPasswords} property.
   * Require an HTTPS connection when setting passwords
   * @see #requireHttpsForPasswords
   */
  public void setRequireHttpsForPasswords(boolean v) { setBoolean(requireHttpsForPasswords, v, null); }

////////////////////////////////////////////////////////////////
// Property "xFrameOptions"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the {@code xFrameOptions} property.
   * The xframe option for preventing XFS attacks.
   * @see #getXFrameOptions
   * @see #setXFrameOptions
   */
  public static final Property xFrameOptions = newProperty(Flags.HIDDEN | Flags.READONLY, BXFrameOptionsEnum.sameorigin, BFacets.make(BFacets.SECURITY, BBoolean.TRUE));
  
  /**
   * Get the {@code xFrameOptions} property.
   * The xframe option for preventing XFS attacks.
   * @see #xFrameOptions
   */
  public BXFrameOptionsEnum getXFrameOptions() { return (BXFrameOptionsEnum)get(xFrameOptions); }
  
  /**
   * Set the {@code xFrameOptions} property.
   * The xframe option for preventing XFS attacks.
   * @see #xFrameOptions
   */
  public void setXFrameOptions(BXFrameOptionsEnum v) { set(xFrameOptions, v, null); }

////////////////////////////////////////////////////////////////
// Property "rememberUserIdCookie"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the {@code rememberUserIdCookie} property.
   * Makes the user id cookie persistent
   * @see #getRememberUserIdCookie
   * @see #setRememberUserIdCookie
   */
  public static final Property rememberUserIdCookie = newProperty(0, true, null);
  
  /**
   * Get the {@code rememberUserIdCookie} property.
   * Makes the user id cookie persistent
   * @see #rememberUserIdCookie
   */
  public boolean getRememberUserIdCookie() { return getBoolean(rememberUserIdCookie); }
  
  /**
   * Set the {@code rememberUserIdCookie} property.
   * Makes the user id cookie persistent
   * @see #rememberUserIdCookie
   */
  public void setRememberUserIdCookie(boolean v) { setBoolean(rememberUserIdCookie, v, null); }

////////////////////////////////////////////////////////////////
// Property "sameSite"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the {@code sameSite} property.
   * Configure SameSite attribute for cookies
   * @see #getSameSite
   * @see #setSameSite
   */
  public static final Property sameSite = newProperty(0, BSameSiteEnum.lax, BFacets.make(BFacets.SECURITY, BBoolean.TRUE));
  
  /**
   * Get the {@code sameSite} property.
   * Configure SameSite attribute for cookies
   * @see #sameSite
   */
  public BSameSiteEnum getSameSite() { return (BSameSiteEnum)get(sameSite); }
  
  /**
   * Set the {@code sameSite} property.
   * Configure SameSite attribute for cookies
   * @see #sameSite
   */
  public void setSameSite(BSameSiteEnum v) { set(sameSite, v, null); }

////////////////////////////////////////////////////////////////
// Property "allowUsernameAutocomplete"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the {@code allowUsernameAutocomplete} property.
   * Disables autocomplete on the username field of the login page if false
   * @see #getAllowUsernameAutocomplete
   * @see #setAllowUsernameAutocomplete
   */
  public static final Property allowUsernameAutocomplete = newProperty(0, true, null);
  
  /**
   * Get the {@code allowUsernameAutocomplete} property.
   * Disables autocomplete on the username field of the login page if false
   * @see #allowUsernameAutocomplete
   */
  public boolean getAllowUsernameAutocomplete() { return getBoolean(allowUsernameAutocomplete); }
  
  /**
   * Set the {@code allowUsernameAutocomplete} property.
   * Disables autocomplete on the username field of the login page if false
   * @see #allowUsernameAutocomplete
   */
  public void setAllowUsernameAutocomplete(boolean v) { setBoolean(allowUsernameAutocomplete, v, null); }

////////////////////////////////////////////////////////////////
// Property "loginTemplate"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the {@code loginTemplate} property.
   * The template for the browser login page.  If null, the default template is used.
   * @see #getLoginTemplate
   * @see #setLoginTemplate
   */
  public static final Property loginTemplate = newProperty(0, BTypeSpec.NULL, BFacets.make(BFacets.make(BFacets.ALLOW_NULL, BBoolean.TRUE), BFacets.make(BFacets.TARGET_TYPE, BString.make("web:LoginTemplate"))));
  
  /**
   * Get the {@code loginTemplate} property.
   * The template for the browser login page.  If null, the default template is used.
   * @see #loginTemplate
   */
  public BTypeSpec getLoginTemplate() { return (BTypeSpec)get(loginTemplate); }
  
  /**
   * Set the {@code loginTemplate} property.
   * The template for the browser login page.  If null, the default template is used.
   * @see #loginTemplate
   */
  public void setLoginTemplate(BTypeSpec v) { set(loginTemplate, v, null); }

////////////////////////////////////////////////////////////////
// Property "gzipEnabled"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the {@code gzipEnabled} property.
   * Enable gzip compression?
   * @see #getGzipEnabled
   * @see #setGzipEnabled
   */
  public static final Property gzipEnabled = newProperty(Flags.HIDDEN, false, null);
  
  /**
   * Get the {@code gzipEnabled} property.
   * Enable gzip compression?
   * @see #gzipEnabled
   */
  public boolean getGzipEnabled() { return getBoolean(gzipEnabled); }
  
  /**
   * Set the {@code gzipEnabled} property.
   * Enable gzip compression?
   * @see #gzipEnabled
   */
  public void setGzipEnabled(boolean v) { setBoolean(gzipEnabled, v, null); }

////////////////////////////////////////////////////////////////
// Property "logFileDirectory"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the {@code logFileDirectory} property.
   * If the web server supports request logging, then it should write logs to this directory.
   * @see #getLogFileDirectory
   * @see #setLogFileDirectory
   */
  public static final Property logFileDirectory = newProperty(Flags.READONLY, BOrd.make("file:^^webLogs"), null);
  
  /**
   * Get the {@code logFileDirectory} property.
   * If the web server supports request logging, then it should write logs to this directory.
   * @see #logFileDirectory
   */
  public BOrd getLogFileDirectory() { return (BOrd)get(logFileDirectory); }
  
  /**
   * Set the {@code logFileDirectory} property.
   * If the web server supports request logging, then it should write logs to this directory.
   * @see #logFileDirectory
   */
  public void setLogFileDirectory(BOrd v) { set(logFileDirectory, v, null); }

////////////////////////////////////////////////////////////////
// Property "clientEnvironments"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the {@code clientEnvironments} property.
   * Supported client environments.
   * @see #getClientEnvironments
   * @see #setClientEnvironments
   */
  public static final Property clientEnvironments = newProperty(0, new BClientEnvironments(), null);
  
  /**
   * Get the {@code clientEnvironments} property.
   * Supported client environments.
   * @see #clientEnvironments
   */
  public BClientEnvironments getClientEnvironments() { return (BClientEnvironments)get(clientEnvironments); }
  
  /**
   * Set the {@code clientEnvironments} property.
   * Supported client environments.
   * @see #clientEnvironments
   */
  public void setClientEnvironments(BClientEnvironments v) { set(clientEnvironments, v, null); }

////////////////////////////////////////////////////////////////
// Property "showStackTrace"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the {@code showStackTrace} property.
   * Will show exception stack traces in error responses when available if set to true.
   * @see #getShowStackTrace
   * @see #setShowStackTrace
   */
  public static final Property showStackTrace = newProperty(0, false, null);
  
  /**
   * Get the {@code showStackTrace} property.
   * Will show exception stack traces in error responses when available if set to true.
   * @see #showStackTrace
   */
  public boolean getShowStackTrace() { return getBoolean(showStackTrace); }
  
  /**
   * Set the {@code showStackTrace} property.
   * Will show exception stack traces in error responses when available if set to true.
   * @see #showStackTrace
   */
  public void setShowStackTrace(boolean v) { setBoolean(showStackTrace, v, null); }

////////////////////////////////////////////////////////////////
// Property "appletModuleCachingType"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the {@code appletModuleCachingType} property.
   * If files aren't available locally, enable jx browser JAR files to be loaded from an external URI.
   * @see #getAppletModuleCachingType
   * @see #setAppletModuleCachingType
   */
  public static final Property appletModuleCachingType = newProperty(0, BAppletModuleCachingType.host, null);
  
  /**
   * Get the {@code appletModuleCachingType} property.
   * If files aren't available locally, enable jx browser JAR files to be loaded from an external URI.
   * @see #appletModuleCachingType
   */
  public BAppletModuleCachingType getAppletModuleCachingType() { return (BAppletModuleCachingType)get(appletModuleCachingType); }
  
  /**
   * Set the {@code appletModuleCachingType} property.
   * If files aren't available locally, enable jx browser JAR files to be loaded from an external URI.
   * @see #appletModuleCachingType
   */
  public void setAppletModuleCachingType(BAppletModuleCachingType v) { set(appletModuleCachingType, v, null); }

////////////////////////////////////////////////////////////////
// Property "webStartConfig"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the {@code webStartConfig} property.
   * Configuration properties for clients using Java Web Start.
   * @see #getWebStartConfig
   * @see #setWebStartConfig
   */
  public static final Property webStartConfig = newProperty(0, new BWebStartConfig(), null);
  
  /**
   * Get the {@code webStartConfig} property.
   * Configuration properties for clients using Java Web Start.
   * @see #webStartConfig
   */
  public BWebStartConfig getWebStartConfig() { return (BWebStartConfig)get(webStartConfig); }
  
  /**
   * Set the {@code webStartConfig} property.
   * Configuration properties for clients using Java Web Start.
   * @see #webStartConfig
   */
  public void setWebStartConfig(BWebStartConfig v) { set(webStartConfig, v, null); }

////////////////////////////////////////////////////////////////
// Property "cacheConfig"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the {@code cacheConfig} property.
   * Configuration properties for clients using file resources that want additional caching.
   * @see #getCacheConfig
   * @see #setCacheConfig
   */
  public static final Property cacheConfig = newProperty(0, new BCacheConfig(), null);
  
  /**
   * Get the {@code cacheConfig} property.
   * Configuration properties for clients using file resources that want additional caching.
   * @see #cacheConfig
   */
  public BCacheConfig getCacheConfig() { return (BCacheConfig)get(cacheConfig); }
  
  /**
   * Set the {@code cacheConfig} property.
   * Configuration properties for clients using file resources that want additional caching.
   * @see #cacheConfig
   */
  public void setCacheConfig(BCacheConfig v) { set(cacheConfig, v, null); }

////////////////////////////////////////////////////////////////
// Property "warmupConfig"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the {@code warmupConfig} property.
   * Configuration for Web Warmup
   * @see #getWarmupConfig
   * @see #setWarmupConfig
   */
  public static final Property warmupConfig = newProperty(Flags.NON_CRITICAL, new BWebWarmupConfig(), null);
  
  /**
   * Get the {@code warmupConfig} property.
   * Configuration for Web Warmup
   * @see #warmupConfig
   */
  public BWebWarmupConfig getWarmupConfig() { return (BWebWarmupConfig)get(warmupConfig); }
  
  /**
   * Set the {@code warmupConfig} property.
   * Configuration for Web Warmup
   * @see #warmupConfig
   */
  public void setWarmupConfig(BWebWarmupConfig v) { set(warmupConfig, v, null); }

////////////////////////////////////////////////////////////////
// Property "hostnameRedirectSettings"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the {@code hostnameRedirectSettings} property.
   * Configuration for hostname redirection
   * @see #getHostnameRedirectSettings
   * @see #setHostnameRedirectSettings
   */
  public static final Property hostnameRedirectSettings = newProperty(0, new BHostnameRedirectSettings(), null);
  
  /**
   * Get the {@code hostnameRedirectSettings} property.
   * Configuration for hostname redirection
   * @see #hostnameRedirectSettings
   */
  public BHostnameRedirectSettings getHostnameRedirectSettings() { return (BHostnameRedirectSettings)get(hostnameRedirectSettings); }
  
  /**
   * Set the {@code hostnameRedirectSettings} property.
   * Configuration for hostname redirection
   * @see #hostnameRedirectSettings
   */
  public void setHostnameRedirectSettings(BHostnameRedirectSettings v) { set(hostnameRedirectSettings, v, null); }

////////////////////////////////////////////////////////////////
// Property "httpHeaderProviders"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the {@code httpHeaderProviders} property.
   * HTTP header configuration
   * @see #getHttpHeaderProviders
   * @see #setHttpHeaderProviders
   */
  public static final Property httpHeaderProviders = newProperty(0, new BHttpHeaderProviders(), null);
  
  /**
   * Get the {@code httpHeaderProviders} property.
   * HTTP header configuration
   * @see #httpHeaderProviders
   */
  public BHttpHeaderProviders getHttpHeaderProviders() { return (BHttpHeaderProviders)get(httpHeaderProviders); }
  
  /**
   * Set the {@code httpHeaderProviders} property.
   * HTTP header configuration
   * @see #httpHeaderProviders
   */
  public void setHttpHeaderProviders(BHttpHeaderProviders v) { set(httpHeaderProviders, v, null); }

////////////////////////////////////////////////////////////////
// Property "hostHeaderValidationSettings"
////////////////////////////////////////////////////////////////

  /**
   * Slot for the {@code hostHeaderValidationSettings} property.
   * @see #getHostHeaderValidationSettings
   * @see #setHostHeaderValidationSettings
   */
  public static final Property hostHeaderValidationSettings = newProperty(0, new BHostHeaderValidationSettings(), null);

  /**
   * Get the {@code hostHeaderValidationSettings} property.
   * @see #hostHeaderValidationSettings
   */
  public BHostHeaderValidationSettings getHostHeaderValidationSettings() { return (BHostHeaderValidationSettings)get(hostHeaderValidationSettings); }

  /**
   * Set the {@code hostHeaderValidationSettings} property.
   * @see #hostHeaderValidationSettings
   */
  public void setHostHeaderValidationSettings(BHostHeaderValidationSettings v) { set(hostHeaderValidationSettings, v, null); }

////////////////////////////////////////////////////////////////
// Action "resetAllConnections"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the {@code resetAllConnections} action.
   * @see #resetAllConnections()
   */
  public static final Action resetAllConnections = newAction(Flags.CONFIRM_REQUIRED, null);
  
  /**
   * Invoke the {@code resetAllConnections} action.
   * @see #resetAllConnections
   */
  public void resetAllConnections() { invoke(resetAllConnections, null, null); }

////////////////////////////////////////////////////////////////
// Type
////////////////////////////////////////////////////////////////
  
  @Override
  public Type getType() { return TYPE; }
  public static final Type TYPE = Sys.loadType(BWebService.class);

//@formatter:on
//endregion /*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

  public static final Logger log = Logger.getLogger(BWebService.class.getName());

  @Override
  public Type[] getServiceTypes()
  {
    return new Type[] { TYPE };
  }

  @Override
  public Feature getLicenseFeature()
  {
    return Sys.getLicenseManager().getFeature("tridium", "web");
  }

  public IWebEnv getWebEnv(WebOp op)
  {
    return getClientEnvironments().getWebEnv(op);
  }

  /**
   * Get the web server.
   *
   * @return the web server or {@code null} if there is no web server child.
   */
  public BWebServer getWebServer()
  {
    BWebServer[] servers = getChildren(BWebServer.class);
    return (servers.length == 0) ? null : servers[0];
  }

////////////////////////////////////////////////////////////////
// Factory
////////////////////////////////////////////////////////////////

  /**
   * Get the first Web Service or throw ServiceNotFoundException.
   * @since Niagara 4.10u2
   */
  public static BWebService getMainService()
  {
    // If the cached service's component space becomes null, that indicates the cached instance
    // was unmounted, so re-resolve it.  Looking up the component space should be a quick,
    // inexpensive call
    BWebService service = cachedService;
    if (service == null || service.getComponentSpace() == null || !service.isRunning())
    {
      try
      {
        service = (BWebService) Sys.getService(TYPE);
        cachedService = service;
      }
      catch (ServiceNotFoundException e)
      {
        // remove the reference so the underlying memory can be GCed
        cachedService = null;
        throw e;
      }
    }

    return service;
  }

////////////////////////////////////////////////////////////////
// Lifecycle
////////////////////////////////////////////////////////////////
  @Override
  public void stationStarted() throws Exception
  {
    super.stationStarted();

    if (getHostHeaderValidationSettings().getValidHostHeaders().isEmpty())
    {
      getHostHeaderValidationSettings().autoPopulateValidHostHeaders();
    }
  }

  @Override
  public void serviceStarted() throws Exception
  {
    super.serviceStarted();
    getComponentSpace().enableMixIn(BWebProfileConfig.TYPE);

    // BUserService.secureOnlyPasswordSet changed to BWebService.requireHttpsForPasswords in 4.1
    BUserService us = (BUserService)Sys.getService(BUserService.TYPE);
    Property prop = us.getProperty("secureOnlyPasswordSet");
    if (prop != null && !us.get(prop).equals(prop.getDefaultValue()))
    {
      setRequireHttpsForPasswords(((BIBoolean) us.get(prop)).getBoolean());
    }

    if (Sys.isStationStarted() && getHostHeaderValidationSettings().getValidHostHeaders().isEmpty())
    {
      getHostHeaderValidationSettings().autoPopulateValidHostHeaders();
    }

    WebUtil.setStationTheme();
  }

  @Override
  public void started()
  {
    if (getChildren(BUserDataConfig.class).length == 0)
    {
      add("UserDataConfig", new BUserDataConfig());
      setDisplayName(getProperty("UserDataConfig"),
        BFormat.make("%lexicon(web:userDataConfig)%"),
        null);
    }

    if (!Flags.has(this, xFrameOptions, Flags.USER_DEFINED_1))
    {
      getHttpHeaderProviders().getXFrameOptions().setXFrameOptions(getXFrameOptions());
      Flags.add(this, xFrameOptions, null, Flags.USER_DEFINED_1);
    }
  }

  @Override
  public void serviceStopped() throws Exception
  {
    super.serviceStopped();
    getComponentSpace().disableMixIn(BWebProfileConfig.TYPE);
  }

  @Override
  protected void enabled()
  {
    super.enabled();
    getWebServer().scheduleRestart();
  }

  @Override
  protected void disabled()
  {
    super.disabled();

    BWebServer server = getWebServer();
    server.post(() -> server.stopWebServer(null));
  }

////////////////////////////////////////////////////////////////
// RestrictedComponent
////////////////////////////////////////////////////////////////

  /**
   * This service type is only allowed to live under the
   * station's frozen ServiceContainer, but multiple instances are allowed.
   */
  @Override
  public void checkParentForRestrictedComponent(BComponent parent, Context cx)
  {
    BIRestrictedComponent.checkParentIsServiceContainer(parent, this);
  }

////////////////////////////////////////////////////////////////
// BISecurityInfoSource
////////////////////////////////////////////////////////////////

  @Override
  public BSecurityInfo getSecurityInfo()
  {
    BSecurityInfo info = new BSecurityInfo();
    info.setSourceName(getDisplayName(null));
    if (isMounted())
    {
      info.setHyperlink(getSlotPathOrd());
    }

    info.add(BSecurityInfo.CERTIFICATE_ALIAS, BString.make(getHttpsCert()));

    return info;
  }

////////////////////////////////////////////////////////////////
// Framework
////////////////////////////////////////////////////////////////

  @Override
  public void added(Property property, Context context)
  {
    if (isRunning())
    {
      UnauthenticatedCache.checkForInit(property.getName());
    }
  }

  @Override
  public void removed(Property property, BValue oldValue, Context context)
  {
    if (isRunning())
    {
      UnauthenticatedCache.checkForInit(property.getName());
    }
  }

  @Override
  public void changed(Property property, Context context)
  {
    super.changed(property, context);

    String name = property.getName();
    if (isRunning())
    {
      UnauthenticatedCache.checkForInit(property.getName());

      if(property.isFrozen())
      {
        BWebServer webServer = getWebServer();
        if (webServer != null)
        {
          webServer.webServiceConfigurationChanged(property, context);
          if (httpPort == property || httpsPort == property ||
            enabled == property || httpEnabled == property ||
            httpsEnabled == property)
          {
            updatePlatformSummaryFields();
          }
          else if (webStartConfig == property)
          {
            WebStartServlet.onWebStartConfigChanged();
          }
        }
      }
    }

    if (property.equals(xFrameOptions))
    {
      getHttpHeaderProviders().getXFrameOptions().setXFrameOptions(getXFrameOptions());
    }
  }

  @Override
  public void checkAdd(String name, BValue value, int flags, BFacets facets, Context context)
  {
    super.checkAdd(name, value, flags, facets, context);
    if ((value instanceof BWebServer) && (getWebServer() != null))
    {
      throw new LocalizableRuntimeException("web", "thereCanOnlyBeOne");
    }
  }

  @Override
  public void descendantsStarted() throws Exception
  {
    super.descendantsStarted();
    verifyWebServer();
  }

  @Override
  public void childParented(Property property, BValue newChild, Context context)
  {
    super.childParented(property, newChild, context);
    verifyWebServer();
  }

  @Override
  public void childUnparented(Property property, BValue oldChild, Context context)
  {
    super.childUnparented(property, oldChild, context);
    verifyWebServer();
  }

  private void verifyWebServer()
  {
    if (!isRunning())
    {
      return;
    }
    if (getWebServer() == null)
    {
      configFail(getLexicon().getText("web.noWebServer"));
    }
    else
    {
      configOk();
    }
  }

  @Override
  public Object fw(int x, Object a, Object b, Object c, Object d)
  {
    switch (x)
    {
      case Fw.CHANGED:
        if (isRunning())
        {
          Property p = (Property)a;
          if (httpPort == p || httpsPort == p)
          {
            updatePlatformSummaryFields();
          }
        }
        break;

      case Fw.STATION_STARTED:
        updatePlatformSummaryFields();
        break;

      case Fw.STARTED:
        fwStarted();
        break;
    }

    return super.fw(x, a, b, c, d);
  }

  private void fwStarted()
  {
    if (!Flags.has(this, hostHeaderValidationSettings, Flags.USER_DEFINED_1))
    {
      String hostNames = getHostHeaderValidationSettings().getValidHostHeaders();
      String updatedHostNames = Arrays.stream(hostNames.trim().split("\\s*[;,]\\s*"))
        .map((hostName) -> {
          if (IPAddressUtil.isIpv6AddressFormat(hostName))
          {
            return "[" + hostName + "]";
          }
          return hostName;
        })
        .collect(Collectors.joining(";"));
      getHostHeaderValidationSettings().setValidHostHeaders(updatedHostNames);
      Flags.add(this, hostHeaderValidationSettings, null, Flags.USER_DEFINED_1);
    }
  }

  private void updatePlatformSummaryFields()
  {
    // inform the platform about the http port and https port if they are enabled
    List<String> keys = new ArrayList<>();
    List<String> values = new ArrayList<>();

    int httpPort = -1;
    int httpsPort = -1;

    if (getHttpEnabled() && getEnabled())
    {
      httpPort = getHttpPort().getPublicServerPort();
    }

    if (getHttpsEnabled() && getEnabled())
    {
      httpsPort = getHttpsPort().getPublicServerPort();
    }

    keys.add("httpport");
    values.add(String.valueOf(httpPort));

    keys.add("httpsport");
    values.add(String.valueOf(httpsPort));

    Nre.getPlatform().reportSummaryFields(
      keys.toArray(new String[keys.size()]),
      values.toArray(new String[values.size()]));
  }

  /**
   * Return the HTTP Connection details for the given Context.
   * <p/>
   * This method is invoked through Fox when a BWbWebView needs to make
   * a connection to the Station through the Web Server.
   * <p/>
   * This method is reflectively invoked from the Fox Broker Channel
   * (BBrokerChannel#getHttpConnectionDetails(Context)).
   *
   * @param cx The Context used in the remote call.
   * @return BComponent HTTP Connection Details
   */
  @SuppressWarnings("unused")
  public BComponent getHttpConnectionDetails(Context cx)
  {
    BComponent details = new BComponent();

    //@since 4.4 these two properties have been included as well
    details.add("enabled", BBoolean.make(getEnabled()));
    details.add("httpEnabled", BBoolean.make(getHttpEnabled()));

    details.add("httpsEnabled", BBoolean.make(getHttpsEnabled()));
    details.add("httpsPort", BInteger.make(getHttpsPort().getPublicServerPort()));
    details.add("httpPort", BInteger.make(getHttpPort().getPublicServerPort()));

    return details;
  }

////////////////////////////////////////////////////////////////
// Actions
////////////////////////////////////////////////////////////////

  public void doResetAllConnections(Context context)
    throws Exception
  {
    new Thread()
    {
      @Override
      public void run()
      {
        BWebServer webServer = getWebServer();
        if (webServer != null)
        {
          webServer.invalidateAllSessions();
        }
      }
    }.start();
  }

  @Override
  public BIcon getIcon()
  {
    return icon;
  }

  private static final BIcon icon = BIcon.std("navOnly/webService.png");

////////////////////////////////////////////////////////////////
// SecurityDashboard
////////////////////////////////////////////////////////////////

  @Override
  public LexiconFormatInfo getSecurityDashboardSectionHeader(Context cx)
  {
    return LexiconFormatInfo.make(TYPE, "securityDashboard.sectionHeader");
  }

  @Override
  public BOrd getSecurityDashboardSectionHyperlinkOrd()
  {
    return getNavOrd().relativizeToSession();
  }

  @Override
  public int getSecurityDashboardItemsVersion()
  {
    return VERSION;
  }

  @Override
  public List<SecurityDashboardItem> getSecurityDashboardItems(Context cx)
  {
    List<SecurityDashboardItem> items = new ArrayList<>();
    addTlsItems(items);
    addCipherSuiteGroupItems(items);
    addStationCertificateItems(items);
    addPasswordParameters(items);
    addSameSiteItem(items);
    addHostHeaderValidationItem(items);
    return items;
  }

  private void addTlsItems(List<SecurityDashboardItem> items)
  {
    // check non TLS enabled and forwarding
    if (getHttpEnabled() && !getHttpsEnabled())
    {
      items.add(SecurityDashboardItem.makeAlert(TYPE,
        "securityDashboard.webTlsProtocolOff.summary",
        "securityDashboard.webTlsProtocolOff.description"));
    }

    if (!getHttpEnabled() && getHttpsEnabled())
    {
      items.add(SecurityDashboardItem.makeOk(TYPE,
        "securityDashboard.webTlsProtocolOn.summary",
        "securityDashboard.webTlsProtocolOn.description"));
    }

    if (getHttpEnabled() && getHttpsEnabled() && getHttpsOnly())
    {
      items.add(SecurityDashboardItem.makeWarning(TYPE,
        "securityDashboard.webForwarding.summary",
        "securityDashboard.webForwarding.description"));
    }

    if (getHttpEnabled() && getHttpsEnabled() && !getHttpsOnly())
    {
      items.add(SecurityDashboardItem.makeAlert(TYPE,
        "securityDashboard.webNonTlsProtocol.summary",
        "securityDashboard.webNonTlsProtocol.description"));
    }

    if (getHttpsEnabled())
    {
      BSecurityItemStatus status = getHttpsMinProtocol().equals(BSslTlsEnum.tlsv1_2) ?
        BSecurityItemStatus.securityStatusOK :
        BSecurityItemStatus.securityStatusWarning;
      items.add(SecurityDashboardItem.make(
        LexiconFormatInfo.make(TYPE, "securityDashboard.webTlsProtocol.summary", getHttpsMinProtocol()),
        LexiconFormatInfo.make(TYPE, "securityDashboard.webTlsProtocol.description", BSslTlsEnum.tlsv1_2),
        status));
    }
  }

  private void addCipherSuiteGroupItems(List<SecurityDashboardItem> items)
  {
    if (getHttpsEnabled())
    {
      BSecurityItemStatus status = getCipherSuiteGroup().equals(BTlsCipherSuiteGroup.recommended) ?
        BSecurityItemStatus.securityStatusOK :
        BSecurityItemStatus.securityStatusWarning;
      items.add(SecurityDashboardItem.make(TYPE,
        LexiconFormatInfo.make(TYPE, "securityDashboard.cipherSuiteGroup.summary", getCipherSuiteGroup()),
        "securityDashboard.cipherSuiteGroup.description", status));
    }
  }

  private void addStationCertificateItems(List<SecurityDashboardItem> items)
  {
    if (getHttpsEnabled())
    {
      try
      {
        X509Certificate certificate = CertManagerFactory.getInstance().getKeyStore()
          .getCertificate(getHttpsCert());

        try
        {
          certificate.checkValidity();

          // Check near expiration
          if ((certificate.getNotAfter().getTime() - new Date().getTime()) < MILLIS_IN_NINETY_DAYS)
          {
            items.add(SecurityDashboardItem.makeWarning(TYPE,
              LexiconFormatInfo.make(TYPE, "securityDashboard.webCertificateNearExpiry.summary",
                  BString.make(getHttpsCert()), BAbsTime.make(certificate.getNotAfter().getTime())),
              "securityDashboard.webCertificateNearExpiry.description"));
          }
          else
          {
            items.add(SecurityDashboardItem.makeOk(TYPE,
              LexiconFormatInfo.make(TYPE, "securityDashboard.webCertificateValid.summary", getHttpsCert()),
              "securityDashboard.webCertificateValid.description"));
          }
        }
        catch (CertificateExpiredException e)
        {
          items.add(SecurityDashboardItem.makeAlert(TYPE,
            LexiconFormatInfo.make(TYPE, "securityDashboard.webCertificateExpired.summary", getHttpsCert()),
            "securityDashboard.webCertificateExpired.description"));
        }
        catch (CertificateNotYetValidException e)
        {
          items.add(SecurityDashboardItem.makeAlert(TYPE,
            LexiconFormatInfo.make(TYPE, "securityDashboard.webCertificateNotYetValidException.summary",
              BString.make(getHttpsCert()), BAbsTime.make(certificate.getNotBefore().getTime())),
            "securityDashboard.webCertificateNotYetValidException.description"));
        }

        // Check self-signed
        if (isSelfSigned(certificate))
        {
          items.add(SecurityDashboardItem.makeWarning(TYPE,
            LexiconFormatInfo.make(TYPE, "securityDashboard.webCertificateSelfSigned.summary", getHttpsCert()),
            "securityDashboard.webCertificateSelfSigned.description"));
        }
        else
        {
          items.add(SecurityDashboardItem.makeOk(TYPE,
            LexiconFormatInfo.make(TYPE, "securityDashboard.webCertificateNotSelfSigned.summary", getHttpsCert()),
            "securityDashboard.webCertificateNotSelfSigned.description"));
        }
      }
      catch (Exception e)
      {
        items.add(SecurityDashboardItem.makeWarning(TYPE,
          LexiconFormatInfo
            .make(TYPE, "securityDashboard.webCertificateNotAvailable.summary", getHttpsCert()),
          "securityDashboard.webCertificateNotAvailable.description"));
      }
    }
  }

  private void addPasswordParameters(List<SecurityDashboardItem> items)
  {
    String summaryKey;
    BSecurityItemStatus status;
    if (getRequireHttpsForPasswords())
    {
      summaryKey = "securityDashboard.requireHttpsForPasswordTrue.summary";
      status = BSecurityItemStatus.securityStatusOK;
    }
    else
    {
      summaryKey = "securityDashboard.requireHttpsForPasswordFalse.summary";
      status = BSecurityItemStatus.securityStatusWarning;
    }
    String descriptionKey = "securityDashboard.requireHttpsForPassword.description";
    items.add(SecurityDashboardItem.make(TYPE, summaryKey, descriptionKey, status));
  }

  private void addSameSiteItem(List<SecurityDashboardItem> items)
  {
    BSameSiteEnum sameSite = getSameSite();
    BSecurityItemStatus status = sameSite == BSameSiteEnum.none ?
      BSecurityItemStatus.securityStatusWarning : BSecurityItemStatus.securityStatusOK;

    items.add(SecurityDashboardItem.make(TYPE,
      LexiconFormatInfo.make(TYPE, "securityDashboard.sameSite.summary", sameSite),
      "securityDashboard.sameSite.description", status));
  }

  /**
   * @since Niagara 4.4u4
   * @param items The list of security dashboard items
   */
  private void addHostHeaderValidationItem(List<SecurityDashboardItem> items)
  {
    if (getHostHeaderValidationSettings().getValidateHostHeader())
    {
      if (getHostHeaderValidationSettings().getValidHostHeaders().contains("*"))
      {
        items.add(SecurityDashboardItem.makeWarning(TYPE, "securityDashboard.hostHeader.wildcard.summary",
          "securityDashboard.hostHeader.description"));
      }
      else
      {
        items.add(SecurityDashboardItem.makeOk(TYPE, "securityDashboard.hostHeader.ok.summary",
          "securityDashboard.hostHeader.description"));
      }
    }
    else
    {
      items.add(SecurityDashboardItem.makeAlert(TYPE, "securityDashboard.hostHeader.validationOff.summary",
        "securityDashboard.hostHeader.description"));
    }
  }

  private boolean isSelfSigned(X509Certificate certificate)
  {
    return certificate.getIssuerDN().equals(certificate.getSubjectDN());
  }

////////////////////////////////////////////////////////////////
// Property Validator
////////////////////////////////////////////////////////////////

  @Override
  public IPropertyValidator getPropertyValidator(Property[] properties, Context context)
  {
    if(ArrayUtil.indexOf(properties, httpsCert) > -1)
    {
      return validator;
    }

    return super.getPropertyValidator(properties, context);
  }

  @Override
  public IPropertyValidator getPropertyValidator(Property property, Context context)
  {
    if(httpsCert.equals(property))
    {
      return validator;
    }

    return super.getPropertyValidator(property, context);
  }


////////////////////////////////////////////////////////////////
// Attributes
////////////////////////////////////////////////////////////////

  // Version 2: updated description for certificates in validity period (see NCCB-41313)
  private static final int VERSION = 2;

  private static final double MILLIS_IN_NINETY_DAYS = BRelTime.makeDays(90).getMillis();
  private CertAliasCasePropertyValidator validator = new CertAliasCasePropertyValidator(httpsCert.getName());

  private static BWebService cachedService;
}
