/*
 * @copyright 2005 Tridium Inc.
 */
package com.tridium.ddfIp.tcp.comm;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.net.Socket;
import java.net.SocketException;
import javax.baja.log.Log;
import javax.baja.nre.util.ByteArrayUtil;
import javax.baja.sys.BAbsTime;
import javax.baja.sys.BFacets;
import javax.baja.sys.Clock;
import javax.baja.sys.Sys;
import com.tridium.ddf.comm.singleTransaction.BIDdfSingleTransactionMgr;
import com.tridium.ddfIp.comm.BDdfIpAdapter;
import com.tridium.ddfIp.comm.BDdfIpAddressPort;
import com.tridium.platform.tcpip.BTcpIpPlatformService;

public class TcpSocketManager
    implements Runnable
{
////////////////////////////////////////////////////////////////
// TcpSocketManager
////////////////////////////////////////////////////////////////

  public TcpSocketManager(BDdfTcpCommunicator tcpComm )
  {
    this.ddfTcpCommunicator=tcpComm;

    if (isTraceOn())
      trace("entered constructor TcpSocketManager");

    // stem.out.println("TcpSocketManager started for "+devName());
    switchState(STATE_IDLE);

    bos = new PipedOutputStream();

    try
    {
      bin = new PipedInputStream(bos);
    }
    catch (Exception e)
    {
      ddfTcpCommunicator.getLog().error(e.toString(),e);
    }
  }

  /**
   * Defines whether or not extra trace information should be printed
   * to the station stdout.
   *
   * This method was limited to package prive access until Niagara AX 3.5.
   *
   * @since Niagara AX 3.5
   */
  protected boolean isTraceOn()
  {
    return getLog().isTraceOn();
  }

  public void startSocketManager()
  {
    running=true;

    if (isTraceOn())
      trace("entered method startSocketManager");

    switchState(STATE_NO_SOCKET);
  }

  /**
   * This method is intended mainly for internal use. Advanced
   * developers who wish to customize this class may override
   * or call this method from a subclass.
   *
   * Indicates whether or not the socket manager currently has
   * an open, client socket connection to the field-device or server.
   *
   * This method was limited to private access until Niagara AX 3.5.
   *
   * @since Niagara AX 3.5
   */
  protected boolean isConnected()
  {
    return (state == STATE_GOT_SOCKET);
  }

  /**
   * This method is intended mainly for internal use. Advanced
   * developers who wish to customize this class may override
   * or call this method from a subclass.
   *
   * Changes the internal state of the socket manager. Valid
   * states are {@link #STATE_GOT_SOCKET}, {@link #STATE_IDLE},
   * {@link #STATE_NO_SOCKET}.
   *
   * This method was limited to private access until Niagara AX 3.5.
   *
   * @param newState the new state
   *
   * @since Niagara AX 3.5
   */
  protected void switchState(int newState)
  {
    if (isTraceOn())
    {
      trace("entered method switchState");
      trace("old state = " + state);
      trace("new state = " + newState);
    }
    if (state == newState)
      return;

    state = newState;

    if (newState == STATE_NO_SOCKET)
    {
      numConnectionFailures = 0; // Resets the number of connection failures so that the send logic can use this
    }

    synchronized (idleMonitor)
    {
      idleMonitor.notifyAll(); // Wakes up the socket manager if it is idling
    }
  }

  public void stopSocketManager()
  {
    if (isTraceOn())
      trace("entered method stopSocketManager");
    switchState(STATE_IDLE);
  }

  /**
   * This method is intended mainly for internal use. Advanced
   * developers who wish to customize this class may override
   * or call this method from a subclass.
   *
   * This is implementation of the idle state.  This method is called
   * by the {@link #finiteStateMachine()} method if the internal state is
   * {@link #STATE_IDLE}
   *
   * This method was limited to private access until Niagara AX 3.5.
   *
   * @since Niagara AX 3.5
   */
  protected void doIdle()
  {
    closeSocket();
    nullStreams();

    // receiveFinal();
    synchronized (idleMonitor)
    {
      try
      {
        idleMonitor.wait(5000); // Waits for a number of seconds or until the idleMonitor is notified
      }
      catch (InterruptedException e)
      {
      }
    }

  }

  /**
   * This method is intended mainly for internal use. Advanced
   * developers who wish to customize this class may override
   * or call this method from a subclass.
   *
   * This method is called in a continuous loop on the socket maanager's
   * own thread.
   *
   * It essentially checks the internal state and either calls
   * {@link #doIdle()}, {@link #initSocketConnection()}, or
   * {@link #readMessage()}.
   *
   * This method was limited to private access until Niagara AX 3.5.
   *
   * @since Niagara AX 3.5
   */
  protected void finiteStateMachine()
  {
    if (isTraceOn())
      trace("entered method finiteStateMachine()");

    switch (state)
    {

    case STATE_IDLE:
      doIdle();
      break;

    case STATE_NO_SOCKET:
      if (ddfTcpCommunicator.getDdfTransactionMgr() instanceof BIDdfSingleTransactionMgr)
        waitForeverForSend(); // Waits forever for the application to request this driver to send something, if nothing ever sent then we don't need to open the socket (in single-transaction mode)!
      initSocketConnection(); // Attempts to initialize a socket connect
      break;

    case STATE_GOT_SOCKET:
      if (ddfTcpCommunicator.getDdfTransactionMgr() instanceof BIDdfSingleTransactionMgr) // If only one
        waitLittleWhileForSend(); // Waits a little while for driver to send (works for a master-slave protocol when the driver acts as master). This adds an extra layer of protection to keep the thread from spinning.
      readMessage(); // Reads a message from the input stream
      break;
    }
  }

  /**
   * This method is intended mainly for internal use. Advanced
   * developers who wish to customize this class may override
   * or call this method from a subclass.
   *
   * This method was limited to private access until Niagara AX 3.5.
   *
   * @since Niagara AX 3.5
   */
  protected String getPrefix()
  {
    return "";
  }

  /**
   * This method is intended mainly for internal use. Advanced
   * developers who wish to customize this class may override
   * or call this method from a subclass.
   *
   * This method is called throughout various locations in this class
   * to print trace text to the station standard output. It is usually
   * only called if the {@link #isTraceOn()} method returns true.
   *
   * Subclasses may override this method to customize each individual
   * line of trace that is printed to the station stdout.
   *
   *
   * This method was limited to private access until Niagara AX 3.5.
   *
   * @param details the text to print to the station stdout as a trace.
   *
   * @since Niagara AX 3.5
   */
  protected void trace(String details)
  {
    getLog().trace(getPrefix() + details + "[timestamp=" + Clock.ticks() + "]");
  }

  /**
   * This method is intended mainly for internal use. Advanced
   * developers who wish to customize this class may override
   * or call this method from a subclass.
   *
   * This gets a reference to the log object that is used to
   * print information to standard output or log errors in the
   * station.
   *
   * This method was limited to private access until Niagara AX 3.5.
   *
   * @return a reference to the log object that is used to
   * print information to standard output or log errors in the
   * station.
   *
   * @since Niagara AX 3.5
   */
  protected Log getLog()
  {
    return Log.getLog(ddfTcpCommunicator.getLog().getLogName()+"_TcpSocketManager");
  }

  /**
   * This method is intended mainly for internal use. Advanced
   * developers who wish to customize this class may override
   * or call this method from a subclass.
   *
   * This method reads a chunk of bytes from the input stream that
   * is connected to the socket. The chunk of bytes is then passed
   * to the internal {@link #bos}, which is piped to the internal
   * {@link #bin}, from which the driver's receiver / transaction mgr
   * component reads bytes (by calling the {@link #readByte()} method)
   * while attempting to extract recognized frames.
   *
   * This method is called by the {@link #readMessage()} method, which
   * is called by the {@link #finiteStateMachine()} method, which is
   * called continuously on the socket manager's own thread when the
   * internal {@link #state} is {@value #STATE_GOT_SOCKET}.
   *
   * This method was limited to private access until Niagara AX 3.5.
   *
   * @since Niagara AX 3.5
   */
  protected void readMessageFromStream()
  {
    synchronized(ioMonitor)
    {
      if (isTraceOn())
        trace("entered method readMessageFromStream");

      final byte ibuf[] = new byte[261];
      final int rxSize;

      try
      {
        if (isTraceOn())
          trace("Reading from input stream, inStream=" + inStream);

        rxSize = inStream.read(ibuf, 0, 261);

        if (isTraceOn())
          trace("finished read from instream, rxSize=" + rxSize);

          // 05/15/2003 : LP : RxSize seems to be -1 on Win32 when TCP slave closes connection
        if (rxSize == -1)
        {
           if (isTraceOn()) trace("bad read! switching state to force reinitialize socket");
           switchState(STATE_NO_SOCKET);
        }
        else
        {
          timeRxEnd = System.currentTimeMillis();

          if (isTraceOn())
          {
            StringWriter s = new StringWriter();
            PrintWriter p = new PrintWriter(s);
            ByteArrayUtil.hexDump(p, ibuf, 0, rxSize);
            p.flush();
            p.close();
            String traceByteArrayPrintOut = s.toString();
            trace(":" + ddfTcpCommunicator.getSlotPath() + ":"+getDebugCurTimeSecAndMs()+ ":RX:\n"+traceByteArrayPrintOut+"\nTcp/Ip response time(milliSec) = " + (timeRxEnd - timeTxEnd));
          }

          numOutstandingRequests = 0;
          bos.write(ibuf, 0, rxSize);
        }
      }
      catch (Exception e)
      {
        processReadStreamException(e);
      }
      considerReinitialize();
    }
  }

  /**
   * The Ddf Transaction layer calls this method in a tight loop.
   * This method blocks until a byte is available (or doesn't block
   * if bytes are already available) and returns the next next
   * available byte.
   *
   * @return the next available incoming byte from the Tcp/Ip socket
   *
   * @throws Exception
   */
  public int readByte()
    throws Exception
  {
    // NOTE: The 'bin' is connected to the 'bos'. The private 'readMessage'
    // method writes all received chunks of data into the 'bos', which
    // gets extracted right here.
    return bin.read();
  }

  /**
   * This method is intended mainly for internal use. Advanced
   * developers who wish to customize this class may override
   * or call this method from a subclass.
   *
   * This method is called by the {@link #readMessageFromStream()}
   * method in the event that an exception occurs during an attempt
   * to read a chunk of bytes from the socket. The particular exception
   * that occurred is passed to this method. Basically, if the exception
   * is an instance of standard SocketException then the internal state
   * is changed to {@link #STATE_NO_SOCKET} thereby causing subsequent
   * iterations through the {@link #finiteStateMachine()} method to
   * attempt to open a new socket to the field-device or server.
   *
   * This method was limited to private access until Niagara AX 3.5.
   *
   * @param e the exception that occurred in the
   * {@link #readMessageFromStream()} method while trying to read a chunk
   * of bytes from the socket.
   *
   * @since Niagara AX 3.5
   */
  protected void processReadStreamException(Exception e)
  {
    if (isTraceOn()) trace("entered method processReadStreamException");

    if (e instanceof SocketException)
    {
      trace("Tcp/Ip SOCKET connection lost to " + getPrefix() + "!!!");
      switchState(STATE_NO_SOCKET);
    }
    else if (e instanceof java.lang.NullPointerException)
    {
      // Do nothing specific
    }
    else if (e instanceof java.io.InterruptedIOException)
    {
      // Do nothing specific
    }
    else
    {
      ddfTcpCommunicator.getLog().error(e.toString(),e);
    }
  }

  /**
   * This method is intended mainly for internal use. Advanced
   * developers who wish to customize this class may override
   * or call this method from a subclass.
   *
   * This method is called by the {@link #readMessageFromStream()} after
   * reading (or at least attempting to read) a chunk of bytes from the
   * socket. The default functionality of this method is to set the internal
   * {@link #state} back to {@link #STATE_NO_SOCKET} <b>if</b> more than
   * one driver request has been transmitted without the
   * {@link #readMessageFromStream()} method at least receiving some bytes
   * in reply. This is tracked via the {@link #numOutstandingRequests}
   * variable.
   *
   * Developers will probably need to override this method if the driver
   * is does not feature a master-slave communication protocol. If the
   * driver does not feature a master-slave communication protocol then
   * this method should probably be overridden with an empty implementation.
   *
   * This method was limited to private access until Niagara AX 3.5.
   *
   * @since Niagara AX 3.5
   */
  protected void considerReinitialize()
  {
    if (isTraceOn()) trace("entered method considerReinitialize");
    // If more than one message sent without reply then
    // Reinitialize!

    if (numOutstandingRequests > 1)
      switchState(STATE_NO_SOCKET);

    // TODO: If the local adapter or if the destinationAdress changes
    // (by the user -- of course) then this needs to also switch to STATE_NO_SOCKET
  }

  /**
   * This method is intended mainly for internal use. Advanced
   * developers who wish to customize this class may override
   * or call this method from a subclass.
   *
   * This method is called by the {@link #readMessage()} method just
   * prior to calling the {@link #readMessageFromStream()} method to
   * read a chunk of bytes from the socket. This method calls
   * 'setSoTimeout()' on the underlying socket and passes in the
   * value that is returned by {@link #getResponseTimeout()}.
   *
   * This method was limited to private access until Niagara AX 3.5.
   *
   * @since Niagara AX 3.5
   */
  protected void setNextReadTimeout()
  {
    if (isTraceOn()) trace("entered method setNextReadTimeout()");
    try
    {
      commSocket.setSoTimeout((int)getResponseTimeout());
    }
    catch (Exception e)
    {
      if (isTraceOn()) trace("exception caught setting so timeout, e=" + e);
    }
  }

  /**
   * This method is intended mainly for internal use. Advanced
   * developers who wish to customize this class may override
   * or call this method from a subclass.
   *
   * This method is called from the {@link #finiteStateMachine()} method
   * continuously in a loop while the internal {@link #state} is
   * {@link #STATE_GOT_SOCKET}. This method calls
   * {@link #setNextReadTimeout()} before it delegates most of its
   * functionality to the {@link #readMessageFromStream()} method
   *
   * This method was limited to private access until Niagara AX 3.5.
   *
   * @since Niagara AX 3.5
   */
  protected void readMessage()
  {
    setNextReadTimeout();
    readMessageFromStream();
  }

  /**
   * This is the entry point for the socket manager's own thread.
   *
   * This basically calls {@link #finiteStateMachine()} in a continuous
   * loop.
   *
   */
  public void run()
  {
    if (isTraceOn()) trace("entered method run");
    while (running)
    {
      finiteStateMachine();
    }
  }

  /**
   * This method is intended mainly for internal use. Advanced
   * developers who wish to customize this class may override
   * or call this method from a subclass.
   *
   * Closes the socket.
   *
   * This method was limited to private access until Niagara AX 3.5.
   *
   * @since Niagara AX 3.5
   */
  protected void closeSocket()
  {
    if (isTraceOn()) trace("entered method closeSocket");

    if (commSocket != null)
    { // Closes old connection
      try
      {
        commSocket.close();
      }
      catch (Exception e)
      {
        if (isTraceOn()) trace("exception caught from closeSocket() while closing commSocket" + e);
      }
    }
  }

  /**
   * This method is intended mainly for internal use. Advanced
   * developers who wish to customize this class may override
   * or call this method from a subclass.
   *
   * This method is called by {@link #initSocketConnection()} if it
   * fails to establish a client TCP socket to the field-device or
   * server.
   *
   * This method basically calls {@link #nullStreams()} and
   * {@link #closeSocket()}.
   *
   * This method was limited to private access until Niagara AX 3.5.
   *
   * @param e the exception that occurs when attempting to establish
   * the client TPC socket connection to the field-device or server.
   *
   * @since Niagara AX 3.5
   */
  protected void failureSocketConnectionInit(Exception e)
  {
    if (isTraceOn())
    {
      trace("entered method failureSocketConnectionInit");
      trace("" + getDebugCurTimeSecAndMs() + " Tcp/Ip - Cannot open socket connection to " + getPrefix() + ".");
    }
    numConnectionFailures++; // Increments failure count

    nullStreams();
    closeSocket();
  }

  /**
   * This method is intended mainly for internal use. Advanced
   * developers who wish to customize this class may override
   * or call this method from a subclass.
   *
   * This method notifies the {@link #connectMonitor} object of a
   * successful socket connection being established. This comes into
   * play in the event that the driver tries to transmit before the
   * socket connection is established (to make the driver wait for
   * the socket connection to be established before transmitting).
   *
   * This method was limited to private access until Niagara AX 3.5.
   *
   * @since Niagara AX 3.5
   */
  protected void notifyConnectMonitor()
  { // Notifies the connectMonitor object so that other threads waiting on
    // Socket connection can be awakened
    synchronized (connectMonitor)
    {
      connectMonitor.notifyAll();
    }
  }

  /**
   * This method is intended mainly for internal use. Advanced
   * developers who wish to customize this class may override
   * or call this method from a subclass.
   *
   * This method is called by the {@link #initSocketConnection()} method
   * as a final step after successfully establishing a socket connection
   * to a field-device or server. The method basically changes the internal
   * {@link #state} to {@link #STATE_GOT_SOCKET}, calls
   * {@link #notifyConnectMonitor()}, and initializes
   * {@link #numOutstandingRequests} to zero.
   *
   * This method was limited to private access until Niagara AX 3.5.
   *
   * @since Niagara AX 3.5
   */
  protected void successSocketConnectionInit()
  {
    if (isTraceOn())
    {
      trace("entered method successSocketConnectionInit");
      trace("" + getDebugCurTimeSecAndMs() + " Tcp/Ip - socket connection established to " + getPrefix() + ".");
    }

    switchState(STATE_GOT_SOCKET);
    numOutstandingRequests = 0;
    notifyConnectMonitor(); // Wakes any threads waiting for socket connection
  }

  /**
   * This method is intended mainly for internal use. Advanced
   * developers who wish to customize this class may override
   * or call this method from a subclass.
   *
   * Gets the text that the integrator or user has entered as the Tcp/Ip
   * address of the field-device or server.
   *
   * This method was limited to private access until Niagara AX 3.5.
   *
   * @return tddfTcpCommunicator.getTcpIpComm().getDestinationAddress().getIpAddress();
   *
   * @since Niagara AX 3.5
   */
  protected String getDestAddress()
  {
    return ddfTcpCommunicator.getTcpIpComm().getDestinationAddress().getIpAddress();
  }

  /**
   * This method is intended mainly for internal use. Advanced
   * developers who wish to customize this class may override
   * or call this method from a subclass.
   *
   * Gets the number of millis to wait / time-out whenever trying to
   * establish a Tcp/Ip connection to the field-device or server.
   *
   * By default, this method returns the socket connection timeout value
   * as entered by the user or integrator
   *
   * This method was limited to private access until Niagara AX 3.5.
   *
   * @return (int)ddfTcpCommunicator.getTcpIpComm().getSocketConnectionTimeout().getMillis();
   *
   * @since Niagara AX 3.5
   */
  protected int getConnectionTimeout()
  {
    return (int)ddfTcpCommunicator.getTcpIpComm().getSocketConnectionTimeout().getMillis();
  }

  /**
   * This method is intended mainly for internal use. Advanced
   * developers who wish to customize this class may override
   * or call this method from a subclass.
   *
   * Indicates the amount of time to wait before timing out during any
   * attempt to read bytes from the underlying socket.
   *
   * By default, this method returns the greater of 100 millis or
   * whatever the user enters on the Tcp Receiver component for
   * the response timeout duration.
   *
   * This method was limited to private access until Niagara AX 3.5.
   *
   * @return the greater of 100 and ddfTcpCommunicator.getReceiver().getResponseTimeout().getMillis();
   *
   * @since Niagara AX 3.5
   */
  protected long getResponseTimeout()
  {
    long respTimeout = ddfTcpCommunicator.getReceiver().getResponseTimeout().getMillis();

    if (respTimeout < 100) // This is an extra layer of protection to keep the socket manager thread from spinning.
      respTimeout = 100;

    return respTimeout;
  }

  /**
   * This method is intended mainly for internal use. Advanced
   * developers who wish to customize this class may override
   * or call this method from a subclass.
   *
   * Performs an ICMP ping on the host that is identified by the given
   * string. This method is called before attempting to establish a
   * socket connection to the field-device or server. It is important
   * to ICMP ping the field-device or server (if at all possible) in order
   * to prevent the underlying operating system from taking an indefinite,
   * undefined amount of time to timeout the socket connection. This is
   * a safeguard against side-effects that could occur due to the varying
   * underlying implementations of the Java socket API. During development,
   * it was observed that some operating systems will block the thread that
   * attempts to connect a socket to a non-existent host for an indefinite
   * or unpredictably long duration. Tridium's Tcp drivers, such as ModbusTCP
   * perform much better on unreliable networks with this safeguard in place.
   *
   * This method was limited to private access until Niagara AX 3.5.
   *
   * @param userIpText the name of the host on which to perform an ICMP
   * ping. This can typically be an IP address or a DNS hostname.
   *
   * @return true if the field-device or server responds to ICMP ping
   * within one second (meaning that it is relatively close and probably
   * currently reliably connected on the network).
   *
   * @since Niagara AX 3.5
   */
  protected boolean icmpPing(String userIpText)
  {
//    BTcpIpPlatformService platTcpIp = (BTcpIpPlatformService)Sys.getService(BTcpIpPlatformService.TYPE);
//
//    platTcpIp.lease();
//
//    try
//    {
//      BInteger pingTime = platTcpIp.ping(new BPingArgs(userIpText, 1));
//      return pingTime.getInt()>=0;
//    }
//    catch (Exception e)
//    {
//      return false;
//    }

    //  The ping() method has been removed from the TcpIpPlatformService, so we cannot do an ICMP
    //  ping any longer.  In refactoring for rt/wb split, I've made this always return true, so
    //  existing drivers should work, although degraded somewhat because the check if a device is
    //  there will always return true.  Anyone using the devIpDriver SHOULD MOVE TO NDRIVER!
    return true;
  }

  /**
   * This method is intended mainly for internal use. Advanced
   * developers who wish to customize this class may override
   * or call this method from a subclass.
   *
   * This method is called by the {@link #initSocketConnection()} method to
   * open a Tcp/Ip client socket connection to the field-device or server.
   *
   * This method was limited to private access until Niagara AX 3.5.
   *
   * @throws Exception any exception passed along from the underlying
   * Niagara AX and/or Java socket API.
   *
   * @throws IOException if the field-device or server does not respond to
   * ICMP ping.
   *
   * @throws SocketException if the {@link #getDestAddress()} is null,
   * empty string. or unspecified.
   *
   * @since Niagara AX 3.5
   */
  protected void connectSocket() throws Exception
  {
    if (isTraceOn()) trace("entered method connectSocket");

    String userIpText = getDestAddress();

    if (isTraceOn()) trace("userIpText=" + userIpText);

    // Sanity check upon userIpText
    if (userIpText == null || userIpText.length() <= 0 ||
        userIpText.equals(BDdfIpAdapter.DEFAULT_IP_ADDR) ||
        userIpText.equals(BDdfIpAddressPort.ipAddress.getDefaultValue().toString()))
    {
      if (isTraceOn()) trace("Could not conclude that the IP address [" + userIpText + "] is valid.");
      throw new SocketException("Could not conclude that the IP address [" + userIpText + "] is valid.");
    }

    // 04/14/2003 : Lenard : Driver will attempt an ICMP "ping" (i.e. same as DOS ping command).
    // If Niagara platform supports ping, then the device must respond
    // To ping before attempting to establish a TCP socket connection to it.
    if(isTraceOn())trace("Pinging  device (same ping as DOS cmd prompt would use) using 500 ms. ping timeout...");

    // Pings the device
    BTcpIpPlatformService platTcpIp = (BTcpIpPlatformService)Sys.getService(BTcpIpPlatformService.TYPE);
    platTcpIp.lease();

    if (!icmpPing(userIpText))
    { // If no response to ping, this throws IOException thereby bailing out of this method
      if(isTraceOn())trace("--- NO RESPONSE to ping (same procedure as DOS cmd prompt would use)!!! ---");
      throw new IOException("Device did not respond to ping (same ping as DOS cmd prompt would use)!");
    }
    else
    { // If we got a response to ping, this method continues...
      if(isTraceOn())trace("Device responded to ping (same ping as DOS cmd prompt would use)!");
    }

    if(isTraceOn())trace("getting new socket");
    commSocket = ddfTcpCommunicator.getTcpIpComm().createSocket();
    // TODO: Consider adding a status string to the ddf tcp helper.
  }

  /**
   * This method is intended mainly for internal use. Advanced
   * developers who wish to customize this class may override
   * or call this method from a subclass.
   *
   * Called by {@link #initSocketConnection()},
   * {@link #failureSocketConnectionInit(Exception)}, and {@link #doIdle()}
   * to set the underlying references to {@link #inStream} and
   * {@link #outStream} to null.
   *
   * This method was limited to private access until Niagara AX 3.5.
   *
   * @since Niagara AX 3.5
   */
  protected void nullStreams()
  {
    if(isTraceOn())trace("entered method nullStreams");

    inStream = null;
    outStream = null;
  }

  /**
   * This method is intended mainly for internal use. Advanced
   * developers who wish to customize this class may override
   * or call this method from a subclass.
   *
   * This method is called by the {@link #initSocketConnection()} method
   * after opening a Tcp/Ip client socket connection to the field-device
   * or server. This method sets the underlying {@link #inStream} and
   * {@link #outStream} references to refer to the underlying input
   * and output streams for the socket.
   *
   * @throws IOException any IOException that is passed along from the
   * Java socket API (commSocket.getInputStream() or
   * commSocket.getOutputStream())
   *
   * This method was limited to private access until Niagara AX 3.5.
   *
   * @since Niagara AX 3.5
   */
  protected void createStreams() throws IOException
  {
    if(isTraceOn())trace("entered method createStreams");

    // get input & output streams
    if(isTraceOn())trace("getting new input stream");
    inStream = commSocket.getInputStream();

    if(isTraceOn())trace("getting new output stream");
    outStream = commSocket.getOutputStream();
  }

  /**
   * This method is intended mainly for internal use. Advanced
   * developers who wish to customize this class may override
   * or call this method from a subclass.
   *
   * This method is called in a continuous loop from the
   * {@link #finiteStateMachine()} method while the {@link #state}
   * is {@link #STATE_NO_SOCKET}.
   *
   * This method was limited to private access until Niagara AX 3.5.
   *
   * @since Niagara AX 3.5
   */
  protected synchronized void initSocketConnection()
  {
    if(isTraceOn())trace("entered method initSocketConnection");
    try
    {
      nullStreams();

      closeSocket();

      connectSocket();

      createStreams();

      successSocketConnectionInit();
    }
    catch (Exception e)
    {
      failureSocketConnectionInit(e);
      try
      {
        Thread.sleep(1000);
      }
      catch (Exception ee)
      {
      }
    }
  }

  /**
   * This method is intended mainly for internal use. Advanced
   * developers who wish to customize this class may override
   * or call this method from a subclass.
   *
   * Notifies the {@link #readMonitor} object that it is time to
   * start attempting to receive a chunk of bytes from the Tcp/Ip
   * socket as a result of the driver transmitting a frame. This
   * only comes into play if the driver transaction manager component
   * is a single transaction manager.
   *
   * This method was limited to private access until Niagara AX 3.5.
   *
   * @since Niagara AX 3.5
   */
  protected void notifyReceiveMonitor()
  {
    synchronized (readMonitor)
    {
      readMonitor.notifyAll(); // Tells receive thread to start receiving
    }
  }


  /**
   * This method is intended mainly for internal use. Advanced
   * developers who wish to customize this class may override
   * or call this method from a subclass.
   *
   * This method is called by the {@link #finiteStateMachine()} method
   * if the {@link #state} is {@link #STATE_NO_SOCKET} and the transaction
   * manager is a single transactio manager. This idea is that there is no
   * need to establish the socket until or unless there is data to send in
   * a master-slave protocol.
   *
   * This method was limited to private access until Niagara AX 3.5.
   *
   * @since Niagara AX 3.5
   */
  protected void waitForeverForSend()
  {
    if(isTraceOn())trace("entered method waitForeverForSend");

    if (!messageSent)
    {
      synchronized (readMonitor)
      {
        try
        {
          readMonitor.wait(); // Waits forever for driver to send a message because it is probably wasteful to try to receive if the driver has not sent anything, since Modbus is a master-slave protocol and the driver is the master
        }
        catch (InterruptedException e)
        {
        }
      }
    }
    if(isTraceOn())trace("exiting method waitForeverForSend");
  }

  /**
   * This method is intended mainly for internal use. Advanced
   * developers who wish to customize this class may override
   * or call this method from a subclass.
   *
   * This method is called by the {@link #finiteStateMachine()} method
   * if the {@link #state} is {@link #STATE_GOT_SOCKET} and the transaction
   * manager is a single transactio manager. This idea is that there is no
   * need to even try to read from socket until or unless some data is sent
   * (at least for a master-slave protocol).
   *
   * This method was limited to private access until Niagara AX 3.5.
   *
   * @since Niagara AX 3.5
   */
  protected void waitLittleWhileForSend()
  {
    if(isTraceOn())trace("entered method waitLittleWhileForSend");

    if (!messageSent)
    {
      synchronized (readMonitor)
      {
        try
        {
          readMonitor.wait(1000); // Waits a little while for driver to send a message because it is probably wasteful to try to receive if the driver has not sent anything, since Modbus is a master-slave protocol and the driver is the master
        }
        catch (InterruptedException e)
        {
        }
      }
    }
    if(isTraceOn())trace("exiting method waitLittleWhileForSend");
  }

  /**
   * This method is intended mainly for internal use. Advanced
   * developers who wish to customize this class may override
   * or call this method from a subclass.
   *
   * This method is called make a thread wait for a socket connection to
   * be established. More specifically, this method waits for the given
   * amount of time for the {@link #connectMonitor} object to be notified.
   *
   * This method was limited to private access until Niagara AX 3.5.
   *
   * @param waitInterval number of milliseconds
   *
   * @since Niagara AX 3.5
   */
  protected void waitForConnect(long waitInterval)
  {
    synchronized (connectMonitor)
    {
      try
      {
        if(isTraceOn())trace("waitForConnect - waiting for up to " + waitInterval + " milliseconds.");

        // Waits a little while for socket manager to connect a socket; connectMonitor is notified by socket manager when socket is connected
        connectMonitor.wait(waitInterval);
        if(isTraceOn())trace("waitForConnect - woke up from wait");
      }
      catch (InterruptedException e)
      {
        if(isTraceOn())trace("waitForConnect interrupted while waiting");
      }
    }

  }

  /**
   * This method is intended mainly for internal use. Advanced
   * developers who wish to customize this class may override
   * or call this method from a subclass.
   *
   * This method is called by the {@link #writeOutputStream(byte[], int, int)}
   * method to make the transmitter thread wait for a socket connection to
   * be established before attempting to transmit bytes through the
   * socket.
   *
   * Waits up to {@link #getConnectionTimeout()} interval of time for the
   * {@link #connectMonitor} to be notified of a successful socket connection
   *
   * This method was limited to private access until Niagara AX 3.5.
   *
   * @since Niagara AX 3.5
   */
  protected void waitAWhileForConnect()
  {
    long waitInterval = getConnectionTimeout();

    // Gives it the greater of transactionTimeout of 13 seconds to timeout
    waitInterval = (waitInterval < 13000) ? 13000 : waitInterval;

    if(isTraceOn())trace("entered method waitLittleWhileForConnect");

    waitForConnect(waitInterval);

    if(isTraceOn())trace("exiting method waitLittleWhileForConnect");
  }

  /**
   * This method is intended mainly for internal use. Advanced
   * developers who wish to customize this class may override
   * or call this method from a subclass.
   *
   * Determines if this socket manager is presently in a mode where it is
   * attempting to establish socket connection (in other words, if the
   * {@link #state} is {@link #STATE_NO_SOCKET}
   *
   * Waits up to {@link #getConnectionTimeout()} interval of time for the
   * {@link #connectMonitor} to be notified of a successful socket connection
   *
   * This method was limited to private access until Niagara AX 3.5.
   *
   * @since Niagara AX 3.5
   */
  protected boolean isConnecting()
  {
    return (state == STATE_NO_SOCKET);
  }

  /**
   * This method is intended mainly for internal use. Advanced
   * developers who wish to customize this class may override
   * or call this method from a subclass.
   *
   * Determines if this socket manager is connecting for the first time
   *
   * Connecting for the first time is when the {@link #isConnecting()}
   * method returns true and when the {@link #numConnectionFailures} is
   * zero.
   *
   * This method was limited to private access until Niagara AX 3.5.
   *
   * @return true if this socket manager is connecting to the field-device
   * or server for the first time since boot.
   *
   * @since Niagara AX 3.5
   */
  protected boolean isConnectingFirstTime()
  {
    return (isConnecting() && (numConnectionFailures == 0));
  }


  /**
   * This method is called by BDdfTcpTransmitter.forceTransmit(...) in order
   * to transmit an array of bytes out of the Tcp/Ip socket.
   *
   * @param data the byte array to transmit bytes from.
   *
   * @param startIndex the index of the first byte in the given array
   * to transmit
   *
   * @param endIndex the last index of the given array to transmit.
   *
   * @throws Exception any exception that occurs in the underlying socket
   * API.
   */
  public void writeOutputStream(byte[] data, int startIndex, int endIndex) throws Exception
  {
    if(isTraceOn())trace("entered method writeOutputStream");
    /*
     * if (isConnectingFirstTime()) { waitLittleWhileForConnect(); }
     */
    if (isConnecting())
    {
      // Need to tell socket manager driver to init socket and then this needs to wait for connection
      synchronized (connectMonitor) // Synchronized on connectMonitor so that socket manager thread connect notify this until it is blocked in the connectMonitor.wait
      {
        notifyReceiveMonitor(); // Wakes up the socket manager thread, it is sleeping in the waitForeverForSendMethod that it called from finiteStateMachineMethod in state STATE_NO_SOCKET

        if (isConnectingFirstTime())
        {
          waitAWhileForConnect(); // Waits for the a relatively long period of time for the first socket connection
        }
        else
        { // If connecting for a subsequent connection, then we assume that the hardware should probably be online and
          // Therefore the TCP socket connection should be established rather quickly.
          // Waits for a relatively short period of time for the TCP socket connection
          waitForConnect(getResponseTimeout()); // Waits for connectMonitor to be notified
        }
      }
    }

    if (isConnected())
    {
      synchronized(ioMonitor)
      {
        numOutstandingRequests++;
        outStream.write(data, startIndex, endIndex);
        outStream.flush();
      }
      notifyReceiveMonitor();
    }
    else
    {
      throw new SocketException();
    }
  }

  /**
   * This method is intended mainly for internal use. Advanced
   * developers who wish to customize this class may override
   * or call this method from a subclass.
   *
   * Called when formatting trace data for standard output.
   *
   * This method was limited to private access until Niagara AX 3.5.
   *
   * @return a string detailing the exact moment with millisecond
   * granularity.
   *
   * @since Niagara AX 3.5
   *
   */
  protected String getDebugCurTimeSecAndMs()
  {
    return BAbsTime.now().toString(SHOW_SECONDS_AND_MILLIS);
  }

////////////////////////////////////////////////////////////////
// Attributes
////////////////////////////////////////////////////////////////

  /**
   * Unless advanced developers are significantly customizing
   * this class, then this field should not need to be directly
   * accessed.
   *
   * The {@link #doIdle()} method waits on this object to
   * be notified.
   *
   * The {@link #switchState(int)} method notifies this object.
   *
   * This field was limited to private access until Niagara AX 3.5.
   *
   * @since Niagara AX 3.5
   */
  protected Object idleMonitor = new Object();

  /**
   * Unless advanced developers are significantly customizing
   * this class, then this field should not need to be directly
   * accessed.
   *
   * Throughout the logic in this class for establishing the Tcp/Ip
   * connection, many methods wait on this monitor.
   *
   * This field was limited to private access until Niagara AX 3.5.
   *
   * @since Niagara AX 3.5
   */
  protected Object connectMonitor = new Object();

  /**
   * Unless advanced developers are significantly customizing
   * this class, then this field should not need to be directly
   * accessed.
   *
   * In general, safe access to the underlying socket should be made
   * indirectly through the public methods on this class.
   *
   * This is a reference to the underlying socket.
   *
   * This field was limited to private access until Niagara AX 3.5.
   *
   * @since Niagara AX 3.5
   */
  protected Socket commSocket;

  /**
   * Unless advanced developers are significantly customizing
   * this class, then this field should not need to be directly
   * accessed.
   *
   * In general, safe access to the underlying socket should be made
   * indirectly through the public methods on this class.
   *
   * This is a reference to the input stream that is tied
   * to the underlying socket (if one has been established).
   *
   * This field was limited to private access until Niagara AX 3.5.
   *
   * @since Niagara AX 3.5
   */
  protected InputStream inStream = null;

  /**
   * Unless advanced developers are significantly customizing
   * this class, then this field should not need to be directly
   * accessed.
   *
   * In general, safe access to the underlying socket should be made
   * indirectly through the public methods on this class.
   *
   * This is a reference to the output stream that is tied
   * to the underlying socket (if one has been established).
   *
   * This field was limited to private access until Niagara AX 3.5.
   *
   * @since Niagara AX 3.5
   */
  protected OutputStream outStream = null;

  /**
   * Unless advanced developers are significantly customizing
   * this class, then this field should not need to be directly
   * accessed.
   *
   * This variable is incremented each time a byte array is
   * transmitted. It is set to zero each time a series of
   * one or more bytes is received and upon initial establishment
   * of the socket. The {@link #considerReinitialize()} method
   * looks at this variable in deciding whether to revert the
   * {@link #state} to {@link #STATE_NO_SOCKET}.
   *
   * This field was limited to private access until Niagara AX 3.5.
   *
   * @since Niagara AX 3.5
   */
  protected int numOutstandingRequests = 0;

  /**
   * Unless advanced developers are significantly customizing
   * this class, then this field should not need to be directly
   * accessed.
   *
   * This variable is used to determine whether or not the socket
   * manager is attempting to establish a socket connection for the
   * very first time.
   *
   * This field was limited to private access until Niagara AX 3.5.
   *
   * @since Niagara AX 3.5
   */
  protected int numConnectionFailures = 0;

  /**
   * Unless advanced developers are significantly customizing
   * this class, then this field should not need to be directly
   * accessed.
   *
   * This is the internal state of the socket manager. This greatly
   * affects how the socket manager thread behaves. Possible values are
   * {@link #STATE_IDLE}, {@link #STATE_NO_SOCKET}, and
   * {@link #STATE_GOT_SOCKET}.
   *
   * @see #switchState(int)
   * @see #finiteStateMachine()
   *
   * This field was limited to private access until Niagara AX 3.5.
   *
   * @since Niagara AX 3.5
   */
  protected int state = STATE_IDLE;

  /**
   * Unless advanced developers are significantly customizing
   * this class, then this field should not need to be directly
   * accessed.
   *
   * This is a reference to the BDdfTcpCommunicator object that owns
   * this socket manager. This is used to get property values from the
   * station database.
   *
   * This field was limited to private access until Niagara AX 3.5.
   *
   * @since Niagara AX 3.5
   */
  protected BDdfTcpCommunicator ddfTcpCommunicator;

  /**
   * Unless advanced developers are significantly customizing
   * this class, then this field should not need to be directly
   * accessed.
   *
   * This variable is used in the {@link #run()} method to keep the
   * socket manager thread continuously looping under normal circumstances
   * and calling the {@link #finiteStateMachine()} continuously.
   *
   * This field was limited to private access until Niagara AX 3.5.
   *
   * @since Niagara AX 3.5
   */
  protected boolean running=true;

  /**
   * Unless advanced developers are significantly customizing
   * this class, then this field should not need to be directly
   * accessed.
   *
   * This is used to isolate the socket manager thread from
   * the BDdfTcpCommunicator's receiver thread. As th3 socket
   * manger independently read bytes from the underlying socket,
   * it writes the bytes to this output stream. This output
   * stream is piped to the {@link #bin}. The BDdfTcpCommunicator's
   * receiver reads bytes from the corresponding {@link #bin}.
   *
   * This field was limited to private access until Niagara AX 3.5.
   *
   * @see #bin
   *
   * @since Niagara AX 3.5
   */
  protected PipedOutputStream bos;

  /**
   * Unless advanced developers are significantly customizing
   * this class, then this field should not need to be directly
   * accessed.
   *
   * The BDdfTcpCommunicator's receiver reads from this input
   * stream to indirectly read bytes from the socket. This is
   * done in the {@link #readByte()} method.
   *
   * This field was limited to private access until Niagara AX 3.5.
   *
   * @see #bos
   *
   * @since Niagara AX 3.5
   *
   */
  protected PipedInputStream bin;

  /**
   * Unless advanced developers are significantly customizing
   * this class, then this field should not need to be directly
   * accessed.
   *
   * The 'receive' logic uses this object throughout the logic
   * for this class to wait until it is time to receive.
   *
   * This field was limited to private access until Niagara AX 3.5.
   *
   * @since Niagara AX 3.5
   */
  protected Object readMonitor = new Object();

  /**
   * Unless advanced developers are significantly customizing
   * this class, then this field should not need to be directly
   * accessed.
   *
   * This monitor ensures that underlying calls to the socket API
   * for sending and receiving are mutually exclusive operations.
   *
   * @since Niagara AX 3.5
   */
  protected Object ioMonitor = new Object();

  protected boolean messageSent = false;

  protected long timeTxEnd, timeRxEnd;

  protected BFacets SHOW_SECONDS_AND_MILLIS = BFacets.make(
        BFacets.make(BFacets.SHOW_SECONDS,true),
        BFacets.make(BFacets.SHOW_MILLISECONDS,true)
      );

////////////////////////////////////////////////////////////////
// Static values
////////////////////////////////////////////////////////////////

  public static final int STATE_IDLE = 0;

  public static final int STATE_NO_SOCKET = 1;

  public static final int STATE_GOT_SOCKET = 2;
}
