/*
 * Copyright 2008 Tridium, Inc. All Rights Reserved.
 */
package com.tridium.ddfHttp.comm.req;

import javax.baja.sys.BajaRuntimeException;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;

import com.tridium.ddf.comm.IDdfDataFrame;
import com.tridium.ddf.comm.req.BDdfReadRequest;
import com.tridium.ddf.comm.rsp.BIDdfResponse;
import com.tridium.ddf.comm.rsp.DdfResponseException;
import com.tridium.ddf.identify.BDdfIdParams;
import com.tridium.ddfHttp.comm.rsp.BDdfHttpInspectReadResponse;
import com.tridium.ddfHttp.identify.BIDdfHttpInspectUrl;

/**
 * This class is used to read the value of an HTTP driver data point
 * by inspecting (parsing/combing) the HTTP response for the data
 * value.  
 * 
 * The response will inspect (parse / comb) the HTTP response data for
 * the data point value by feeding the HTTP response data and a regular
 * expression for the data point through a Perl 5 regular expression
 * engine.
 * 
 * This assumes that the pointId of the proxyExt and/or the pingParameters of
 * the device implement BIDdfHttpInspectUrl. The URL is
 * determined based on the deviceId and the pointId.
 * 
 * If only the deviceId implements BIDdfHttpInspectUrl then
 * the URL from the deviceId will be used.
 * 
 * If only the pointId implements BIDdfHttpInspectUrl
 * then the URL from the pointId will be used.
 * 
 * If both the deviceId and pointId define a URL then
 * the URL for the pointId will be treated as being
 * relative to the URL for the deviceId. More specifically,
 * the HTTP connection will be established to the URL that
 * results by combining the deviceId URL with the pointId
 * URL.
 * 
 *  
 * @author    Lenard Perkins
 * @creation  01 Jan 08
 * @version   $Revision$ $Date: 02/19/2009 3:42:00 PM$
 * @since     Baja 1.0
 */
public class BDdfHttpInspectReadRequest
    extends BDdfReadRequest
{
  /*-
    class BDdfHttpInspectReadRequest
    {
    }
   -*/
/*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
/*@ $com.tridium.ddfHttp.comm.req.BDdfHttpInspectReadRequest(2950692683)1.0$ @*/
/* Generated Fri Feb 22 14:52:47 EST 2008 by Slot-o-Matic 2000 (c) Tridium, Inc. 2000 */

////////////////////////////////////////////////////////////////
// Type
////////////////////////////////////////////////////////////////
  
  public Type getType() { return TYPE; }
  public static final Type TYPE = Sys.loadType(BDdfHttpInspectReadRequest.class);

/*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/
  
  /**
   * The devHttpDriver framework uses the byte array that is returned
   * as the URL to browse to over the network. The corresponding response
   * will inspect (parse / comb) the HTTP response for data values.
   * 
   * The URL optionally encodes any data to be sent by placing a question mark after the web address
   * and then placing data fields as a series of key=value pairs that are delimited by the ampersand
   * character.
   * 
   * By default, the devHttpDriver framework will establish an HTTP GET connection to the URL. If
   * the URL begins with "POST " (that is to say, the string POST all upper case followed by a
   * space) then the devHttpDriver will establish an HTTP POST connection to the web address portion
   * of the URL (that is to say, the text on the left side of the question mark in the URL). In this
   * scenario, the data (that is to say, the text on the right side of the question mark in the URL)
   * will be placed into the HTTP data stream in the location where form data is typically placed
   * by Web Browsers.
   * 
   * This request class uses the URL that is defined on the proxyExt's readParameters the device's
   * deviceId.
   * 
   * Please review the javadoc for this class's declaration for more details about how the URL is
   * formed.
   */
  public byte[] toByteArray()
  {
    BDdfIdParams readParameters = getReadParameters();
    BDdfIdParams deviceId = getDeviceId();

    // Gets the 'URL' that might be encoded in the read parameters 
    String readUrl;
    if (readParameters instanceof BIDdfHttpInspectUrl)
    {
      readUrl = ((BIDdfHttpInspectUrl)readParameters).getUrl();
      if (readUrl.trim().length()==0 || readUrl.trim().equals("-"))
        readUrl = null;
      
    }
    else
      readUrl = null;
    
    // Gets the 'URL' that might be encoded in the device id
    String deviceUrl;
    if (deviceId instanceof BIDdfHttpInspectUrl)
    {
      deviceUrl = ((BIDdfHttpInspectUrl)deviceId).getUrl();
      if (deviceUrl.trim().length()==0 || deviceUrl.trim().equals("-"))
        deviceUrl = null;
    }
    else
      deviceUrl = null;
    
    // If no URL in neither readURL nor deviceUrl
    if (readUrl == null && deviceUrl == null)
      throw new BajaRuntimeException("No URL!");
    else
    {
      // Else, a URL is encoded in the readUrl and/or the deviceUrl
      
      // If a URL is encoded in both the readUrl and the deviceUrl
      // then this combines them both into a single URL (assuming
      // that the readUrl is relative to the deviceUrl)
      if (deviceUrl != null && readUrl !=null )
      {
        // Avoids placing a double set of slashes between the readUrl
        // and deviceUrl when they are combined
        if (deviceUrl.endsWith("/"))
          deviceUrl = deviceUrl.substring(0, deviceUrl.length()-1);
        if (readUrl.startsWith("/"))
          readUrl = readUrl.substring(1);
        
        // Combines the deviceUrl and the readUrl (readUrl assumed
        // to be relative in this scenario)
        urlAddress = deviceUrl + '/' + readUrl;
      }
      else if (deviceUrl == null)
      {
        // If no deviceUrl then assume the readUrl is the full
        // url by itself
        urlAddress = readUrl;
      }
      else if (readUrl == null)
      {
        // If no readUrl then assume the deviceUrl is the full url
        // by itself
        urlAddress = deviceUrl;
      }
      else // This shouldn't be possible based on our if statements above
        throw new IllegalStateException("Program Execution should not have made it here");
      
      // Returns the ascii bytes for the Url. This has special meaning to
      // the Ddf Http Communicator
      return urlAddress.getBytes();
    }
  }

  public BIDdfResponse processReceive(IDdfDataFrame ddfDataFrame) throws DdfResponseException
  {
    // The urlAddress should have already been assigned during the most
    // recent call to the 'toByteArray' method. This method is called by
    // the devDriver framework after it calls 'toByteArray' and receives
    // the subsequent response from the field-bus.
    return new BDdfHttpInspectReadResponse(ddfDataFrame, urlAddress); 
  }
  
  protected String urlAddress="";

}
