/*
 * Copyright 2008 Tridium, Inc. All Rights Reserved.
 */
package com.tridium.ddfHttp.comm.req;

import javax.baja.sys.BajaRuntimeException;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;

import com.tridium.ddf.comm.IDdfDataFrame;
import com.tridium.ddf.comm.req.BDdfPingRequest;
import com.tridium.ddf.comm.rsp.BIDdfResponse;
import com.tridium.ddf.comm.rsp.DdfResponseException;
import com.tridium.ddf.identify.BDdfIdParams;
import com.tridium.ddfHttp.comm.rsp.BDdfHttpInspectPingResponse;
import com.tridium.ddfHttp.identify.BIDdfHttpInspectUrl;

/**
 * This request class is used to perform a Niagara AX device
 * status ping scan by connecting to a URL over HTTP and
 * reading the HTTP response. This is associated to a driver's
 * HTTP device class through its deviceId property, whose value
 * is an instance of BDdfInspectDeviceId (which declares
 * this class as its ping request type)
 * 
 * This assumes that the deviceId and/or the pingParameters of
 * the device implement BIDdfHttpInspectUrl. The URL is
 * determined based on the deviceId and the pingParameters.
 * 
 * If only the deviceId implements BIDdfHttpInspectUrl then
 * the URL from the deviceId will be used.
 * 
 * If only the pingParameters implements BIDdfHttpInspectUrl
 * then the URL from the pingParameters will be used.
 * 
 * If both the deviceId and pingParameters define a URL then
 * the URL for the pingParameters will be treated as being
 * relative to the URL for the deviceId. More specifically,
 * the HTTP connection will be established to the URL that
 * results by combining the deviceId URL with the pingParams
 * URL.
 * 
 * @see com.tridium.ddfHttp.identify.BDdfHttpInspectDeviceId
 * @see com.tridium.ddfHttp.identify.BIDdfHttpInspectUrl
 * @see BDdfHttpInspectReadRequest
 *
 * @author    Lenard Perkins
 * @creation  01 Jan 08
 * @version   $Revision$ $Date: 02/19/2009 3:42:00 PM$
 * @since     Baja 1.0
 */
public class BDdfHttpInspectPingRequest
    extends BDdfPingRequest
{
  /*-
    class BDdfHttpInspectReadRequest
    {
    }
   -*/
/*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
/*@ $com.tridium.ddfHttp.comm.req.BDdfHttpInspectPingRequest(2950692683)1.0$ @*/
/* Generated Fri Feb 22 14:52:47 EST 2008 by Slot-o-Matic 2000 (c) Tridium, Inc. 2000 */

////////////////////////////////////////////////////////////////
// Type
////////////////////////////////////////////////////////////////
  
  public Type getType() { return TYPE; }
  public static final Type TYPE = Sys.loadType(BDdfHttpInspectPingRequest.class);

/*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/
  
  
  /**
   * For HTTP drivers, the byte array from the request should be the String bytes for the URL.
   * 
   * The URL optionally encodes any data to be sent by placing a question mark after the web address
   * and then placing data fields as a series of key=value pairs that are delimited by the ampersand
   * character.
   * 
   * By default, the devHttpDriver framework will establish an HTTP GET connection to the URL. If
   * the URL begins with "POST " (that is to say, the string POST all upper case followed by a
   * space) then the devHttpDriver will establish an HTTP POST connection to the web address portion
   * of the URL (that is to say, the text on the left side of the question mark in the URL). In this
   * scenario, the data (that is to say, the text on the right side of the question mark in the URL)
   * will be placed into the HTTP data stream in the location where form data is typically placed
   * by Web Browsers.
   * 
   * This request class uses the URL that is defined on the device's deviceId and/or pingParameters.
   * Please review the javadoc for this class's declaration for more details about how the URL is
   * formed.
   * 
   */
  public byte[] toByteArray()
  {
    BDdfIdParams pingParameters = getPingParameters();
    BDdfIdParams deviceId = getDeviceId();

    // Gets the 'URL' that might be encoded in the ping parameters 
    String pingUrl;
    if (pingParameters instanceof BIDdfHttpInspectUrl)
    {
      pingUrl = ((BIDdfHttpInspectUrl)pingParameters).getUrl();
      if (pingUrl.trim().length()==0 || pingUrl.trim().equals("-"))
        pingUrl = null;
    }
    else
      pingUrl = null;
    
    // Gets the 'URL' that might be encoded in the device id
    String deviceUrl;
    if (deviceId instanceof BIDdfHttpInspectUrl)
    {
      deviceUrl = ((BIDdfHttpInspectUrl)deviceId).getUrl();
      if (deviceUrl.trim().length()==0 || deviceUrl.trim().equals("-"))
        deviceUrl = null;
    }
    else
      deviceUrl = null;
    
    // If no URL in neither pingUrl nor deviceUrl
    if (pingUrl == null && deviceUrl == null)
      throw new BajaRuntimeException("No URL!");
    else
    {
      // Else, a URL is encoded in the pingUrl and/or the deviceUrl
      
      // If a URL is encoded in both the pingUrl and the deviceUrl
      // then this combines them both into a single URL (assuming
      // that the pingUrl is relative to the deviceUrl)
      if (deviceUrl != null && pingUrl !=null )
      {
        // Avoids placing a double set of slashes between the pingUrl
        // and deviceUrl when they are combined
        if (deviceUrl.endsWith("/"))
          deviceUrl = deviceUrl.substring(0, deviceUrl.length()-1);
        if (pingUrl.startsWith("/"))
          pingUrl = pingUrl.substring(1);
        
        // Combines the deviceUrl and the pingUrl (readUrl assumed
        // to be relative in this scenario)
        urlAddress = deviceUrl + '/' + pingUrl;
      }
      else if (deviceUrl == null)
      {
        // If no deviceUrl then assume the pingUrl is the full
        // url by itself
        urlAddress = pingUrl;
      }
      else if (pingUrl == null)
      {
        // If no pingUrl then assume the deviceUrl is the full url
        // by itself
        urlAddress = deviceUrl;
      }
      else // This shouldn't be possible based on our if statements above
        throw new IllegalStateException("Program Execution should not have made it here");
      
      // Returns the ascii bytes for the Url. This has special meaning to
      // the Ddf Http Communicator
      return urlAddress.getBytes();
    }
  }

  public BIDdfResponse processReceive(IDdfDataFrame ddfDataFrame) throws DdfResponseException
  {
    // The urlAddress should have already been assigned during the most
    // recent call to the 'toByteArray' method. This method is called by
    // the devDriver framework after it calls 'toByteArray' and receives
    // the subsequent response from the field-bus.
    
    return new BDdfHttpInspectPingResponse(ddfDataFrame, urlAddress); 
  }
  
////////////////////////////////////////////////////////////////
//Type
////////////////////////////////////////////////////////////////
  
  /**
   * This is the urlAddress whose byte array was returned from the 'toByteArray'
   * method. This variable is actually assigned in the 'toByteArray' method.
   * Therefore, developers should not assume that this value is valid __UNLESS__
   * it is known that the 'toByteArray' method has already been called. Please
   * note that this will always be the case during response processing. Afterall,
   * there would be no response without for transmitting (over HTTP) the request.
   */
  protected String urlAddress="";

}
