/*
 * Copyright 2008 Tridium, Inc. All Rights Reserved.
 */
package com.tridium.ddfHttp.comm;

import java.security.AccessController;
import java.security.PrivilegedAction;
import java.security.SecureRandom;
import java.util.Base64;
import java.util.regex.Matcher;
import java.util.regex.MatchResult;
import java.util.regex.Pattern;

import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.net.URL;
import java.security.MessageDigest;
import java.util.Random;

import javax.baja.naming.BIpHost;
import javax.baja.net.Http;
import javax.baja.net.HttpConnection;
import javax.baja.net.UrlConnection;
import javax.baja.security.BUsernameAndPassword;
import javax.baja.nre.util.TextUtil;

/**
 * An instance of this class is used to perform the HTTP logon to each unqiue URL that an instance
 * of DdfHttpHelper encounters.
 *
 * The relationship between this and DdfHttpHelper is many-to-one. One DdfHttpHelper has a reference
 * to many DdfAuthenticationHelpers. More specifically, one DdfHttpHelper has as many references to
 * DdfAuthenticationHelpers as it has encountered unique URLs that require authentication (user name /
 * password logon).
 *
 * While this is devHttpDriver's best attempt at authentication, its creators realize that some
 * web servers could implement authentication differently (hopefully just sligtly). For that reason,
 * all fields are granted protected access and all methods are split up into many
 * smaller methods, all of which are granted protected access. This should allow
 * developers to customize any pieces of the authentication that might be necessary on a
 * per-driver basis.
 *
 * To use a custom DdfAuthenticationHelper, please make a custom DdfHttpHelper
 * and override DdfHttpHelper.makeAuthenticationHelper to return an instance
 * of the driver's own customized descendant of this class.
 *
 * To plug-in the custom DdfHttpHelper (which uses the custom DdfAuthenticationHelper, override
 * the 'makeHttpHelper' method on the driver's transmitter class which extends BDdfHttpTransmitter.
 *
 * @see DdfHttpHelper
 * @see BDdfHttpTransmitter
 *
 * @author    Lenard Perkins
 * @creation  01 Jan 08
 * @version   $Revision$ $Date: 02/19/2009 3:42:00 PM$
 * @since     Baja 1.0
 */
public class DdfAuthenticationHelper
{

////////////////////////////////////////////////////////////////
// Construction
////////////////////////////////////////////////////////////////

  /**
   *
   * @param webHost encapsulates the IP address of the web server that this
   * instance will authenticate.
   *
   * @param webPort the int port of the web server that this instance will
   * authenticate.
   *
   * @param webUsernameAndPassword the BUsernameAndPassword from the baja
   * security module. This specifies the user name and password to use for
   * authentication.
   */
  public DdfAuthenticationHelper(URL javaUrl, BUsernameAndPassword webUsernameAndPassword, UrlConnection httpReplyRequestingAuth)
  {
    this.webHost = new BIpHost(javaUrl.getHost());
    this.webPort = (javaUrl.getPort() == -1) ? Http.DEFAULT_HTTP_PORT : javaUrl.getPort();
    this.webUsernameAndPassword = webUsernameAndPassword;
    this.webAuthenticateHeader = httpReplyRequestingAuth.getResponseHeader("WWW-Authenticate");
    this.webUri = javaUrl.getFile();
  }

////////////////////////////////////////////////////////////////
// API
////////////////////////////////////////////////////////////////

  /**
   * Checks if the WWW-Authenticate response header of the UrlConnection that
   * was passed to the constructor indicates that the web server requires
   * authentication based on the HTTP Digest specification.
   *
   * The HTTP digest authentication is a somewhat secure method of
   * authentication over an unsecure connection.
   *
   * @return true if using digest authentication false for anything else.
   *
   * @throws Exception
   */
  public boolean isDigestAuthentication()
    throws Exception
  {
    return getAuthenticationType(webAuthenticateHeader).equalsIgnoreCase("digest");
  }

  /**
   * Checks if the WWW-Authenticate response header of the UrlConnection that
   * was passed to the constructor indicates that the web server requires
   * authentication based on the HTTP Basic specification.
   *
   * The HTTP basic is a completely unsecure method of encryption. It simply
   * hides the user name and password from anybody that might be reviewing
   * the data over a plain network sniffer.
   *
   * @return true if using basic authentication false for anything else.
   *
   * @throws Exception
   */
  public boolean isBasicAuthentication() throws Exception
  {
    return getAuthenticationType(webAuthenticateHeader).equalsIgnoreCase("basic");
  }

  /**
   * Reviews the WWW-Authenticate response header of the UrlConnection that was passed to the
   * constructor and returns the String that identifies the type of authentication that the web
   * server requires (for example: basic or digest).
   *
   * NOTE: Currently, devHttpDriver supports Basic and Digest authentication.
   *
   * @return the String that identifies the type of authentication that the web server requires (for
   *         example: basic or digest).
   *
   * @throws Exception
   */
  public String getAuthenticationType(String webAuthenticateHeader)
    throws Exception
  {
    // Pulls out the first word from the authentication header, this is the
    // expected authentication type
    Pattern p = Pattern.compile("\\s*(\\S*)", Pattern.CASE_INSENSITIVE);
    Matcher oroMatcher = p.matcher(webAuthenticateHeader);

    // If found...
    if (oroMatcher.find())
    {
      // Prepares to extract the first word from the matched text
      MatchResult oroResult = oroMatcher.toMatchResult();

      // Extracts the first word from the authentication header
      return oroResult.group( 1 );
    }
    else // Else, there is first word in the authentication header!
      return null;

  }

  /**
   * Adds the appropriate authorization credentials to the HTTP request
   * header of the given HttpConnection.
   *
   * NOTE: Currently, devHttpDriver supports Basic and Digest authentication.
   *
   * The authentication type is chosen based on the authentication type
   * indicated by the response header in the http connection that was
   * passed to the constructor.
   *
   * @param httpRequestConn the given HttpConnection to add request
   * credentials to.
   *
   * @return true if this method really adds a request header to the given
   * HttpConnection. False if not.
   *
   * @throws Exception
   */
  public boolean addRequestAuthorization( HttpConnection httpRequestConn )
    throws Exception
  {
    // If using 'Digest' authentication
    if (isDigestAuthentication())
    {
      // Then this adds the appropriate semi-secure digest authentication header
      return addDigestAuthorization(httpRequestConn);
    }
    // If using 'Basic' authentication
    else if (isBasicAuthentication())
    {
      // Then this adds the appropriate Base64 (unsecure) basic authentication header
      return addBasicAuthorization(httpRequestConn);
    }
    // Else, some other type of authentication is used
    else
      return false; // Returns false to indicate that the required authentication mechanism is unsupported.
  }

////////////////////////////////////////////////////////////////
// Basic (Base64) Authentication
////////////////////////////////////////////////////////////////


  /**
   * Adds a Base64 (unsecure) basic authentication header to the given HttpConnection.
   *
   * Uses the user name and password that were passed to the constructor.
   *
   * @param httpRequestConn the HttpConnection to add a basic authentication header to.
   *
   * @return true if a basic authentication header is really added to the given HttpConnection
   * or false if not. At present, this will return false if the user name / password
   * that was passed to the constructor was null.
   */
  public boolean addBasicAuthorization( HttpConnection httpRequestConn )
  {
    if (webUsernameAndPassword==null)
    {
      return false;
    }
    else
    {
      // The Basic authentication mechanism combines the user name and password into one ASCII string
      String basicAuthorizationCredentials =
        webUsernameAndPassword.getUsername() + ":" +
          AccessController.doPrivileged((PrivilegedAction<String>)webUsernameAndPassword.getPassword()::getValue);
      // Performs basic, Base64 encryption of the basicAuthorizationCredentials. This is not very
      // secure. The only thing it accomplishes as far as being secure is that it hides the user
      // name and password from peering eyes on a network sniffer.
      String authorization = new String(Base64.getEncoder().encode(basicAuthorizationCredentials.getBytes()));

      httpRequestConn.setRequestHeader(BDdfHttpTransmitter.AUTHORIZATION,
                                       BDdfHttpTransmitter.BASIC + authorization);
      return true;

    }
  }

////////////////////////////////////////////////////////////////
// Digest Authentication
////////////////////////////////////////////////////////////////

  /**
   * Adds a 'digest' (semi-secure) authentication header to the given HttpConnection.
   *
   * The digest authentication scheme is that described in RFC 2617.
   *
   * @param httpRequestConn
   *          the HttpConnection to add a digest authentication header to.
   *
   * @return true if a basic authentication header is really added to the given HttpConnection or
   *         false if not. At present, this will return false if the user name / password that was
   *         passed to the constructor was null. At present, this will also return false if the Http
   *         Response Header of the HttpConnection that was passed to the constructor does not
   *         contain the necessary key/value pairs (realm, nonce, and qop) to perform the digest
   *         authentication.
   *
   */
  public boolean addDigestAuthorization( HttpConnection httpRequestConn )
    throws Exception
  {
    // If the web server did not include a "WWW-Authenticate" header then we cannot
    // determine what type of authentication is required
    if (webAuthenticateHeader == null)
    {
      return false;
    }

    // Gets the realm, nonce, and qop
    String realm = getValue("realm", webAuthenticateHeader);
    String nonce = getValue("nonce", webAuthenticateHeader);
    String qop   = getValue("qop",   webAuthenticateHeader);

    // If a null user name / password was given to this method then we cannot really
    // authenticate since a user name and password is the entire purpose of authentication
    if (nonce == null ||  webUsernameAndPassword == null)
    {
       return false;
    }

    // The term 'nc' is commonly understood to those (unlike this developer) who are
    // well educated in the theory of digest authentication. To me, it stands roughly
    // for 'num connections' or better yet, 'elapsed connection attempts'
    nc++;

    // response is H(H(A1) ":" unq(nonce) ":" nc ":" unq(cnonce) ":" unq(qop) ":" H(A2))
    //   where A1 is unq(username) ":" realm ":" passwd
    //   and   A2 is method ":" unq(uri)

    // The term 'nc' is commonly understood to those (unlike this developer) who are
    // well educated in the theory of digest authentication. To me, it is a response
    // code the encrypts the digest credentials.
    String cnonce = getCnonce();

    // Builds up the digest authentication text to go in the request header
    StringBuffer response = new StringBuffer();
    response.append("Digest username=").append(quote(webUsernameAndPassword.getUsername())).append(", ");
    response.append("realm=")          .append(quote(realm)) .append(", ");
    response.append("nonce=")          .append(quote(nonce)) .append(", ");
    response.append("uri=")            .append(quote(webUri))   .append(", ");
    response.append("nc=")             .append(nc).append(", ");
    response.append("qop=")            .append(unquote(qop)) .append(", ");
    response.append("cnonce=\"")       .append(cnonce)       .append("\", ");
    response.append("response=\"");

    response.append(

                     hexMD5
                     (
                       hexMD5(
                         unquote(webUsernameAndPassword.getUsername()) + ':' +
                           unquote(realm) + ':' +
                           AccessController.doPrivileged((PrivilegedAction<String>)webUsernameAndPassword.getPassword()::getValue)
                              ) + ':' +
                       unquote(nonce)                        + ':' +
                       nc                                    + ':' +
                       cnonce                                + ':' +
                       unquote(qop)                          + ':' +
                       hexMD5(
                                  httpRequestConn.getRequestMethod() + ":" + webUri
                              )
                      )
                    );

    response.append("\"");

    httpRequestConn.setRequestHeader("Authorization", response.toString());
    return true;
  }

  /**
   * Computes the CNONCE. The term CNONCE is universally understood by those that
   * are well educated in digest authentication theory. It is essentially an
   * encrypted version of the credentials (user name / password) to log into
   * a web server.
   *
   * @return the CNONCE
   *
   * @throws Exception
   */
  protected String getCnonce()
    throws Exception
  {
    ByteArrayOutputStream bout = new ByteArrayOutputStream();
    DataOutputStream out = new DataOutputStream(bout);

    out.writeUTF("cnonce");
    out.writeInt(nc);
    out.writeUTF(webHost.getHostname());
    out.writeInt(webPort);
    out.writeLong(System.currentTimeMillis());
    out.writeInt(clientRandom.nextInt());

    return hexMD5(new String(bout.toByteArray()));
  }

  /**
   * Encrypts the given string using the MD5 encryption algorithm. Delegates
   * to java.security.MessageDigest to perform the computation
   *
   * @param in the given string to encrypt
   *
   * @return the MD5 encrypted version of the given string
   *
   * @throws Exception
   */
  protected String hexMD5(String in)
    throws Exception
  {
    StringBuffer result = new StringBuffer();
    MessageDigest md5 = MessageDigest.getInstance("MD5");
    byte[] toHash = md5.digest(in.getBytes());
    for (int i = 0; i < toHash.length; i++)
    {
      result.append(TextUtil.toLowerCase(TextUtil.byteToHexString(toHash[i])));
    }
    return result.toString();
  }


////////////////////////////////////////////////////////////////
// Parsing WWW-Authenticate Header
////////////////////////////////////////////////////////////////

  /**
   * Extracts a property value by the given key from the givenTextToSearch. This scans
   * for the property value if it is enclosed inside quotation marks or even if not.
   *
   * @param key the name of value to extract from the given textToSearch
   *
   * @param textToSearch the text from which to extract the property value. This
   * is intended to be the WWW-Authenticate header of a UrlConnection.
   *
   * @return the value on the right side of the equals sign, after the key within
   * the given textToSearch or null if not found (any quotation marks surrounding
   * the value are not returned).
   *
   * @see #getInQuotes(String, String)
   * @see #getNotInQuotes(String, String)
   *
   * @throws Exception
   */
  protected String getValue(String key, String textToSearch)
    throws Exception
  {
    if (textToSearch != null) // Sanity check
    {
      // Scans for the value for the given key (allows for the text to
      // be in quotation marks or not)
      String quotedMatch = getInQuotes(key, textToSearch);

      // If not found inside quotation marks then let's try without quotation marks
      if (quotedMatch == null)
      {
        // Scans for the value for the given key without looking for the value to
        // be inside quotation marks
        return getNotInQuotes(key, textToSearch);
      }
      else // If found inside quotation marks
      {
        // Then this returns the text that was found inside quotation marks
        return quotedMatch;
      }
    }
    else
    {
      return null;
    }

  }

  /**
   * Extracts a property value by the given key from the givenTextToSearch. This scans
   * for the property value if it is enclosed inside quotation marks.
   *
   * @param key the name of value to extract from the given textToSearch
   *
   * @param textToSearch the text from which to extract the property value. This
   * is intended to be the WWW-Authenticate header of a UrlConnection.
   *
   * @return the value on the right side of the equals sign, within quotation marks,
   * after the key within the given textToSearch or null if not found (the surrounding
   * quotation marks are not returned).
   *
   * @throws Exception
   */
  protected String getInQuotes(String key, String textToSearch)
    throws Exception
  {
    Pattern p = Pattern.compile("\\b"+key+"\\s*=\\s*\"(.*?)\"", Pattern.CASE_INSENSITIVE);
    Matcher oroMatcher = p.matcher(textToSearch);

    if (oroMatcher.find())
    {
      MatchResult oroResult = oroMatcher.toMatchResult();
      return oroResult.group( 1 );
    }
    else
      return null;
  }

  /**
   * Extracts a property value by the given key from the given textToSearch. This scans
   * for the property value if it is _not_ enclosed inside quotation marks.
   *
   * @param key the name of value to extract from the given textToSearch
   *
   * @param textToSearch the text from which to extract the property value. This
   * is intended to be the WWW-Authenticate header of a UrlConnection.
   *
   * @return the value on the right side of the equals sign after the key
   * within the given textToSearch or null if not found.
   *
   * @throws Exception
   */
  protected String getNotInQuotes(String key, String textToSearch) throws Exception
  {
    Pattern p = Pattern.compile("\\b"+key+"\\s*=\\s*([^\\s,]*)", Pattern.CASE_INSENSITIVE);
    Matcher oroMatcher = p.matcher(textToSearch);

    if (oroMatcher.find())
    {
      MatchResult oroResult = oroMatcher.toMatchResult();
      return oroResult.group( 1 );
    }
    else
      return null;
  }

////////////////////////////////////////////////////////////////
// Quotation Mark Utilities
////////////////////////////////////////////////////////////////

  /**
   * Adds quotation marks around the given string if necessary.
   *
   * @param in the given string to add quotation marks around.
   *
   * @return the given string with surrounding quotation marks.
   */
  protected static String quote(String in)
  {
    if (in.length() == 0)
      return "\"\"";
    else if (in.charAt(0) == '"')
      return in;
    else
      return '"' + in + '"';
  }

  /**
   * Removes any quotation marks from around the given string.
   *
   * @param in the given string from which to remove surrounding quotation marks.
   *
   * @return the given string without any surrounding quotation marks that.
   */
  protected static String unquote(String in)
  {
    if (in.length() == 0) return in;

    StringBuffer result = new StringBuffer(in);
    if (result.charAt(0) == '"')
    {
      result.deleteCharAt(0);
    }
    if (result.charAt(result.length() - 1) == '"')
    {
      result.deleteCharAt(result.length() - 1);
    }
    return result.toString();
  }
////////////////////////////////////////////////////////////////
// Main Method Used For Command Line Tests
////////////////////////////////////////////////////////////////

//  private DdfAuthenticationHelper()
//  {
//
//  }
//
//  public static void main( String[] argv )
//    throws Exception
//  {
////    Perl5Compiler regExpCompiler = new Perl5Compiler();
////    PatternMatcher oroMatcher = new Perl5Matcher();
////    Pattern reRealmInQuotes = regExpCompiler.compile( "\\brealm\\s*=\\s*\"(.*?)\"");
////    Pattern reRealmWithoutQuotes = regExpCompiler.compile( "\\brealm\\s*=\\s*(\\S*?)");
//
//    String htmlText = "WWW-Authenticate: Digest realm=                            \"Admin Camera Protection\"          , nonce=\"MDAwMDAwMDAzMTo5MjI6MTM3LjAxOS4wNjAuMTQ3\", algorithm  =  MD5 ,,,, qop=\"auth\"";
////    if (oroMatcher.contains(htmlText, reRealmInQuotes))
////    {
////      MatchResult oroResult = oroMatcher.getMatch();
////      System.out.println( "In Quotes Result:"+oroResult.group( 1 ) );
////    }
////
////    if (oroMatcher.contains(htmlText, reRealmWithoutQuotes))
////    {
////      MatchResult oroResult = oroMatcher.getMatch();
////      System.out.println( "Not In Quotes Result:"+oroResult.group( 1 ) );
////    }
//    DdfAuthenticationHelper authHelper = new DdfAuthenticationHelper();
//
//    System.out.println("Authentication type:"+authHelper.getAuthenticationType(htmlText));
//    System.out.println("Realm:"+authHelper.getValue("realm", htmlText));
//    System.out.println("nonce:"+authHelper.getValue("nonce", htmlText));
//    System.out.println("algorithm:"+authHelper.getValue("algorithm", htmlText));
//    System.out.println("qop:"+authHelper.getValue("qop", htmlText));
//
//  }

////////////////////////////////////////////////////////////////
// Attributes
////////////////////////////////////////////////////////////////

  /**
   * This is used to generate random numbers for authentication.
   */
  protected static SecureRandom clientRandom = new SecureRandom();

  /**
   * This keeps track of the number of authentication attempts during the life-cycle.
   * This is a universally understood portion of the 'digest' authentication mechanism.
   */
  protected int nc = 0;

  /**
   * This encapsulates the IP address of the web server that this instance will
   * authenticate.
   */
  protected BIpHost webHost = null;

  /**
   * This is the port on the web server that this instance will authenticate.
   */
  protected int webPort = Http.DEFAULT_HTTP_PORT;

  /**
   * This is the user name and password that this instance will use to authenticate
   * to the web server.
   */
  protected BUsernameAndPassword webUsernameAndPassword = null;

  /**
   * This is the WWW-Authenticate header that is provided by the web server when
   * it asks for a log-in over digest authentication.
   */
  protected String webAuthenticateHeader = null;

  /**
   * This is the file portion of the URL to ask the web server for authentication
   * to. This is used for 'digest' authentication.
   */
  protected String webUri = null;
}
