/*
 * @copyright 2005 Tridium Inc.
 */
package com.tridium.ddf.ui.device;

import javax.baja.driver.BDeviceFolder;
import javax.baja.driver.BIDeviceFolder;
import javax.baja.driver.ui.device.DeviceModel;
import javax.baja.registry.TypeInfo;
import javax.baja.sys.BComponent;
import javax.baja.sys.BajaRuntimeException;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;
import javax.baja.nre.util.Array;
import javax.baja.workbench.mgr.MgrColumn;
import javax.baja.workbench.mgr.MgrTypeInfo;

import com.tridium.ddf.BDdfDevice;
import com.tridium.ddf.IDdfFacetConst;
import com.tridium.ddf.ui.DdfMgrModelUtil;
import com.tridium.ddf.ui.DdfUiLexicon;

/**
 * The DdfDeviceModel uses introspection to define the columns
 * in the database type of the Ddf Device Manager.
 *
 * @author lperkins
 */
public class DdfDeviceModel
  extends DeviceModel
  implements IDdfFacetConst
{
  public DdfDeviceModel(BDdfDeviceManager manager){super(manager);}

////////////////////////////////////////////////////////////////
// MgrModel
////////////////////////////////////////////////////////////////

  /**
   * Automatically determines the MgrColumns for the database list.
   *
   * Calls DdfMgrModelUtil.makeColumns
   *
   * @see DdfMgrModelUtil.makeColumns
   */
  protected MgrColumn[] makeColumns()
  {
    return DdfMgrModelUtil.makeColumns(getManager());
  }

  public void init()
  {
    // Looks at the 'current value' as a 'BIDeviceFolder'
    folder = (BIDeviceFolder)getManager().getCurrentValue();

    // Gets the device TYPE and device folder TYPE for
    // the Ddf Device Manager
    try
    {
      deviceType = folder.getDeviceType();
      folderType = folder.getDeviceFolderType();
    }
    catch(Exception e)
    {
      // this occurs when using offline
    }

    if (deviceType == null)
    {
      throw new BajaRuntimeException(
         DdfUiLexicon.LEX.getText("MustSpecifyDeviceType",
         new Object[]{
           folder.
             getType().
               getTypeSpec()}));

    }


    // The device type cannot be abstract. The device manager for devDriver is just not designed to allow this
    // since it automatically builds the discovery and database columns based on the deviceId and other props
    // of a default instance of the deviceType
    if (deviceType.isAbstract())
    {
      throw new BajaRuntimeException( DdfDeviceManagerLexicon.deviceTypeCannotBeAbstract(folder, deviceType));
    }

    super.init();
  }

  /*
   * Overrides the load method in order to subscribe deeply to the
   * communicator components on the devices that are in the database.
   *
   * This is important for devices with their own communications port
   * such as Tcp/Ip devices. It allows for properties that are under the
   * communicator to be flagged as MGR_INCLUDE. Such properties are already
   * added as a column to the database. This needs to subscribe deeply
   * under the communicator components of the devices so that the values
   * in the columns will update to reflect any changes that are ever made
   * to the communicator components.
   *
   */
  public void load(BComponent target)
  {
    super.load(target);

    // Gets all rows in the database table
    BComponent[] dbRows = getTable().getComponentModel().getRows();

    // Loops through each
    for (int i=0; i<dbRows.length; i++)    // Loops through each row in the database table
      if (dbRows[i] instanceof BDdfDevice) // Any row that represents a BDdfDevice needs to
        ((BDdfDeviceManager)getManager()). // Have its 'communicator' component registered deeply
          registerDeviceCommunicator((BDdfDevice)dbRows[i]); // For component events
  }

  /**
   * Return the concrete types of <code>network.deviceType</code>.
   */
  public MgrTypeInfo[] getNewTypes()
  {
    // DISUSSION: The available types when the user clicks 'New' on the device manager is the list of all concrete
    // types that extend the deviceType from as determined in the 'load(...)' method. However, to better support
    // multiple types of devices, if the deviceType is something that we will call a proto device then we will not
    // include it in the 'New' types list. However, the rest of the device manager, such as the columns in the
    // discovered list and database list, will still be created based on the proto deviceType. We allow the
    // developer to define whether or not a deviceType is a proto device by using the <defs> ... </defs> secions
    // in their driver's module-include.xml file. The syntax to define a device as a proto device is:
    // <defs>
    //   <def name="proto|module:type" value="true"/>
    // </defs>

    // The device type cannot be abstract. The device manager for devDriver is just not designed to allow this
    // since it automatically builds the discovery and database columns based on the deviceId and other props
    // of a default instance of the deviceType
    if (deviceType.isAbstract()) // NOTE: We already checked for this in the .init method so this is just a sanity check
    {
      throw new BajaRuntimeException( DdfDeviceManagerLexicon.deviceTypeCannotBeAbstract(folder, deviceType));
    }
    else // The deviceType must be concrete
    {
      // Checks if the developer defined the deviceType as a prototype device in the driver's module-include.xml
      String protoDef = Sys.getRegistry().getDef( "proto:"+deviceType.getTypeSpec().toString(), "false").toLowerCase();

      // If the user mapped the special <def...> per the previous line of code to the value 'true' in the driver's
      // module-include.xml file...
      if (protoDef.equals("true"))
      {
        // Gets an array that includes the device type (since it is concrete) and any concrete subclasses
        // Let's store this array in a Baja Util array wrapper to allow us easy removal on the following line
        Array<TypeInfo> bajaArrayConcreteDeviceTypes = new Array<>( Sys.getRegistry().getConcreteTypes(deviceType.getTypeInfo()) );

        // Removes the deviceType itself from the available concrete types since the deviceType is just a
        // protocol device to be used to construct the device manager only.
        bajaArrayConcreteDeviceTypes.remove(deviceType.getTypeInfo());

        // Returns an array of the appropriate type, trimmed down to exact size (the baja util array is a hybrid
        // of an array and a vector, when the previous line of code removed the deviceType, the baja util array
        // would have likely just swapped some entries around (and updated some internal notion of a 'last' index)
        // in the array but not actually shrink the array.
        return MgrTypeInfo.makeArray(bajaArrayConcreteDeviceTypes.trim());
      }
      else
      {
        // Returns an array including the deviceType (since we already know it must be concrete) and all concrete
        // descendants
        return MgrTypeInfo.makeArray(deviceType);
      }
    }
  }

  protected BIDeviceFolder folder;
  protected Type deviceType = BDdfDevice.TYPE;
  protected Type folderType = BDeviceFolder.TYPE;


}
