/*
 * @copyright 2005 Tridium Inc.
 */
package com.tridium.ddf.ui;

import javax.baja.agent.AgentFilter;
import javax.baja.agent.AgentInfo;
import javax.baja.agent.AgentList;
import javax.baja.sys.Context;
import javax.baja.sys.Sys;
import javax.baja.ui.CommandArtifact;
import javax.baja.nre.util.Array;
import javax.baja.util.Lexicon;
import javax.baja.workbench.fieldeditor.BWbFieldEditor;
import javax.baja.workbench.mgr.BAbstractManager;
import javax.baja.workbench.mgr.MgrController;
import javax.baja.workbench.mgr.MgrController.IMgrCommand;
import javax.baja.workbench.mgr.MgrController.MgrCommand;

import com.tridium.ddf.BDdfDevice;
import com.tridium.ddf.BDdfNetwork;
import com.tridium.ddf.BDdfPointDeviceExt;
import com.tridium.ddf.IDdfFacetConst;
import com.tridium.ddf.discover.BIDdfDiscoveryHost;
import com.tridium.ddf.discover.auto.BDdfAutoDiscoveryPreferences;
import com.tridium.ddf.ui.device.BDdfDeviceManager;
import com.tridium.ddf.ui.device.BIDdfDeviceMgrAgent;
import com.tridium.ddf.ui.device.DdfDeviceManagerLexicon;
import com.tridium.ddf.ui.point.BDdfPointManager;
import com.tridium.ddf.ui.point.BIDdfPointMgrAgent;

/**
 * This class contains the like functionality that is shared
 * between the ddf device controller and the ddf point
 * controller.
 *
 * @author lperkins
 *
 */
public class DdfMgrControllerUtil
  implements IDdfFacetConst
{
  /**
   * Cannot instantiate! Access static methods only.
   */
  protected DdfMgrControllerUtil(){}

////////////////////////////////////////////////////////////////
// Util
////////////////////////////////////////////////////////////////

  /**
   * This call-back is made by the DdfDeviceController and
   * DdfPointController when the user clicks the "Discover"
   * button.
   */
  public static CommandArtifact doDiscover(BAbstractManager mgr, Context cx)
    throws Exception
  {
    BIDdfDiscoveryHost discoveryHost = DdfMgrUtil.findDiscoveryHost(mgr);

    return doDiscover(mgr, discoveryHost);
  }

  /**
   * Kicks off a ddf Auto Discovery Job
   *
   * @param discoveryHost the BIDdfDiscoveryHost that is in the database at, or closest above the object
   * on which the manager is defined
   *
   * @throws Exception
   */
  protected static CommandArtifact doDiscover(BAbstractManager mgr, BIDdfDiscoveryHost discoveryHost)
    throws Exception
  {
    // Splits the manager into learn mode (if its not already split)
    mgr.getController().learnMode.setSelected(true);

    // Gets the discovery preferences from the network or point-device-extension
    BDdfAutoDiscoveryPreferences discoveryJobPrefsDefault =
      (BDdfAutoDiscoveryPreferences)discoveryHost.getDiscoveryPreferences().newCopy();

    // Verifies that the driver developer actually desires to have a discovery job kicked off in the background
    // If not, then this means that the driver developer does not wish to submit a discovery
    // Job. Instead, we'll just let it suffice that we have gone into learnMode. The driver
    // Developer could have configured a discoveryFolder with frozen discovery objects. That
    // Would certainly be a valid use-case for the scenario where the driver developer does
    // Not require a discovery job.
    if (discoveryJobPrefsDefault.getNeedsJob())
    {
      BDdfAutoDiscoveryPreferences discoveryJobPrefs =
        discoveryJobPrefsDefault.getDoNotAskAgain()?
        discoveryJobPrefsDefault: // <-- Does this if 'doNotAskAgain' is True.
        (BDdfAutoDiscoveryPreferences)BWbFieldEditor.dialog( // <-- Does this if 'doNotAskAgain' is False
              mgr,DdfDeviceManagerLexicon.discoveryParameters,discoveryJobPrefsDefault);// Prompts the user for discovery preferences

      if (discoveryJobPrefs!=null)
        // Submits a discovery job, gives it the discovery preferences, and registers the job with the DdfMgrLearn
        mgr.getLearn().setJob(
            discoveryHost.submitDiscoveryJob(
                discoveryJobPrefs));
    }

    return null;
  }

  /**
   * Makes the commands that the ddf adds to the device manager.
   *
   * @param superCommands the standard commands for the device manager.
   *
   * @param mgr the device manager
   *
   *
   * @return a new array consisting of the superCommands plus DdfDeviceMgrAgentCommands for each 'BIDdfDeviceMgrAgent'
   * agent that is declared on the network.
   */
  public static final IMgrCommand[] makeCommands(IMgrCommand[] superCommands, BDdfDeviceManager mgr)
  {
    BDdfNetwork network = DdfMgrUtil.findDdfNetwork(mgr);

    Array<IMgrCommand> deviceMgrCmdAgents = new Array<>(IMgrCommand.class);

    AgentList ddfDeviceMgrAgentList = network.getAgents().filter(AgentFilter.is(BIDdfDeviceMgrAgent.TYPE));

    AgentInfo[] ddfDeviceMgrAgents = ddfDeviceMgrAgentList.list();

    for (int i=0; i<ddfDeviceMgrAgents.length; i++)
    {
      BIDdfDeviceMgrAgent ddfDeviceMgrAgent = (BIDdfDeviceMgrAgent)ddfDeviceMgrAgents[i].getInstance();
      deviceMgrCmdAgents.add( DdfDeviceMgrAgentCommand.make(mgr,ddfDeviceMgrAgent));
    }

    // Returns an array consisting of the superCommands and the DdfDeviceMgrAgentCommand's for each BIDdfDeviceMgrAgent
    return MgrController.append(superCommands, deviceMgrCmdAgents.trim());
  }

  /**
   * Makes the commands that the ddf would like to add to the point manager.
   *
   * @param superCommands the standard commands for the point manager.
   *
   * @param mgr the point manager
   *
   *
   * @return a new array consisting of the superCommands plus DdfPointMgrAgentCommands for each 'BIDdfPointMgrAgent'
   * agent that is declared on the point-device-ext or on the device.
   */
  public static final IMgrCommand[] makeCommands(IMgrCommand[] superCommands, BDdfPointManager mgr)
  {
    Array<IMgrCommand> pointMgrCmdAgents = new Array<>(IMgrCommand.class);

    // Finds any agents that are on the device
    BDdfDevice device = DdfMgrUtil.findDdfDevice(mgr);

    AgentList ddfPointMgrAgentListOnDevice = device.getAgents().filter(AgentFilter.is(BIDdfPointMgrAgent.TYPE));

    AgentInfo[] ddfPointMgrAgentsOnDevice = ddfPointMgrAgentListOnDevice.list();

    for (int i=0; i<ddfPointMgrAgentsOnDevice.length; i++)
    {
      BIDdfPointMgrAgent ddfPointMgrAgent = (BIDdfPointMgrAgent)ddfPointMgrAgentsOnDevice[i].getInstance();
      pointMgrCmdAgents.add( DdfPointMgrAgentCommand.make(mgr,ddfPointMgrAgent));
    }

    // Finds any agents that are on the point-device-ext
    BDdfPointDeviceExt ptDevExt = DdfMgrUtil.findDdfPointDeviceExt(mgr);

    AgentList ddfPointMgrAgentListOnPtDevExt = ptDevExt.getAgents().filter(AgentFilter.is(BIDdfPointMgrAgent.TYPE));

    AgentInfo[] ddfPointMgrAgentsOnPtDevExt = ddfPointMgrAgentListOnPtDevExt.list();

    for (int i=0; i<ddfPointMgrAgentsOnPtDevExt.length; i++)
    {
      BIDdfPointMgrAgent ddfPointMgrAgent = (BIDdfPointMgrAgent)ddfPointMgrAgentsOnPtDevExt[i].getInstance();
      pointMgrCmdAgents.add( DdfPointMgrAgentCommand.make(mgr,ddfPointMgrAgent));
    }

    // Returns an array consisting of the superCommands and the DdfPointMgrAgentCommand's for each BIDdfPointMgrAgent
    return MgrController.append(superCommands, pointMgrCmdAgents.trim());
  }

  public static void updateCommands(BAbstractManager mgr)
  {
    IMgrCommand[] mgrCommands = mgr.getController().getCommands();

    // Enables or disables all DdfDeviceMgrAgentCommands or DdfPointMgrAgentCommands appropriately
    for (int i=0; i<mgrCommands.length; i++)
    {
      if (mgrCommands[i] instanceof DdfDeviceMgrAgentCommand)
      {
        DdfDeviceMgrAgentCommand ddfDeviceMgrCommand = (DdfDeviceMgrAgentCommand)mgrCommands[i];
        ddfDeviceMgrCommand.agent.update((BDdfDeviceManager)mgr, ddfDeviceMgrCommand);// Casting the mgr should be safe because the ddf only places device agents onto the device manager
      }
      else if (mgrCommands[i] instanceof DdfPointMgrAgentCommand)
      {
        DdfPointMgrAgentCommand ddfPointMgrCommand = (DdfPointMgrAgentCommand)mgrCommands[i];
        ddfPointMgrCommand.agent.update((BDdfPointManager)mgr, ddfPointMgrCommand);// Casting the mgr should be safe because the ddf only places point agents onto the point manager

      }
    }
  }

////////////////////////////////////////////////////////////////
// DdfDeviceMgrAgentCommand
////////////////////////////////////////////////////////////////

  public static class DdfDeviceMgrAgentCommand
      extends MgrCommand
  {
    BIDdfDeviceMgrAgent agent; // The agent that defines the doInvoke functionality

    public static final DdfDeviceMgrAgentCommand make(BAbstractManager manager, BIDdfDeviceMgrAgent ddfAgent)
    {
      // Gets the string that the agent defines as the "Ui Name"
      String uiName = ddfAgent.getUiName();

      // If the agent does not specify one, then let's use a question mark or two as the button label. The
      // Developer should be able to notice this early in testing his or her driver.
      if (uiName == null)
        return new DdfDeviceMgrAgentCommand(manager, ddfAgent, "??");
      else
      {
        // Gets the developer's lexicon
        Lexicon lex = Sys.loadModule(ddfAgent.getType().getTypeSpec().getModuleName()).getLexicon();

        // If the developer provides no lexicon, then the best we can do is use the exact uiName
        // From the ddfAgent
        if (lex == null)
          return new DdfDeviceMgrAgentCommand(manager, ddfAgent, uiName);
        else
        {
          // Checks if the lexicon specifies a lexicon base (which means the lexicon defines a .label and possibly a
          // .toolbar image, etc).
          String lexLabel = lex.get(uiName + ".label");

          // If no lexicon key with a ".label" after the uiName, then let's use either the lexicon's text
          // For the uiName itself or the uiName itself if all else fails [lex.getText(...) accomplishes this]
          if (lexLabel == null)
            return new DdfDeviceMgrAgentCommand(manager, ddfAgent, lex.getText(uiName));

          // Else, there is indeed a key with a ".label" so we use a different form of the DdfDeviceMgrAgentCommand's
          // Constructor that asks its core superclass to use the .label, .toolbar, etc. from the particular lexicon
          else
            return new DdfDeviceMgrAgentCommand(manager, lex, ddfAgent);
        }
      }
    }

    DdfDeviceMgrAgentCommand(BAbstractManager manager, BIDdfDeviceMgrAgent ddfAgent, String label)
    {
      super(manager, label);
      this.agent = ddfAgent;
      setFlags(ddfAgent.getFlags());
    }

    DdfDeviceMgrAgentCommand(BAbstractManager manager, Lexicon lex, BIDdfDeviceMgrAgent ddfAgent)
    {
      super(manager, lex, ddfAgent.getUiName());
      this.agent = ddfAgent;
      setFlags(agent.getFlags());
    }

    /**
     * This method is called when the corresponding user-interface item on the manager is clicked.
     */
    public CommandArtifact doInvoke() throws Exception
    {
      // This MgrCommand's owner will be the device manager
      BDdfDeviceManager ddfDeviceMgr = (BDdfDeviceManager) getOwner();

      // Calls the doInvoke method on the instance of the BIDdfDeviceMgrAgent
      CommandArtifact retVal = agent.doInvoke(ddfDeviceMgr, DdfMgrUtil.findDdfNetwork(ddfDeviceMgr));

      // Updates the MgrCommands to reflect any changes that could have happened
      // in the 'doInvoke' callback
      ddfDeviceMgr.getController().updateCommands();

      return retVal;

    }

    /**
     * @return the agent, this will be an instance of the developer's agent class.
     */
    public BIDdfDeviceMgrAgent getAgent()
    {
      return agent;
    }
  }

////////////////////////////////////////////////////////////////
// DdfPointMgrAgentCommand
////////////////////////////////////////////////////////////////

  public static class DdfPointMgrAgentCommand
      extends MgrCommand
  {
    BIDdfPointMgrAgent agent; // The agent that defines the doInvoke functionality

    public static final DdfPointMgrAgentCommand make(BAbstractManager manager, BIDdfPointMgrAgent ddfAgent)
    {
      // Gets the string that the agent defines as the "Ui Name"
      String uiName = ddfAgent.getUiName();

      // If the agent does not specify one, then let's use a question mark or two as the button label. The
      // Developer should be able to notice this early in testing his or her driver.
      if (uiName == null)
        return new DdfPointMgrAgentCommand(manager, ddfAgent, "??");
      else
      {
        // Gets the developer's lexicon
        Lexicon lex = Sys.loadModule(ddfAgent.getType().getTypeSpec().getModuleName()).getLexicon();

        // If the developer provides no lexicon, then the best we can do is use the exact uiName
        // From the ddfAgent
        if (lex == null)
          return new DdfPointMgrAgentCommand(manager, ddfAgent, uiName);
        else
        {
          // Checks if the lexicon specifies a lexicon base (which means the lexicon defines a .label and possibly a
          // .toolbar image, etc).
          String lexLabel = lex.get(uiName + ".label");

          // If no lexicon key with a ".label" after the uiName, then let's use either the lexicon's text
          // For the uiName itself or the uiName itself if all else fails [lex.getText(...) accomplishes this]
          if (lexLabel == null)
            return new DdfPointMgrAgentCommand(manager, ddfAgent, lex.getText(uiName));

          // Else, there is indeed a key with a ".label" so we use a different form of the DdfDeviceMgrAgentCommand's
          // Constructor that asks its core superclass to use the .label, .toolbar, etc. from the particular lexicon
          else
            return new DdfPointMgrAgentCommand(manager, lex, ddfAgent);
        }
      }
    }

    DdfPointMgrAgentCommand(BAbstractManager manager, BIDdfPointMgrAgent ddfAgent, String label)
    {
      super(manager, label);
      this.agent = ddfAgent;
      setFlags(ddfAgent.getFlags());
    }

    DdfPointMgrAgentCommand(BAbstractManager manager, Lexicon lex, BIDdfPointMgrAgent ddfAgent)
    {
      super(manager, lex, ddfAgent.getUiName());
      this.agent = ddfAgent;
      setFlags(ddfAgent.getFlags());
    }

    /**
     * This method is called when the corresponding user-interface item on the manager is clicked.
     */
    public CommandArtifact doInvoke() throws Exception
    {
      // This MgrCommand's owner will be the point manager
      BDdfPointManager ddfPointMgr = (BDdfPointManager) getOwner();

      // Calls the doInvoke method on the instance of the BIDdfPointMgrAgent
      CommandArtifact retVal = agent.doInvoke(ddfPointMgr, DdfMgrUtil.findDdfNetwork(ddfPointMgr), DdfMgrUtil.findDdfDevice(ddfPointMgr), DdfMgrUtil.findDdfPointDeviceExt(ddfPointMgr));

      // Updates the MgrCommands to reflect any changes that could have happened
      // in the 'doInvoke' callback
      ddfPointMgr.getController().updateCommands();

      return retVal;
    }
  }

}
