/*
 * @copyright 2005 Tridium Inc.
 */
package com.tridium.ddf.ui;

import javax.baja.control.BControlPoint;
import javax.baja.control.BStringPoint;
import javax.baja.sys.BComplex;
import javax.baja.sys.BSimple;
import javax.baja.sys.BStruct;
import javax.baja.sys.BValue;
import javax.baja.sys.Context;
import javax.baja.sys.Flags;
import javax.baja.sys.Property;
import javax.baja.nre.util.Array;
import javax.baja.util.Lexicon;
import javax.baja.workbench.BWbEditor;
import javax.baja.workbench.fieldeditor.BWbFieldEditor;
import javax.baja.workbench.mgr.MgrColumn;
import javax.baja.workbench.mgr.MgrEditRow;

import com.tridium.ddf.BDdfDevice;
import com.tridium.ddf.DdfFacets;
import com.tridium.ddf.IDdfFacetConst;
import com.tridium.ddf.discover.BIDdfDiscoveryGroup;
import com.tridium.ddf.point.BDdfProxyExt;
import com.tridium.workbench.propsheet.BPropertySheetFE;

/**
 * This class contains utility methods to help the Ddf Device Manager
 * and the Ddf Point Manager automatically generate the MgrColumn arrays
 * that are needed for the database and discovery tables.
 *
 * The method 'getColumnsFor' reviews a prototype database or discovery
 * object and returns an array of MgrColumns objects for use in the
 * database or discovery table's table-model.
 *
 *
 * @author lperkins
 */
public class DdfMgrColumnUtil
  implements IDdfFacetConst
{

  protected DdfMgrColumnUtil(){} // Not instantiatable!

////////////////////////////////////////////////////////////////
// DdfMgrColumnUtil - Public API
////////////////////////////////////////////////////////////////
  /**
   * Constructs a set of MgrColumn flags for a particular property.
   *
   * @param owner is the BComplex that owns the particular property
   * @param prop is the property for which to get MgrColumn flags
   */
  public static int getMgrColumnFlags(BComplex owner, Property prop)
  {
    int mgrColumnFlags = 0;

    // MgrColumn.EDITABLE is a misnommer! It really means that the property
    // will be displayed on the MgrEdit dialog. Whether or not the user has
    // to take special action to reveal it, and whether or not the user can
    // actually edit it from the MgrEdit dialog is governed by the UNSEEN
    // and READONLY flags of MgrColumn.

    if (DdfFacets.isMgrInclude(owner,prop))
    {
      if (!DdfFacets.isMgrNotInDialog(owner,prop))
      {
        // As long as the property is not faceted as MGR_NOT_IN_DIALOG.
        mgrColumnFlags |= MgrColumn.EDITABLE; // Then we set this flag so that the property is included in the MgrEdit dialog
      }
      if (DdfFacets.isMgrOptionalInTable(owner,prop))
      {
        // If the property is flagged as optional in the table
        mgrColumnFlags |= MgrColumn.UNSEEN; // Then MgrColumn.UNSEEN will take it out of the main table but still allow the end user to put it in the table if they wish.
      }
    }
    else // I think I have sufficiently designed this class so that this 'else' scenario should not happen.
    {
      mgrColumnFlags |= MgrColumn.UNSEEN;
    }

    // If the property is flagged or facetted as readonly, then it will be grayed out if displayed
    // In the MgrEdit dialog
    if (Flags.isReadonly(owner,prop) || DdfFacets.isMgrReadOnly(owner,prop))
    {
      mgrColumnFlags |= MgrColumn.READONLY;
    }

    return mgrColumnFlags;
  }

  /**
   * Gets the MgrColumn[] array for the given sample instance of a database or discovery
   * object.
   *
   * @param databaseOrDiscoveryPrototypeObject
   *
   *
   * @return the MgrColumn[] array for the given sample instance of a database or discovery
   * object.
   */
  public static MgrColumn[] getColumnsFor(BComplex databaseOrDiscoveryPrototypeObject)
  {
    // If given a proxy then we need the proxy's parent point instead
    if (databaseOrDiscoveryPrototypeObject instanceof BDdfProxyExt) // This is necessary for the point manager's
    {                                                                // Database model because the point manager returns the proxy to us as
      BControlPoint parentPt = ((BDdfProxyExt)databaseOrDiscoveryPrototypeObject).getParentPoint(); // We need the true database component, which is the parent point

      if (parentPt==null)                                                 // However, sometimes the database model passes us an unmounted point
      {                                                                   // If that happens, we'll instantiate a default parent point
        parentPt = new BStringPoint();                                    // And place a copy of the given proxy as proxyExt
        parentPt.setProxyExt((BDdfProxyExt)databaseOrDiscoveryPrototypeObject.newCopy() );
      }

      databaseOrDiscoveryPrototypeObject = parentPt;
    }

    // Uses an Array object to temporarily store all of the MgrColumns
    Array<MgrColumn> retVals = new Array<>(MgrColumn.class);

    // Gets the MgrColumns and puts them in the ARray
    retVals.addAll(                              // Adds MgrColumn.PropPath objects's for
        DdfMgrColumnUtil.getColumnsRecursive(   // All BSimple and 'summary' BComplex props that are recursively
            databaseOrDiscoveryPrototypeObject));  // Under the sample databaseOrDiscoveryPrototypeObject.
    return retVals.trim();
  }

////////////////////////////////////////////////////////////////
// Util
////////////////////////////////////////////////////////////////

  /**
   * Creates an array of DdfMgrProp[] for all BSimple, MGR_INCLUDE properties under the given BComplex.
   */
  protected static MgrColumn[] getDirectColumnsFor(BComplex complex)
  {
    Array<MgrColumn.Prop> mgrColumnProps = new Array<>(MgrColumn.Prop.class);   // This object gathers the entries for the return array

    Property[] complexProps = complex.getPropertiesArray();   // Gets all of the child properties

    for (int i=0; i<complexProps.length; i++)                 // Loops through all child properties
    {
      BValue defaultValue = complexProps[i].getDefaultValue();
      if (
          (
              defaultValue.getType().is(BSimple.TYPE) ||   // Looks for BSimple properties
              (defaultValue.getType().is(BStruct.TYPE) &&  // Or BStructs with a custom field-editor
               (!BWbFieldEditor.makeFor(defaultValue)
                .getType().is(BPropertySheetFE.TYPE))
              )
          ) &&
          DdfFacets.isMgrInclude(complex,complexProps[i])) // With MGR_INCLUDE in the property facets
      {
        mgrColumnProps.add(                                 // Then we include it in the Ddf Device Manager
            new DdfMgrProp(                                 //   By adding a new DdfMgrProp object
                complexProps[i],                            //   For the BSimple, MGR_INCLUDE property
                getMgrColumnFlags(                          //   And for which we automatically determine
                    complex,                                //    MgrColumn flags Based on the property's flags
                    complexProps[i])));
      }
    }
    return mgrColumnProps.trim();
  }

  /**
   * This method returns an array of MgrColumn.PropPath objects for each BSimple that
   * has the INCLUDE facets. Recurses under under BComplexes that have the MGR_INCLUDE
   * facet.
   *
   * @param complex
   *
   * @return see the method's description for details.
   */
  protected static MgrColumn[] getColumnsRecursive(BComplex complex)
  {
    Property[] complexProps = complex.getFrozenPropertiesArray();

    Array<MgrColumn> mgrColumnProps = new Array<>(MgrColumn.class);

    // FIRST: Adds DdfMgrProp columns for all BSimples that have the MGR_INCLUDE facet and are directly under the given complex
    mgrColumnProps.addAll( getDirectColumnsFor(complex) );

    // SECOND: Adds DdfMgrPropPath columns for properties under child MGR_INCLUDE, complex properties
    for (int i=0; i<complexProps.length; i++)
    {
      // Checks if we should recurse under the child property (yes if any of the following:)
      if (
           // If a control point was passed in here and the child property is the proxy ext
           // This special case relieves the user of having to flag their proxy extension
           // As MGR_INCLUDE. Granted, they'll still have to flag one or more BSimples there-under.
           ( complex instanceof BControlPoint &&
             complexProps[i].equals(BControlPoint.proxyExt)
           )

           ||//Or...
           (
             // If a ddf device was passed here and the child property is the deviceId or pingParameters
             // This special case relieves the user of having to flag their deviceId or pingParameters
             // As MGR_INCLUDE. Granted, they'll still have to flag one or more BSimples there-under.
             complex instanceof BDdfDevice &&
             (  complexProps[i].equals(BDdfDevice.deviceId) ||
                complexProps[i].equals(BDdfDevice.pingParameters)
             )
           )

           ||// Or...
           (
             // If the property is a BComplex and it is flagged as MGR_INCLUDE and the developer
             // has not defined a custom field editor
            complexProps[i].getDefaultValue().getType().is(BComplex.TYPE) &&
            DdfFacets.isMgrInclude(complex,complexProps[i]) &&
            (BWbFieldEditor.makeFor(complex.get(complexProps[i]))
                .getType().is(BPropertySheetFE.TYPE))

           )
         )
      {
              mgrColumnProps.addAll(                           // Then this recursively determines
                  getRecursive(                                // The prop-path for all BSimples anywhere
                      (BComplex)complex.get(complexProps[i]),  // Recursively underneath
                      new Property[]{complexProps[i]}));
      }
    }
    return mgrColumnProps.trim();
  }

  /**
   * Returns an MgrColumn.PropPath array identifying each BSimple under the given BComplex that is facet'ted as MGR_INCLUDE.
   *
   * If the given complex has child properties that are also complex and those child properties are
   * flagged as MGR_INCLUDE then this method will recurse.
   *
   * @param complex - the BComplex to get Simples from underneath.
   *
   * @param propPathSoFar - The prop path so far as an Array
   *
   *
   * @return the MgrColumn.PropPath[] array identifying all BSimple's (with the MGR_INDLUDE facet) recursively under the given complexValue
   */
  protected static MgrColumn.PropPath[] getRecursive(BComplex complex, final Property[] propPathSoFar)
  {
    Array<MgrColumn.PropPath> mgrColumnProps = new Array<>(MgrColumn.PropPath.class); // Gathers the entries for this method's return array

    Property[] childProps = complex.getFrozenPropertiesArray(); // Gets all properties under the given complex

    for (int i=0; i<childProps.length; i++)                     // Loops through all properties under the given complex
    {
      Array<Property> fullPath = new Array<>(propPathSoFar);                // Duplicates the propPathSoFar for safety to the siblings
      fullPath.add(childProps[i]);                              // Adds to the fullPath

      // If the property is a BSimple and has the MGR_INCLUDE facet
      // Then we add an DdfMgrPropPath object to the mgrColumnProps array
      if (
          (
            childProps[i].getDefaultValue().getType().is(BSimple.TYPE)
            ||
            (childProps[i].getDefaultValue().getType().is(BStruct.TYPE) &&  // Or BStructs with a custom field-editor
                (!BWbFieldEditor.makeFor(childProps[i].getDefaultValue())
                 .getType().is(BPropertySheetFE.TYPE))
               )
             )
             &&
          DdfFacets.isMgrInclude(complex,childProps[i]) )      // If the property has the MGR_INCLUDE facet
      {
            mgrColumnProps.add(                                 //   Adds an MgrColumn.PropPath to the mgrColumnProps array
                new DdfMgrPropPath(
                  fullPath.trim(),
                    getMgrColumnFlags(complex,childProps[i]))); //   Determines the MgrColumn flags based on the BSimple's flags
      }

      // Else, if the property is a BComplex and it has the MGR_INCLUDE facet
      // Then we recurse.
      else if (childProps[i].getDefaultValue().getType().is(BComplex.TYPE) &&
               DdfFacets.isMgrInclude(complex,childProps[i]))
      {
        // Then we recurse under the BComplex property
        mgrColumnProps.addAll(              // Adds an MgrColumn.PropPath for all BSimples recursively underneath
            getRecursive(                   // Calls this method again
                (BComplex)complex.get(childProps[i]), // Passes in the BComplex, MGR_INCLUDE, property
                     fullPath.trim()));          // Also passes in the fullPath so that we can continue building the
                                                          // Prop-path ultimately for the BSimples at the end of the tree.
      }
    }
    return mgrColumnProps.trim();
  }

  /**
   * Converts the given array of properties into a string containing
   * the names of each property separated by period characters.
   *
   * The getMgrColumnName method calls this method to construct
   * a lexicon key. It will look into the driver's lexicon to see
   * if the driver developer customizes the property name.
   */
  static String toDotNotation(Property[] propPath)
  {
    String dotNotation = "";

    for (int i=0; i<propPath.length; i++)
    {
      if (dotNotation.length()>0)
        dotNotation += ".";

      dotNotation+=propPath[i].getName();
    }

    // Removes "proxyExt." -- helps for the point manager where most DdfMgrColumn's are prop paths under
    // A control point staring with the control point's proxyExt property
    if (dotNotation.startsWith("proxyExt."))
      dotNotation = dotNotation.substring(9);

    return dotNotation;
  }

  /**
   * Gets the name of the mgr column for the given prop path.
   */
  static String getMgrColumnName(Property[] props)
  {
    String mgrColumnName = props[props.length-1].getDefaultDisplayName(null);
    String propPathInDotNotation = toDotNotation(props);

    // Issue 12158 - Fix point and device manager lexicon for column names
    for (int i = 0; i < props.length; i++)
    {
      // This uses the lexicon value for the dot notation for the property
      // closest to the end of the given array of props.
      Lexicon propLex = props[i].getDeclaringType().getModule().getLexicon();

      if (propLex!=null)
      {
        mgrColumnName = propLex.get(propPathInDotNotation, mgrColumnName);
      }

    }


    return mgrColumnName;
  }


////////////////////////////////////////////////////////////////
// DdfMgrProp
////////////////////////////////////////////////////////////////

  /**
   * MgrColumn.PropPath does not provide access to the propPath.
   *
   * This extends to allow for that.
   *
   * @author lperkins
   *
   */
  public static class DdfMgrPropPath
    extends MgrColumn.PropPath
  {
    public final Property[] props;

    public DdfMgrPropPath(Property props[], int flags)
    {
      super(getMgrColumnName(props),props,flags);
      this.props=props;
    }
    public Property[] getPropPath()
    {
      return props;
    }
    public BWbEditor toEditor(MgrEditRow[] rows, int colIndex, BWbEditor currentEditor)
    {
      return super.toEditor(rows, colIndex, null);
    }

    public Object get(Object row)
    {
      // Issue 11725 - 'Discovered' list fails to paint if outer-most level
      // consists of exactly one discovery group.
      if (row instanceof BIDdfDiscoveryGroup)
      {
        // NOTE: For groups, the fw only calls this method to paint 1st col
        return row.toString();
      }
      else
      {
        return super.get(row);
      }
    }
    public String toDisplayString(Object row, Object value, Context cx)
    {
      // Issue 11725 - 'Discovered' list paints incorrectly if exactly one
      // discovery item and the item is a discovery group.
      if (row instanceof BIDdfDiscoveryGroup)
      {
        // NOTE: For groups, the fw only calls this method to paint 1st col
        return row.toString();
      }
      else
      {
        return super.toDisplayString(row, value, cx);
      }
    }



  } // End of static inner class DdfMgrPropPath

////////////////////////////////////////////////////////////////
// DdfMgrProp
////////////////////////////////////////////////////////////////

  /**
   * MgrColumn.Prop does not provide access to the 'prop'.
   *
   * This extends to allow for that.
   *
   * @author lperkins
   *
   */
  public static class DdfMgrProp
    extends MgrColumn.Prop
  {
    public final Property prop;

    public DdfMgrProp(Property prop, int flags)
    {
      super(prop,flags);
      this.prop=prop;
    }

    public Property getProp()
    {
      return prop;
    }

    public BWbEditor toEditor(MgrEditRow[] rows, int colIndex, BWbEditor currentEditor)
    {
      return super.toEditor(rows, colIndex, null);
    }

  } // End of static inner class DdfMgrProp

} // End of class DdfMgrColumnUtil
