/*
 * @copyright 2005 Tridium Inc.
 */
package com.tridium.ddf.poll;

import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Vector;

import javax.baja.driver.util.BIPollable;
import javax.baja.driver.util.BPollFrequency;
import javax.baja.status.BStatusValue;
import javax.baja.sys.BStruct;
import javax.baja.sys.Context;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;
import javax.baja.nre.util.IntHashMap;

import com.tridium.ddf.IDdfFacetConst;
import com.tridium.ddf.comm.BIDdfCommunicator;
import com.tridium.ddf.comm.req.BIDdfReadRequest;

/**
 * The BDdfPollGroup objects poll on behalf of a group of related BIDdfPollable objects that can
 * be retrieved in the response to a single, ddf poll request.
 *
 * This class provides reasonable default behavior, therefore, extending it is not necessary and discouraged.
 *
 * Each BIDdfPollable should make its getDdfReadRequestType return the corresponding Niagara AX type
 * of the class that implements BIDdfReadRequest and is designed to request the data for the particular
 * BIDdfPollable.  All BIDdfPollable values that share the same BIDdfReadableRequest instance should return
 * the same ddf poll group code Object from their getDdfPollGroupCode method.
 *
 * By default, BDdfProxyExt provides sufficient default behavior and overrides these methods
 * accordingly. The easiest way to take advantage of this class is simply by extending BProxyExt
 * and properly defining the readParameters and pointId properties. All proxies under a device
 * with the equivalent readParameters structure will be assigned to the same, single, BDdfPollGroup.
 */
public class BDdfPollGroup
  extends BStruct
  implements BIDdfPollable, IDdfFacetConst, BIPollable
{
  /*-
  class BDdfPollGroup
  {
  }
  -*/
/*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
/*@ $com.tridium.ddf.poll.BDdfPollGroup(4019403574)1.0$ @*/
/* Generated Thu Oct 25 11:30:22 EDT 2007 by Slot-o-Matic 2000 (c) Tridium, Inc. 2000 */

////////////////////////////////////////////////////////////////
// Type
////////////////////////////////////////////////////////////////

  public Type getType() { return TYPE; }
  public static final Type TYPE = Sys.loadType(BDdfPollGroup.class);

/*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

  /**
   * Niagara AX generally requires an empty constructor.
   */
  public BDdfPollGroup()
  {
    this.code=null;
    mySubscribedPollables = new Vector<>(8);
    myDdfPollables = new Vector<>(8);
    subscribed = false;
  }

////////////////////////////////////////////////////////////////
// BDdfPollGroup
////////////////////////////////////////////////////////////////

  public void readFail(String reason)
  {
    // Does nothing! The DdfRequestUtil already has this part taken care of. That is, it already calls readOk & readFail on all of the readable source objects of this BDdfPollGroup
  }

  public void readOk(BStatusValue out)
  {
    // Does nothing! The DdfRequestUtil already has this part taken care of. That is, it already calls readOk & readFail on all of the readable source objects of this BDdfPollGroup
  }

  /**
   * The descendant class may call this method to get a list of the
   * ddf pollables related to this poll group.
   */
  protected final BIDdfPollable[] getPollables()
  {
    BIDdfPollable[] ddfPollables = new BIDdfPollable[myDdfPollables.size()];
    myDdfPollables.copyInto(ddfPollables);

    return ddfPollables;
  }

  /**
   * The descendant class may call this method to get a list of the
   * ddf pollables for this poll group that are presently subscribed for
   * polling.
   */
  protected final BIDdfPollable[] getSubscribedPollables()
  {
    BIDdfPollable[] ddfPollables = new BIDdfPollable[mySubscribedPollables.size()];
    mySubscribedPollables.copyInto(ddfPollables);

    return ddfPollables;
  }

  /**
   * Registers a ddf pollbable to this group so that it can subscribe and
   * unsubscribe from the group as necessary. The poll group will be registered
   * to the poll scheduler whenever at least one of its registered BIDdfPollables
   * is subscribed.
   *
   * This method is called from public static "register" method which all
   * BIDdfPollables should call from their Niagara-AX "started" call-back.
   */
  protected void registerPollable(BIDdfPollable ddfPollable)
  {
    if(!myDdfPollables.contains(ddfPollable))
      myDdfPollables.add(ddfPollable);
  }

  /**
   * Completely removes a ddf pollable value from a group.
   *
   * If the given ddfPollable was the last ddfPollable registered to this group then
   * the group itself will unsubsribe from the poll scheduler
   *
   * This method is called by the public static unregister method, which
   * BIDdfPollables should call from their 'stopped' method.
   */
  protected void unregisterPollable(BIDdfPollable ddfPollable)
  {
    try
    {
      readUnsubscribed(ddfPollable);  //remove from subscribed list
    }
    catch(Exception e)
    {

    }

    myDdfPollables.remove(ddfPollable);  //remove from registered list
  }

   /**
   * This call-back is made when the point enters a subscribed
   * state based on the current status and tuning.  The driver
   * should register for changes or begin polling.  Any IO should
   * be done asynchronously on another thread - never block the
   * calling thread.  The result of reads should be to call the
   * readOk() or readFail() method.
   */
  protected void readSubscribed(BIDdfPollable ddfPollable)
    throws Exception
  {
    if(!mySubscribedPollables.contains(ddfPollable))
      mySubscribedPollables.add(ddfPollable);

    if(!subscribed)
    {
      ddfPollable.getDdfCommunicator().getDdfPollScheduler().subscribe((BIDdfPollable)this);
      subscribed = true;
    }
  }

  /**
   * This call-back is made when the point exits the subscribed
   * state based on the current status and tuning.  The driver
   * should unregister for changes of cease polling.  Any IO should
   * be done asynchronously on another thread - never block the
   * calling thread.
   */
  protected void readUnsubscribed(BIDdfPollable ddfPollable)
    throws Exception
  {
    mySubscribedPollables.remove(ddfPollable);
    if(mySubscribedPollables.size() == 0)
    {
      ddfPollable.getDdfCommunicator().getDdfPollScheduler().unsubscribe((BIDdfPollable)this);
      subscribed = false;
    }
  }

  /**
   * This method is used internally to delegate some of the default
   * behavior to a default instance of the BIDdfPollable
   * @return
   */
  protected BIDdfPollable getDefaultPollable()
  {
    BIDdfPollable ddfPollables[] = getPollables();

    if (ddfPollables!=null && ddfPollables.length>0)
      return getPollables()[0];
    else
      throw new IllegalStateException("No BIDdfPollables in BDdfPollGroup: "+this);

  }

////////////////////////////////////////////////////////////////
// BIPollable
////////////////////////////////////////////////////////////////

  /**
   * Loops through all BIDdfPollables and returns the fastest poll frequency
   * Among them
   */
  public BPollFrequency getPollFrequency()
  {
    BPollFrequency fastestYet = BPollFrequency.slow; // Tracks the fastest poll frequency found among the group's proxies
    Iterator<BIDdfPollable> i = mySubscribedPollables.iterator();     // Loops through all proxies that are subscribed for polling

    while (i.hasNext())
    {
      BIDdfPollable ddfPollable = i.next();

      if (ddfPollable instanceof BIPollable)               // If any have a poll frequency
      {
        BPollFrequency fr = ((BIPollable)ddfPollable).getPollFrequency();
        if (fr.compareTo(fastestYet) < 0)            // The BPollFrequencies with lower ordinal values are faster than those with higher ordinal values
          fastestYet = fr;                           // Then let's keep track of the fastest frequency of them all
      }
    }
    return fastestYet;                               // Returns the fastest BPollFrequency from among the proxies, of by default, BPollFrequency.slow if no proxies or none have poll frequencies
  }

////////////////////////////////////////////////////////////////
// BIDdfPollable
////////////////////////////////////////////////////////////////

  /**
   * The descendant class should construct a DdfReadRequestUtil to help the ddf communicator
   * perform field-bus communication to poll this group of BIDdfPollables. It should call
   * the getDdfPollables method
   * and update each registered BIDdfPollable with the new data.
   */
  public BIDdfReadRequest makePollRequest()
  {
    // Determines the type of read request for the requestId
    BIDdfReadRequest defaultReadRequest = getDefaultPollable().makePollRequest();

    // Arranges for all BIDdfPollables to be updated with the result
    defaultReadRequest.setReadableSource(getPollables());

    return defaultReadRequest;
  }


  public BIDdfCommunicator getDdfCommunicator()
  {
    return getDefaultPollable().getDdfCommunicator();
  }

  public Object getPollGroupCode()
  {
    return code;
  }

  public Type getReadRequestType()
  {
    return getDefaultPollable().getReadRequestType();
  }

////////////////////////////////////////////////////////////////
// Util
////////////////////////////////////////////////////////////////

 /**
  * Ddf pollable values in the ddf need to call this method from their own
  * Niagara AX subscribed and/or readSubscribed call-back and pass themselves in, along with
  * the Context that Niagara AX passed to them.
  */
 public static void readSubscribed(BIDdfPollable valueToBePolled, Context cx)
      throws Exception
  {
    Object ddfPollGroupCode = valueToBePolled.getPollGroupCode();

    if (ddfPollGroupCode != null)
    {
      // Gets the ddf poll group for the given ddf pollable
      BDdfPollGroup pg = BDdfPollGroup.getPollGroup(valueToBePolled);
      pg.readSubscribed(valueToBePolled); // Subscribes the ddf pollable to the
                                          // BDdfPollGroup for polling
    }
  }

 /**
  * Ddf pollable values in the ddf need to call this method from their own
  * Niagara AX unsubscribed and/or readUnsubscried call-back and pass themselves in, along with
  * the Context that Niagara AX passed to them.
  */
 public static void readUnsubscribed(BIDdfPollable ddfValueToBePolled,
      Context cx) throws Exception
  {
    Object ddfPollGroupCode = ddfValueToBePolled.getPollGroupCode();

    if (ddfPollGroupCode != null)
    {
      // Gets the ddf poll group for the given ddf pollable
      BDdfPollGroup ezPg = BDdfPollGroup.getPollGroup(ddfValueToBePolled);
      ezPg.readUnsubscribed(ddfValueToBePolled); // Unsubscribes the ddf
                                                  // pollable from the
                                                  // BDdfPollGroup for polling
    }
  }

 /**
  * Ddf pollable values in the ddf need to call this method from their own
  * Niagara AX server-side started callback.
  */
  public static void register(BIDdfPollable valueToBePolled)
    throws Exception
  {
    Object ddfPollGroupCode = valueToBePolled.getPollGroupCode();

    if (ddfPollGroupCode!=null)
    {
      // Gets the ddf poll group for the given ddf pollable
      BDdfPollGroup pg = BDdfPollGroup.getPollGroup( valueToBePolled );
      pg.registerPollable(valueToBePolled); // Registers to the ddf pollable to the BDdfPollGroup for future polling
    }
  }

 /**
  * Ddf pollable values in the ddf need to call this method from their own
  * Niagara AX server-side stopped callback..
  */
  public static void unregister(BIDdfPollable ddfValueToBePolled)
    throws Exception
  {
    Object ddfPollGroupCode = ddfValueToBePolled.getPollGroupCode();

   if (ddfPollGroupCode != null)
    {
      // Gets the ddf poll group for the given ddf pollable
      BDdfPollGroup ezPg = BDdfPollGroup.getPollGroup( ddfValueToBePolled );
      ezPg.unregisterPollable(ddfValueToBePolled ); // Unregisters the ddf pollable from any future polling
    }
  }

 /**
  * This method returns an instance of BDdfPollGroup for the given
  * BIDdfPollable. Only one instance of a BDdfPollGroup will exist for
  * all BIDdfPollables that return the same, ddf poll group code. If that one
  * instance does not yet exist, one will be created, hashed, and returned. Otherwise
  * the one instance will be retrieved from a table and returned.
  *
  * This static method can be used as a public lookup to retrieve the instance of
  * BDdfPollGroup that is related to the given ddfPollable, regardless of what driver that
  * ddfPollable is in and regardless of the ultimate BDdfPollGroup type.
  */
  protected static BDdfPollGroup getPollGroup(BIDdfPollable ddfPollable)
  {
    Type readReqType = ddfPollable.getReadRequestType();
    Object pgCode = ddfPollable.getPollGroupCode();

    synchronized (readReqType) // Thread safety!
    {
      // First lets lookup a HashTable for the particular read request type
      Hashtable<Object, BDdfPollGroup> codesToGroups = reqTypesToGroupCodes.get(readReqType.getId());

      if (codesToGroups == null) // Let's create one if we have to
      {
        codesToGroups = new Hashtable<>();
        reqTypesToGroupCodes.put(readReqType.getId(), codesToGroups);
      }

      // Next lets get the poll group instance that is hashed to the
      // ddfPollable's code
      BDdfPollGroup pg = codesToGroups.get(pgCode);

      if (pg == null) // Let's create one if we have to
      {
        pg = new BDdfPollGroup();
        pg.code = pgCode;

        codesToGroups.put(pgCode, pg);
      }
      return pg;
    }
  }
////////////////////////////////////////////////////////////////
// Attributes
////////////////////////////////////////////////////////////////

  // This hashtable maps all driver poll group Types to a second Hashtable mapping codes to poll group instances
  static HashMap<Integer, Hashtable<Object, BDdfPollGroup>> reqTypesToGroupCodes = new HashMap<>();

  Object code; // This code is like a coalesce key. All BIDdfPollables that are a member of this particular instance of BDdfPollGroup should return the same code
  Vector<BIDdfPollable> mySubscribedPollables; // A list of all BIDdfPollables that this is subscribed for polling on behalf of
  Vector<BIDdfPollable> myDdfPollables; // A list of all proxies in general that can be polled by this poll group, regardless of whether or not they have presently requested to be registed for updates
  boolean subscribed = false;
}
