/*
 * @copyright 2005 Tridium Inc.
 */
package com.tridium.ddf.identify;

import javax.baja.sys.BInterface;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;

import com.tridium.ddf.IDdfFacetConst;

/**
 * This is fundamental to developer driver framework's automatic implementation of
 * the point-auto features (this is when an end-use right-clicks a
 * driver's control point with proxy ext and chooses auto or emergency
 * auto, when the point does not have a fall-back value or any other logic
 * linked to its priority input properties.
 *
 * Implementing this interface is only necessary if your driver requires
 * special requests be sent when the host control system (Niagara AX)
 * relinquishes its control over a point. The majority of protocols that
 * we have encountered do not require this!
 *
 * To take advantage of the developer driver framework's automatic implementation of the
 * point-auto feature, the driver developer should:
 *
 * 1.) Define a class that extends BDdfIdParams and implements this interface. If you
 * defined a class that implements BIDdfWriteParams then you should consider using
 * that class but then make it implement this interface too.
 *
 * 2.) Define a proxy ext class that extends BDdfProxyExt, if a suitable
 * one is not already defined. It is certainly valid to re-use a proxy ext
 * class created to take advantage of the corresponding BIDdfReadParams. And
 * it is highly encouraged to re-use the proxy ext class created to take
 * advantage of the corresponding BIDdfWriteParams interface.
 *
 * 3.) Define a request class that extends BDdfRequest and implements
 * BIDdfWriteRequest (or that extends BDdfWriteRequest for convenience). Note,
 * this is not a typographical error. The automatic point-auto feature also
 * uses special subclasses of BIDdfWriteRequest.
 *
 * 4.) Re-define the writeParameters property on the request created in step
 * three and make the default value be an instance of the class that
 * implements this interface.
 *
 * 5.) Re-define the writeParameters property on the proxy ext created in step
 * two and make the default value be an instance of the class that
 * implements this interface. Please note that if you already followed the
 * steps outlined for BIDdfWriteParams then you have already done this step.
 * =========================================================
 * To auto (relinquish control of) a control point in a device in a
 * driver that is built on the ddf, the
 * ddf:
 *
 * 1.)Calls getWriteParameters on the proxy extension.
 * (if getWriteParameters returns something that implements this
 * interface, then the ddf proceeds to the next
 * steps, otherwise, the developer driver framework does not perform any
 * special driver communication to auto (relinquish control of) a
 * control point.
 *
 * 2.)Calls getDdfAutoRequestType() on the instance of this
 * interface that it received in step 1. The Type needs to
 * be a class that implements BIDdfWriteRequest, even this
 * it is really an 'auto' request. The ddf
 * considers the 'auto' request to be a special case of a
 * 'write' requests.
 *
 * 3.)Instantiates an instance of the special write request class
 * that corresponds to the special write request type that the
 * ddf received in step 2.
 *
 * 4.)Calls setWriteParameters on the special write request
 * (for auto'ing) and passes in a copy of what it received back
 * in step 1.
 *
 * 5.)Calls setPointId on the special, write request (for auto'ing)
 *  and passes in a copy of the BDdfIdParams object that the proxy's
 * getPointId method returns.
 *
 * 6.)Calls setDeviceId on the special, write request (for auto'ing)
 * and passes in a copy of the BDdfIdParams object that the proxy's
 * device's getDeviceId method returns.
 *
 * @author lperkins
 *
 * @see com.tridium.ddf.comm.req.BDdfWriteRequest
 * @see com.tridium.ddf.comm.req.BIDdfWriteRequest
 * @see com.tridium.ddf.point.BDdfProxyExt
 *
 */
public interface BIDdfAutoParams
  extends BInterface, IDdfFacetConst

{
  /**
   * The implementing class needs to return the Type that
   * represents the ddf write requests used whenever a point is "auto'ed"
   * in Niagara from the driver whose writeParameters can be an instance of
   * this class.
   *
   * This is fundamental to developer driver framework's automatic processing of the point-
   * auto feature.
   *
   */
  public Type getAutoRequestType();

  ////////////////////////////////////////////////////////////////
  // Niagara AX Type
  ////////////////////////////////////////////////////////////////
  
  public static final Type TYPE = Sys.loadType(BIDdfAutoParams.class);
}
