/*
 * @copyright 2005 Tridium Inc.
 */
package com.tridium.ddf.discover;

import javax.baja.job.BJob;
import javax.baja.sys.Context;

import com.tridium.ddf.IDdfFacetConst;

/**
 * The 'ddf auto discovery job' passes an instance of this class
 * to each discovery request just after instantiating each
 * discovery request.
 *
 * Components that perform the actual field-bus communication
 * for a discovery process should implement this interface or
 * have a nested class that implements this interface.
 *
 * When a component performs the actual field-bus communication,
 * it should do the following:
 *
 * 1.) Create an instance BIDdfDiscoveryRequest.
 *
 * 2.) Call setDiscovery on the BIDdfDiscoveryRequest and pass
 * in itself or an instance of a nested class that implements this
 * interface.
 *
 * 3.) Call communicate on its communicator and pass in the
 * BIDdfDiscoveryRequest
 *
 * 4.) The ddf will transmit the necessary bytes
 * onto the field-bus.
 *
 * 5A.) If the ddf receives back a discovery
 * response, then the developer driver frame work will call the
 * discoverOk on this object (the BIDdfDiscoveryRequest's
 * discoverer) and pass in the discoveryObjects from the discovery
 * response.
 *
 * 5B.) If the ddf does not receive back a
 * discovery resposne, then the ddf will call
 * the discoveryFail method on this object (BIDdfDiscoveryRequest's
 * discover).
 * BDdfAutoDiscoveryJob implements this interface.
 *
 * @author lperkins
 *
 */
public interface IDdfDiscoverer
  extends IDdfFacetConst
{
  /**
   * The discovery job or a nested instance of the discovery job
   * will automatically receive the discovery objects from the
   * discovery request here. The 'ddf' calls this method automatically
   * after it matches up the developer's discovery request with
   * the disocvery response. Just prior to calling this method, the
   * ddf calls 'parseDiscoveryObjects' on the discovery response. Those
   * discovery objects are passed here, giving the 'ddf' a hook back
   * into the job and thereby passing the discovered objects back to
   * the job.
   */
  public void discoverOk(BIDdfDiscoveryObject[] discoveryObjects);
  /**
   * The discovery job or a nested instance of the discovery job
   * will automatically receive notification of failure to obtain
   * a discovery response here.
   *
   * The 'ddf' calls this method if it detects a timeout after
   * transmitting the discovery request. The 'reason' in that case
   * indicates that a timeout occured.
   */
  public void discoverFail(String reason);

  /**
   * The ddf passes this context to the discover response's 'parseDiscoveryObjects'
   * method. This tells the developer whether the job is discovering devices or
   * points. This Context is only important if the developer is using the same request
   * type for both a device learn and a point learn. Such a scenario is probably
   * unlikely.
   */
  public Context getDiscoverContext();

  /**
   * As mentioned, the discovery request is passed an instance of an IDdfDiscoverer.
   *
   * The developer may then call this method to get a direct reference to the
   * BJob that is performing the automatic discovery. With that reference,
   * the developer may, for example, call 'progress' to update the percent-complete
   * that is displayed in the JobBar. The developer can also call 'getJob().getLog().message(...)
   * to place a message onto the job bar.
   *
   * Please note, that progress is automatically reported provided that the developer's
   * 'discovery preferences' has a 'min' and 'max' discovery parameters that actually
   * 'loop' (by the term 'loop', we mean that the discovery parameters returns a new
   * instance of itself each time the getNext() method is called.
   *
   * Otherwise, the discovery process will probably only submit a single discovery request
   * whose discovery response returns all of the discovery objects. If that is the case, then
   * the developer could be using the BIDdfMultiFrameResponse and/or BIDdfTransmitAckResp to
   * perform his or her own complicated discovery scheme. In that case, the developer
   * may call 'getJob' on the discovery requests' "discoverer" and call 'progress',
   * 'getLog().message(...)', etc.
   */
  public BJob getJob();

}
