/*
 * @copyright 2005 Tridium Inc.
 */
package com.tridium.ddf.comm.rsp;

import javax.baja.status.BStatusValue;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;

import com.tridium.ddf.comm.req.IDdfReadable;

/**
 * Each **read** request that driver developers define need to return
 * a response from the request's processReceive method. The response
 * needs to implement BIDdfReadResponse. This can be most easily
 * accomplished by extending BDdfResponse and implementing this
 * interface. However, this interface extends BIDdfResponse to allow
 * you to use your own class, instead, if you need more flexibility.
 * You don't have to you your own class though, if you extend BDdfRequest,
 * because BDdfRequest conveniently implements BIDdfRequest. If you
 * do not extend BDdfRequest then you will have to implement
 * BIDdfResponse as well.
 *
 * @author lperkins
 */
public interface BIDdfReadResponse
  extends BIDdfResponse
{

  /**
   * Implementing this method is fundamental to the ddf's
   * auto-poll feature for driver points. When one or more driver points
   * under a device need polled that share the equivalent read parameters,
   * the ddf will instantiate the read request type that
   * is identified by the read parameters, assign the read parameters to the
   * read request, assign all points that share the read parameters to the
   * request, and transmit the read request. Upon receiving a successful
   * read response (an instance that implements this interface), the ddf
   * driver framework will loop through all of the points under the device
   * that shared the same read parameters, cast each point to IDdfReadable,
   * and pass each point successively to this method. The developer driver
   * framework will take the return value from this method and pass it
   * to the readOk method on the point, thereby updating its value in Niagara.
   *
   * When implenting this interface, driver devlopers must implement this
   * method and parse a BStatusValue from the response data for the given
   * readableSource. If necessary, we suggest that the driver developer can
   * check if the readableSource object is an instance of their driver's proxy
   * extension class. If so, the driver developer can cast the readableSource
   * object to their driver's proxy extension class and then access the point's
   * pointId property. The driver developer should design the pointId property
   * in such a way that it provides the information necessary to parse the
   * particular point's value from the read response.
   *
   *
   * @param readableSource
   * @return a BStatusValue to pass to the readOk method of the readableSource
   */
  public BStatusValue parseReadValue(IDdfReadable readableSource);
  
  ////////////////////////////////////////////////////////////////
  // Niagara AX Type
  ////////////////////////////////////////////////////////////////
  public static final Type TYPE = Sys.loadType(BIDdfReadResponse.class);
  
}
