/*
 * @copyright 2005 Tridium Inc.
 */
package com.tridium.ddf.comm.req.util;

import javax.baja.status.BStatusValue;

import com.tridium.ddf.IDdfFacetConst;
import com.tridium.ddf.comm.defaultComm.DdfDefaultCommLexicon;
import com.tridium.ddf.comm.req.BIDdfWriteRequest;
import com.tridium.ddf.comm.req.IDdfReadable;
import com.tridium.ddf.comm.req.IDdfWritable;
import com.tridium.ddf.comm.rsp.BIDdfReadResponse;
import com.tridium.ddf.comm.rsp.BIDdfResponse;
import com.tridium.ddf.comm.rsp.DdfResponseException;
import com.tridium.ddf.point.BIDdfWritable;
import com.tridium.ddf.point.DdfPointLexicon;

public class DdfWriteRequestUtil
  implements IDdfFacetConst
{
////////////////////////////////////////////////////////////////
// Util
////////////////////////////////////////////////////////////////
  
  /**
   * This is a callback method that the BDdfCommunicator calls when it pairs up a write response
   * with a write request.
   * 
   * This loops through all writableSource objects for the ddfReq and calls writeOk on 
   * each of them, passing in the result of calling getWriteValue on writable source object.
   * 
   * @param the BIDdfRequest to which a BIDdfResponse just matched.
   * @param the BIDdfResponse that matches up with the BIDdfWriteRequest ddfRsp
   */
  public static void processResponse(BIDdfWriteRequest ddfReq, BIDdfResponse ddfRsp)
  {
    if (ddfReq.getAutoWriteOk())
    {
      IDdfWritable[] writableSource = ddfReq.getWritableSource();
      
      if (writableSource!=null)
        updateWritables(ddfRsp,writableSource);
    }
  }
  
  /**
   * This is a callback method that the BDdfCommunicator calls when it decides to give up on receiving
   * a response message for this message.
   * 
   * This loops through all writableSource objects for the ddfReq and calls writeFail on 
   * each of them. After that, this calls DdfRequestUtil.processTimeout.
   */
  public static void processTimeout(BIDdfWriteRequest ddfReq)
  {
    if (ddfReq.getAutoWriteFailOnTimeout())
    {
      IDdfWritable[] writableSource = ddfReq.getWritableSource();
      
      if (writableSource!=null)
        for (int i=0; i<writableSource.length; i++)
          writableSource[i].writeFail(DdfRequestLexicon.requestTimeout);
    }
  }
  
  /**
   * This is a callback method that the BDdfCommunicator calls when it pairs up a write response
   * with a write request, late, after the write request has alwritey timed out.
   * 
   * If the getAutoWriteOkLate method of the ddfRsp returns true then this loops through all
   * writableSource objects for the ddfReq and calls writeOk on each of them, passing in the
   * result of calling getWriteValue on the writableSource object.
   * 
   * @param the BIDdfWriteRequest to which a BIDdfWriteResponse just matched, late, after time-out.
   * @param the BIDdfResponse that matches up with the BIDdfWriteRequest ddfRsp
   */
  public static void processLateResponse(BIDdfWriteRequest ddfReq, BIDdfResponse ddfRsp)
  {
    if (ddfReq.getAutoWriteOkLate())
    {
      IDdfWritable[] writableSource = ddfReq.getWritableSource();
      
      if (writableSource!=null)
        updateWritables(ddfRsp,writableSource);    }
  }
  /**
   * This is a callback method that DdfRequestUtil calls when it pairs up the received
   * frame(s) with a request message but the frames indicate that an error condition exists
   * in the device preventing it from responding successfully.
   * 
   * @param an ddfReq that just received a response
   * @param the ddfRsp that was just received for the ddfReq
   */
  public static void processErrorResponse(BIDdfWriteRequest ddfReq, DdfResponseException errorRsp)
  {
    if (ddfReq.getAutoWriteFailOnError())
    {
      IDdfWritable[] writableSource = ddfReq.getWritableSource();
      
      if (writableSource!=null)
        for (int i=0; i<writableSource.length; i++)
          writableSource[i].writeFail(
              DdfDefaultCommLexicon.responseError(errorRsp));
    }
    
  }  
  /**
   * This method is called by the updateWritables method IF the write
   * response also happens to implement BIDdfReadResponse.
   * 
   * This method calls readOk on any of the req/rsp's writable points
   * for any that happen to also be readable points (in the event that
   * the response happens to also implement BIDdfReadResponse).
   * 
   * @param ddfReadRsp
   * @param writableSource
   */
  private static void updateReadValues(BIDdfReadResponse ddfReadRsp, IDdfWritable[] writableSource)
  {
    if (writableSource!=null) // Sanity check
      for (int i=0; i<writableSource.length; i++) // Loops through the writables
        if (writableSource[i] instanceof IDdfReadable) // If any are readable, then let's call readOk...
          DdfReadRequestUtil.callReadOk((IDdfReadable)writableSource[i],ddfReadRsp);
  }
  
  /**
   * This method is called by the processLateResponse and processResponse methods.
   * Its purpose is to call writeOk on all writable points for the req/rsp transaction.
   * @param writableSource
   */
  private static void updateWriteValues(IDdfWritable[] writableSource)
  {
    // Loops through all other writable sources
    for (int i=0; i<writableSource.length; i++)
    { 
      // Gets the particular writable source's write value
      BStatusValue writableSourceValue = writableSource[i].getWriteValue();
      
      // TODO: This will not update properly if the writableSource's writeValue changes after the recent request has been transmitted,
      // Because then the writeValue becomes the next value to write. Of course, that next value will likely be in the queue to be written, but still,
      // We could be updating the writeValue incorrectly. 
      writableSource[i].writeOk(writableSourceValue); // The zero'eth writableSource is the one whose value was just written.
    }
  }
  
  private static void updateWritables(BIDdfResponse ddfRsp, IDdfWritable[] writableSource)
  { 
    updateWriteValues(writableSource);
    
    // If the write response can be used to update values
    if (ddfRsp instanceof BIDdfReadResponse) // Then this will attempt to update each writableSource that is also an IDdfReadable
      updateReadValues((BIDdfReadResponse)ddfRsp, writableSource);
  }
  
  /**
   * This method is called by the doWrite method of BDdfProxyExt to write itself in a generic fashion. This
   * generic fashion can hopefully be applied to phantom points, when they are ready for prime-time in release 3.2.
   * @param writable
   * @param out
   */
  public static void doWrite(BIDdfWritable writable, BStatusValue out)
  { 
    // Asks the writable for an instance of BIDdfWriteRequest that is capable
    // Of Writing itself out.
    BIDdfWriteRequest writeReq = writable.makeWriteRequest();
    
    if (writeReq==null)
    {
      // If no writeReq was defined then this calls writeFail on the point and logs the error
      String writeFailReason = DdfPointLexicon.writeNotImplemented(writable.toString());
      
      writable.writeFail(writeFailReason);
      writable.getDdfCommunicator().getLog().error(writeFailReason);
    }
    else
      writable.getDdfCommunicator().communicate(writeReq);
  }
  /**
   * This method is called by the doWrite method of BDdfProxyExt to write itself in a generic fashion. This
   * generic fashion can hopefully be applied to phantom points, when they are ready for prime-time in release 3.2.
   * @param writable
   */
  public static void doAuto(BIDdfWritable writable)
  { 
    // Asks the writable for an instance of BIDdfWriteRequest that is capable
    // Of Writing itself out.
    BIDdfWriteRequest autoReq = writable.makeAutoRequest();
    
    if (autoReq == null)
    { 
      // Please notice that the lack of an auto-request is not an error. It is simply something that we trace.
      if (writable.getDdfCommunicator().getLog().isTraceOn())
        writable.getDdfCommunicator().getLog().trace(DdfPointLexicon.noAutoRequestDefined(writable.toString()));
      //else, trace is not on therefore we don't need to log anything
    }
    else // Transmits the auto-request
      writable.getDdfCommunicator().communicate(autoReq);
  }
  
}
