/*
 * @copyright 2005 Tridium Inc.
 */
package com.tridium.ddf.comm.req.util;

import javax.baja.space.BComponentSpace;
import javax.baja.sys.BComponent;

import com.tridium.ddf.DdfLexicon;
import com.tridium.ddf.IDdfFacetConst;
import com.tridium.ddf.comm.BIDdfCommunicating;
import com.tridium.ddf.comm.BIDdfCommunicator;
import com.tridium.ddf.comm.IDdfDataFrame;
import com.tridium.ddf.comm.defaultComm.BDdfNullCommunicator;
import com.tridium.ddf.comm.req.BIDdfCustomRequest;
import com.tridium.ddf.comm.req.BIDdfDiscoverRequest;
import com.tridium.ddf.comm.req.BIDdfPingRequest;
import com.tridium.ddf.comm.req.BIDdfReadRequest;
import com.tridium.ddf.comm.req.BIDdfRequest;
import com.tridium.ddf.comm.req.BIDdfWriteRequest;
import com.tridium.ddf.comm.rsp.BIDdfDiscoverResponse;
import com.tridium.ddf.comm.rsp.BIDdfReadResponse;
import com.tridium.ddf.comm.rsp.BIDdfResponse;
import com.tridium.ddf.comm.rsp.DdfResponseException;
import com.tridium.ddf.identify.BDdfIdParams;
import com.tridium.fox.sys.broker.BFoxStationSpace;

/**
 * This class contains methods that allow the communicator to process
 * communication errors, responses, and timeouts in an orderly fashion.
 * 
 * This class also features the 'communicateSync' method that allows
 * developers to perform driver communications in a blocking (synchronous)
 * fashion.
 *
 * @author    lperkins
 * @creation  Oct 9, 2006
 * @version   $Revision$ $Date$
 * @since     Niagara 3.0
 */
public abstract class DdfRequestUtil
  implements IDdfFacetConst
{
////////////////////////////////////////////////////////////////
// Util
////////////////////////////////////////////////////////////////
  
  /**
   * This is a callback method that the BDdfCommunicator calls when it pairs up a response message
   * with a request message.
   *
   * If the request is a ddf custom request then this method calls
   * processResponse on the request to allow the implementor to take
   * appropriate, custom action. After that this calls notifyAll on
   * the object to optionally allow stuff such as the poll method to
   * wait on the request for the transaction to be completed.
   *
   * @param an ddfReq that just received a response
   * @param the ddfRsp that was just received for the ddfReq
   */
  public static void processResponse(BIDdfRequest ddfReq, BIDdfResponse ddfRsp)
  {
    try
    {
      // Attempts to update the device id in the response, if it was not initialized by the
      // Request's processReceive logic.
      if (ddfRsp.getDeviceId().getType() == BDdfIdParams.TYPE)
      {
        ddfRsp.setDeviceId((BDdfIdParams)ddfReq.getDeviceId().newCopy());
      }
      if (ddfReq instanceof BIDdfReadRequest)
      {
        DdfReadRequestUtil.processResponse((BIDdfReadRequest)ddfReq, (BIDdfReadResponse)ddfRsp);
      }
      if (ddfReq instanceof BIDdfPingRequest)
      {
        DdfPingRequestUtil.processResponse((BIDdfPingRequest)ddfReq,ddfRsp);
      }
      if (ddfReq instanceof BIDdfWriteRequest)
      {
        DdfWriteRequestUtil.processResponse((BIDdfWriteRequest)ddfReq, ddfRsp);
      }
      if (ddfReq instanceof BIDdfDiscoverRequest && ddfRsp instanceof BIDdfDiscoverResponse)
      {
        DdfDiscoveryRequestUtil.processResponse(
            (BIDdfDiscoverRequest)ddfReq,
            (BIDdfDiscoverResponse)ddfRsp);
      }
      if (ddfReq instanceof BIDdfCustomRequest)
      {
        ((BIDdfCustomRequest)ddfReq).processResponse(ddfRsp);
      }
    }
    finally
    {
      synchronized(ddfReq)
      {
        ddfReq.notifyAll(); // Wakes up any threads that are waiting in the communicateSync method of the ddfReq
      }
    }
  }
  /**
   * This is a callback method that the BDdfCommunicator calls when
   * it decides to give up on receiving a response for a ddf request.
   *
   * If the request is a ddf custom request then this method calls
   * processTimeout on the request to allow the implementor to take
   * appropriate, custom action. After that this calls notifyAll on
   * the object to optionally allow stuff such as the poll method to
   * wait on the request for the transaction to be completed.
   *
   * @param a ddfReq that just timeout out
   */
  public static void processTimeout(BIDdfRequest ddfReq)
  {
    try
    {
      if (ddfReq instanceof BIDdfCustomRequest)
      {
        ((BIDdfCustomRequest)ddfReq).processTimeout();
      }
      if (ddfReq instanceof BIDdfReadRequest)
      {
        DdfReadRequestUtil.processTimeout((BIDdfReadRequest)ddfReq);
      }
      if (ddfReq instanceof BIDdfPingRequest)
      {
        DdfPingRequestUtil.processTimeout((BIDdfPingRequest)ddfReq);
      }
      if (ddfReq instanceof BIDdfWriteRequest)
      {
        DdfWriteRequestUtil.processTimeout((BIDdfWriteRequest)ddfReq);
      }
      if (ddfReq instanceof BIDdfDiscoverRequest)
      {
        DdfDiscoveryRequestUtil.processTimeout((BIDdfDiscoverRequest)ddfReq);
      }
    }
    finally
    {
      synchronized(ddfReq)
      {
        ddfReq.notifyAll(); // Wakes up any threads that are waiting in the communicateSync method of the ddfReq
      }
    }
  }

  /**
   * This is a callback method that the BDdfCommunicator calls if it
   * recieves a response to a BIDdfRequest message after it has timed out.
   *
   * If the request is a ddf custom request then this method calls
   * processLateResponse on the request to allow the implementor to take
   * appropriate, custom action..
   *
   * @param an ddfReq that just received a late response, after previously timing out
   * @param the ddfRsp that was just received for the ddfReq
   */
  public static void processLateResponse(BIDdfRequest ddfReq, BIDdfResponse ddfRsp)
  {
    // Attempts to update the device id in the response, if it was not initialized by the
    // Request's processReceive logic.
    if (ddfRsp.getDeviceId().getType() == BDdfIdParams.TYPE)
    {
      ddfRsp.setDeviceId((BDdfIdParams)ddfReq.getDeviceId().newCopy());
    }
    if (ddfReq instanceof BIDdfReadRequest)
    {
      DdfReadRequestUtil.processLateResponse((BIDdfReadRequest)ddfReq, (BIDdfReadResponse)ddfRsp);
    }
    if (ddfReq instanceof BIDdfPingRequest)
    {
      DdfPingRequestUtil.processLateResponse((BIDdfPingRequest)ddfReq,ddfRsp);
    }
    if (ddfReq instanceof BIDdfWriteRequest)
    {
      DdfWriteRequestUtil.processLateResponse((BIDdfWriteRequest)ddfReq, ddfRsp);
    }
    if (ddfReq instanceof BIDdfDiscoverRequest && ddfRsp instanceof BIDdfDiscoverResponse)
    {
      DdfDiscoveryRequestUtil.processLateResponse(
          (BIDdfDiscoverRequest)ddfReq,
          (BIDdfDiscoverResponse)ddfRsp);
    }
    if (ddfReq instanceof BIDdfCustomRequest)
    {
      ((BIDdfCustomRequest)ddfReq).processLateResponse(ddfRsp);
    }
}

  /**
   * After transmitting this request, the BDdfCommunicator will pass all data frames that it receives
   * here.
   *
   * This method in turn calls ddfProcessReceive as a hook for descendant classes to implement.
   *
   *  This request needs to take one of the following steps:
   *   1. Ignore the frame and return null.
   *   2. Collect the frame and return null.
   *   3. Return a DdfResponse for the data frame and any previously collected frames that comprise
   *   the response (the latter is not typical).
   *   4. Throw an DdfResponseException or subclass there-of to indicate the the frame
   *   forms a complete message but indicates an error condition in the device preventing
   *   a successful response.
   * @param iDdfDataFrame
   * @return
   */
  public static BIDdfResponse processReceive(BIDdfRequest ddfReq, IDdfDataFrame iDdfDataFrame)
    throws DdfResponseException
  {
    if (ddfReq instanceof BIDdfReadRequest)
      return DdfReadRequestUtil.processReceive((BIDdfReadRequest)ddfReq,iDdfDataFrame);
    // else, NOTE: Ping messages do not need special processing
    else
      return ddfReq.processReceive(iDdfDataFrame);
  }

  /**
   * This is a callback method that the BDdfCommunicator calls when it pairs up the received
   * frame(s) with a request message but the frames indicate that an error condition exists
   * in the device preventing it from responding successfully.
   *
   * @param an ddfReq that just received a response
   * @param the ddfRsp that was just received for the ddfReq
   */
  public static void processErrorResponse(BIDdfRequest ddfReq, DdfResponseException errorRsp)
  {
    try
    {
      if (ddfReq instanceof BIDdfReadRequest)
      {
        DdfReadRequestUtil.processErrorResponse((BIDdfReadRequest)ddfReq, errorRsp);
      }
      if (ddfReq instanceof BIDdfPingRequest)
      {
        DdfPingRequestUtil.processErrorResponse((BIDdfPingRequest)ddfReq,errorRsp);
      }
      if (ddfReq instanceof BIDdfWriteRequest)
      {
        DdfWriteRequestUtil.processErrorResponse((BIDdfWriteRequest)ddfReq, errorRsp);
      }
      if (ddfReq instanceof BIDdfDiscoverRequest)
      {
        DdfDiscoveryRequestUtil.processErrorResponse(
            (BIDdfDiscoverRequest)ddfReq, errorRsp);
      }
      if (ddfReq instanceof BIDdfCustomRequest)
      {
        ((BIDdfCustomRequest)ddfReq).processErrorResponse(errorRsp);
      }
    }
    finally
    {
      synchronized(ddfReq)
      {
        ddfReq.notifyAll(); // Wakes up any threads that are waiting in the waitForTransactionComplete method of the ddfReq
      }
    }
  }

  /**
   * This method simulates a synchronous communication mechanism.
   *
   * Note: The developer driver framework functions in a completely asynchronous
   * fashion. If the developer needs to take some action as a result of the request
   * timing out or receiving a response, then the correct way to accomplish this
   * is by making the request implement BIDdfCustomRequest. By doing that, the
   * developer driver framework will make some callbacks onto the request itself
   * when the request either times out or gets a response. In the latter case, the
   * response is even passed to the callback method.
   *
   * If a developer insists on a synchronous communications interface then the
   * developer may call this method, pass in the communicator, and pass in the
   * request. This method will block until the response has been received for the
   * request or until the request times out.
   *
   * @param communicator
   * @param req
   * 
   * @throws the BajaRuntimeException from BIDdfCommunicator.communicator if
   * the 'isCommEnabled' method of BDdfCommunicator returns false (typically
   * indicating that the BDdfCommunicator is either disabled or fault).
   */
  public static void communicateSync(BIDdfCommunicator communicator, BIDdfRequest req)
  {
    synchronized(req) // Locks the request
    {                 // Sends the request as a transaction. Since the request is locked,
                      // The communicator cannot call 'notifyAll' on the request until
                      // The request calls 'wait'
      try
      {
        communicator.communicate(req);
        if (communicator instanceof BComponent)
        {
          BComponent realCommComponent = (BComponent)communicator;
          BComponentSpace realCommComponentSpace = realCommComponent.getComponentSpace();
          if ( (realCommComponentSpace == null) ||                     // Sanity check... or
               (realCommComponentSpace instanceof BFoxStationSpace) || // If the communicator is running on the client-side (for video) 
               (realCommComponent.isRunning()))                        // Or if the station has not yet shut down
          {
            req.wait();   // Unlocks the request and waits for the communicator to call 'notifyAll' on the request
          }
        }
        else if (communicator instanceof BDdfNullCommunicator)
        {
          BIDdfCommunicating commParent = ((BDdfNullCommunicator)communicator).getDdfCommunicatingComponent();
          BIDdfCommunicator realComm = commParent.getDdfCommunicator();
          
          if (realComm instanceof BComponent)
          {
            BComponent realCommComponent = (BComponent)realComm;
            BComponentSpace realCommComponentSpace = realCommComponent.getComponentSpace();
//System.out.println("realCommComponentSpace.getClass().getName() = "+realCommComponentSpace.getClass().getName());
            if ( (realCommComponentSpace == null) ||                     // Sanity check... or
                 (realCommComponentSpace instanceof BFoxStationSpace) || // If the communicator is running on the client-side (for video) 
                 (realCommComponent.isRunning()))                        // Or if the station has not yet shut down
            {
              req.wait();   // Unlocks the request and waits for the communicator to call 'notifyAll' on the request
            }
          }
        }
        else
        {
          req.wait();
        }
      }
      catch (InterruptedException ie)
      {
        if (communicator.getLog().isTraceOn())
          communicator.getLog().trace(DdfLexicon.ddfCommunicateSyncError,ie);
      }
    }
  }
}
