/*
 * @copyright 2005 Tridium Inc.
 */
package com.tridium.ddf.comm.req.util;

import javax.baja.sys.BajaRuntimeException;

import com.tridium.ddf.IDdfFacetConst;
import com.tridium.ddf.comm.IDdfDataFrame;
import com.tridium.ddf.comm.defaultComm.DdfDefaultCommLexicon;
import com.tridium.ddf.comm.req.BIDdfReadRequest;
import com.tridium.ddf.comm.req.IDdfReadable;
import com.tridium.ddf.comm.rsp.BIDdfReadResponse;
import com.tridium.ddf.comm.rsp.BIDdfResponse;
import com.tridium.ddf.comm.rsp.DdfResponseException;

/**
 * Processes read requests on behalf of the transaction manager.
 *
 * @author lperkins
 *
 */
public class DdfReadRequestUtil
  implements IDdfFacetConst
{

  ////////////////////////////////////////////////////////////////
//Util
////////////////////////////////////////////////////////////////

  /**
   * This is a callback method that the BDdfCommunicator calls when it pairs up a read response
   * with a read request.
   *
   * This loops through all readableSource objects for the ddfReq and calls readOk on
   * each of them, passing in the result of calling parseReadValue on the ddf response
   * passing in the readableSource object. After that, this calls DdfRequestUtil.processResponse
   *
   * @param the BIDdfRequest to which a BIDdfResponse just matched.
   * @param the BIDdfResponse that matches up with this DdfReadRequestUtil ddfRsp
   */
  public static void processResponse(BIDdfReadRequest ddfReq, BIDdfReadResponse ddfRsp)
  {
    if (ddfReq.getAutoReadOk()) // If the request indicates that it wants the developer driver
    {                           // Framework to automatically call "readOk"
      callReadOk(ddfReq,ddfRsp);
    }
  }

  /**
   * This method calls the readOk method on all IDdfReadables that the given request and
   * response can update.
   */
  private static void callReadOk(BIDdfReadRequest ddfReq, BIDdfReadResponse ddfRsp)
  {
    // Gets all data points that the request/response can update
    IDdfReadable[] readableSource = ddfReq.getReadableSource();
    
    if (readableSource!=null) // Sanity check
    { 
      // Loops throuh all data points that the req/rsp can update
      for (int i=0; i<readableSource.length; i++)
      {
        callReadOk(readableSource[i],ddfRsp);
      }
    }
  }

  static void callReadOk(IDdfReadable readable, BIDdfReadResponse readRsp)
  {
    try
    { 
      // Calls parseReadValue on the repsonse, passes in the data point, and passes
      // The return value to the data point's "readOk" method.
      readable.readOk(readRsp.parseReadValue(readable));
    }
    catch (Exception e)
    { 
      // If an exceptions are thrown then this calls the data point's
      // "readFail" method with a localized string that includes information
      // About the exception
      readable.readFail(DdfDefaultCommLexicon.parseError(e));
    }

  }

  /**
   * This is a callback method that the BDdfCommunicator calls when it decides to give up on receiving
   * a response message for this message.
   *
   * This loops through all readableSource objects for the ddfReq and calls readFail on
   * each of them. After that, this calls DdfRequestUtil.processTimeout.
   */
  public static void processTimeout(BIDdfReadRequest ddfReq)
  {
    if (ddfReq.getAutoReadFailOnTimeout())
    {
      IDdfReadable[] readableSource = ddfReq.getReadableSource();
      if (readableSource!=null)
        for (int i=0; i<readableSource.length; i++)
          readableSource[i].readFail(DdfRequestLexicon.requestTimeout);
    }
  }

  /**
   * This is a callback method that the BDdfCommunicator calls when it pairs up a read response
   * with a read request, late, after the read request has already timed out.
   *
   * If the getAutoReadOkLate method of the ddfRsp returns true then this loops through all
   * readableSource objects for the ddfReq and calls readOk on each of them, passing in the
   * result of calling parseReadValue on the ddf response and passing in the readableSource
   * object. After that, this calls DdfRequestUtil.processResponse
   *
   * @param the BIDdfReadRequest to which a BIDdfReadResponse just matched, late, after time-out.
   * @param the BIDdfReadResponse that matches up with this DdfReadRequestUtil ddfRsp
   */
  public static void processLateResponse(BIDdfReadRequest ddfReq, BIDdfReadResponse ddfRsp)
  {
    if (ddfReq.getAutoReadOkLate())
    {
      callReadOk(ddfReq,ddfRsp);
    }
  }

  /**
   * After transmitting a read request, the BDdfCommunicator will pass all data frames that it receives
   * for the request here. This request needs to take one of the following steps:
   *   1. Ignore the frame and return null.
   *   2. Collect the frame and return null.
   *   3. Return a DdfResponse for the data frame and any previously collected frames that comprise
   *   the response (the latter is not typical).
   * @param iDdfDataFrame
   * @return the BIDdfResponse that is returned by calling processReceive on the ddfReq passing in
   * the iDdfDataFrame. One caveat though: the BIDdfResponse must be a BIDdfReadResponse
   */
  public static BIDdfResponse processReceive(BIDdfReadRequest ddfReq, IDdfDataFrame iDdfDataFrame)
    throws DdfResponseException
  {
    BIDdfResponse ddfRsp = ddfReq.processReceive(iDdfDataFrame);
    
    //  Null is ok because it indicates that the frame was not for the request, otherwise, the returned
    // response must be a read response in the special case of a read reqeust    
    if (ddfRsp==null || ddfRsp instanceof BIDdfReadResponse)  
      return ddfRsp;
    else // Throws two BajaRuntimeExceptions nested together, one identifing the specific driver by name
      throw new BajaRuntimeException( // That needs fixed and the other that indicates that the request needs
                  DdfRequestLexicon.pleaseFixDriver(ddfReq.getType()), // To pair up with a BIDdfReadResponse
                  new BajaRuntimeException(
                      DdfRequestLexicon.readReqMustUseReadRsp(ddfReq.getType())));
  }
  /**
   * This is a callback method that DdfRequestUtil calls when it pairs up the received
   * frame(s) with a request message but the frames indicate that an error condition exists
   * in the device preventing it from responding successfully.
   *
   * @param a ddfReq that just received a response
   * @param the ddfRsp that was just received for the ddfReq
   */
  public static void processErrorResponse(BIDdfReadRequest ddfReq, DdfResponseException errorRsp)
  {
    if (ddfReq.getAutoReadFailOnError())
    {
      IDdfReadable[] readableSource = ddfReq.getReadableSource();
      
      if (readableSource!=null)
        for (int i=0; i<readableSource.length; i++)
          readableSource[i].readFail(
              DdfDefaultCommLexicon.responseError(errorRsp));
    }

  }
}
