/*
 * @copyright 2005 Tridium Inc.
 */
package com.tridium.ddf.comm.req.util;

import com.tridium.ddf.IDdfFacetConst;
import com.tridium.ddf.comm.defaultComm.DdfDefaultCommLexicon;
import com.tridium.ddf.comm.req.BIDdfPingRequest;
import com.tridium.ddf.comm.req.IDdfPingable;
import com.tridium.ddf.comm.rsp.BIDdfResponse;
import com.tridium.ddf.comm.rsp.DdfResponseException;
import com.tridium.ddf.identify.BDdfIdParams;

/**
 * Processes ping transactions on behalf of the transaction
 * manager.
 * 
 * @author lperkins
 *
 */
public class DdfPingRequestUtil
  implements IDdfFacetConst
{
  
////////////////////////////////////////////////////////////////
// Util
////////////////////////////////////////////////////////////////
  
  /**
   * This is a callback method that the BDdfCommunicator calls when it pairs up a ping response
   * with this ping message. If the getAutoPingOk method of the pingReq returns true then this
   * method loops through all of the pingable source objects on the pingReq and calls pingOk
   * for each.
   * 
   * @param pingReq is a ping request that just received a matching response
   * @param ddfRsp is the response that matches the pingReq.
   */
  public static void processResponse(BIDdfPingRequest pingReq, BIDdfResponse ddfRsp)
  {
    if (pingReq.getAutoPingOk())
    {
      IDdfPingable[] pingableSource = pingReq.getPingableSource();
      
      if (pingableSource!=null)
        for (int i=0; i<pingableSource.length; i++)
        {
          pingableSource[i].setDeviceId( (BDdfIdParams)
                ddfRsp.getDeviceId().newCopy());
          pingableSource[i].pingOk();
        }
    }
  }
  
  /**
   * This is a callback method that the BDdfCommunicator calls when it decides to give up on receiving
   * a response message for a ping message.
   * 
   * @param pingReq is the ping request that just timed out
   */
  public static void processTimeout(BIDdfPingRequest pingReq)
  {
    if (pingReq.getAutoPingFailOnTimeout())
    {
      IDdfPingable[] pingableSource = pingReq.getPingableSource();
      
      if (pingableSource!=null)
        for (int i=0; i<pingableSource.length; i++)
          pingableSource[i].pingFail(DdfDefaultCommLexicon.requestTimeOut);
    }
  }
  
  /**
   * This is a callback method that the BDdfCommunicator calls if it recieves a response to a ping
   * request after the ping request has timed out.
   * 
   * @param pingReq is a ping request that just received a matching response, although the ping request
   * has already timed-out.
   * @param ddfRsp is the response that matches the pingReq, albeit, late.
   */
  public static void processLateResponse(BIDdfPingRequest pingReq, BIDdfResponse ddfRsp)
  {
    if (pingReq.getAutoPingOkLate())
    {
      IDdfPingable[] pingableSource = pingReq.getPingableSource();
      
      if (pingableSource!=null)
        for (int i=0; i<pingableSource.length; i++)
        {
          pingableSource[i].setDeviceId( (BDdfIdParams)
              ddfRsp.getDeviceId().newCopy());
          
          pingableSource[i].pingOk();
        }
    }
  }
  
  /**
   * This is a callback method that DdfRequestUtil calls when it pairs up the received
   * frame(s) with a request message but the frames indicate that an error condition exists
   * in the device preventing it from responding successfully.
   * 
   * @param a ddfReq that just received a response
   * @param the ddfRsp that was just received for the ddfReq
   */
  public static void processErrorResponse(BIDdfPingRequest pingReq, DdfResponseException errorRsp)
  {
    if (pingReq.getAutoPingFailOnError())
    {
      IDdfPingable[] pingableSource = pingReq.getPingableSource();
      
      if (pingableSource!=null)
        for (int i=0; i<pingableSource.length; i++)
          pingableSource[i].pingFail(
              DdfDefaultCommLexicon.responseError(errorRsp));
    }
  }
}
