/*
 * @copyright 2005 Tridium Inc.
 */
package com.tridium.ddf.comm.req;

import javax.baja.sys.BInterface;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;

import com.tridium.ddf.comm.IDdfDataFrame;
import com.tridium.ddf.comm.rsp.BIDdfResponse;
import com.tridium.ddf.comm.rsp.DdfResponseException;

/**
 * The driver developer should make request message
 * types implement this method if the driver developer
 * wishes to perform his or her own special processing
 * on the response.
 *
 * Please keep in mind that devDriver automatically
 * provides default behavior in BIDdfReadRequest to
 * update control points, in BIDdfPingRequest to
 * update pingable devices, networks, etc, in BIDdfWriteRequest
 * to update the value of one or more driver points, and in 
 * BIDdfDiscoveryRequest to find one or more driver points
 * or devices. Therefore, a driver developer should only
 * implement this interface if he or she needs to perform
 * additional operations upon the response.
 *
 * @author lperkins
 *
 */
public interface BIDdfCustomRequest
  extends BInterface
{
  /**
   * This is a callback method that the BDdfCommunicator calls if it recieves a response to this
   * message after it has timed out.
   *
   * @param ddfRsp
   */
  public void processLateResponse(BIDdfResponse ddfRsp);

  /**
   * After transmitting this request, the BDdfCommunicator will pass frames that it receives
   * here. If you implement the getTag method then the ddf communicator will only pass data frames
   * whose tag's hashcode matches your request tag's hashcode. If your request returns null from
   * the getTag method then all received data frames will be passed here, until the request times
   * out or returns a BIDdfResponse from this method.
   *
   *  This request needs to take one of the following steps:
   *   1. Ignore the frame and return null. The transaction remains open.
   *   2. Collect the frame and return a BIDdfMultiFrameResponse whose <i>isComplete</i> method returns <i>false</i>. In which case, you need to implement your own collection
   *      mechanism. For example, this could be as simple as putting them all in a Vector in the BIDdfMultiFrameResponse. In
   *      this scenario, the transaction remains open. The transaction response timeout will be reset to the <i>responseTimeout</i> of the
   *      returned <i>BIMultiFrameResponse</i> thereby allowing more time for the remainder of the transaction to complete.
   *   3. Return a BIDdfResponse for the data frame and any previously collected frames that you determine together make up a completed response.
   *      This completes the transaction. Alternatively, if you have any previously collected frames then you may return the instance of
   *      BIDdfMultiFrameResponse as long as its <i>isComplete</i> method subsequently returns <i>true</i>. Both of these scenarios
   *      complete the transaction.  
   *   4. Throw an DdfResponseException or subclass there-of to indicate the the frame
   *   forms a complete message but indicates an error condition in the device preventing
   *   a successful response. This completes the transaction.
   *
   *   WARNING: In scenario's 2 and 3, please copy the frame's bytes as the frame's byte array could be a direct reference to an internal
   *   buffer in the receiver.
   *   
   * @param iDdfDataFrame
   * 
   * @return BIDdfResponse but please review the full description for this method.
   * 
   * @throws DdfResponseException but please review the full description for this method.
   */
  public BIDdfResponse processReceive(IDdfDataFrame iDdfDataFrame)
    throws DdfResponseException;

  /**
   * This is a callback method that the BDdfCommunicator calls when it pairs up a response message
   * with this message.
   *
   * @param ddfRsp
   */
  public void processResponse(BIDdfResponse ddfRsp);
  /**
   * The BDdfTransactionManager calls this method on responses
   * when they timeout. The response should update any database
   * or discovery items accordingly.
   */
  public void processTimeout();

  /**
   * This is a callback method that the BDdfCommunicator calls when it pairs up the received
   * frame(s) to this request but the receive frame(s) report an error condition that inhibits
   * the device from responding successfully.
   *
   * @param errorRsp - the DdfResponseException that was thrown from the processReceive method.
   */
  public void processErrorResponse(DdfResponseException errorRsp);

  ////////////////////////////////////////////////////////////////
  // Niagara AX Type
  ////////////////////////////////////////////////////////////////
  public static final Type TYPE = Sys.loadType(BIDdfCustomRequest.class);
}
