/*
 * @copyright 2005 Tridium Inc.
 */
package com.tridium.ddf.comm.defaultComm;

import java.util.Hashtable;

import javax.baja.sys.Action;
import javax.baja.sys.BComplex;
import javax.baja.sys.BComponent;
import javax.baja.sys.BRelTime;
import javax.baja.sys.BString;
import javax.baja.sys.BValue;
import javax.baja.sys.Context;
import javax.baja.sys.Flags;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;
import javax.baja.sys.Clock.Ticket;
import javax.baja.util.Queue;

import com.tridium.ddf.clock.BDdfScheduler;
import com.tridium.ddf.comm.BIDdfCommunicator;
import com.tridium.ddf.comm.BIDdfTransactionMgr;
import com.tridium.ddf.comm.BIDdfUnsolicitedMgr;
import com.tridium.ddf.comm.IDdfDataFrame;
import com.tridium.ddf.comm.req.BDdfRawTransmitRequest;
import com.tridium.ddf.comm.req.BIDdfRequest;
import com.tridium.ddf.comm.rsp.BIDdfMultiFrameResponse;
import com.tridium.ddf.comm.rsp.BIDdfResponse;
import com.tridium.ddf.comm.rsp.IDdfTransmitAckResponse;

/**
 * BDdfTransactionMgr - Goes with BDdfCommunicator, BDdfTransmitter, and BDdfReceiver
 *
 * Instead of extending this class directly, there are two reasonable subclasses that might
 * be a better choice: BDdfMultipleTransactionMgr and BDdfSingleTransactionMgr
 *
 * @author    lperkins
 * @creation  Oct 16, 2006
 * @version   $Revision$ $Date$
 * @since     Niagara 3.2
 */
public abstract class BDdfTransactionMgr
  extends BComponent
  implements BIDdfTransactionMgr
{
  /*-
  class BDdfTransactionMgr
  {
    actions
    {
      checkOutstandingTimeout(ddfRequest:BValue)
        -- Internal use only.
        flags{hidden}
        default{[BString.make("OverridePlease")]}
    }
  }
  -*/
/*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
/*@ $com.tridium.ddf.comm.defaultComm.BDdfTransactionMgr(252137309)1.0$ @*/
/* Generated Tue Dec 11 13:23:07 EST 2007 by Slot-o-Matic 2000 (c) Tridium, Inc. 2000 */

////////////////////////////////////////////////////////////////
// Action "checkOutstandingTimeout"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>checkOutstandingTimeout</code> action.
   * Internal use only.
   * @see com.tridium.ddf.comm.defaultComm.BDdfTransactionMgr#checkOutstandingTimeout()
   */
  public static final Action checkOutstandingTimeout = newAction(Flags.HIDDEN,BString.make("OverridePlease"),null);
  

////////////////////////////////////////////////////////////////
// Type
////////////////////////////////////////////////////////////////
  
  public Type getType() { return TYPE; }
  public static final Type TYPE = Sys.loadType(BDdfTransactionMgr.class);

/*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

  /**
   * Invoke the <code>checkOutstandingTimeout</code> action.
   * @see com.tridium.ddf.comm.defaultComm.BDdfTransactionMgr#checkOutstandingTimeout
   * 
   * NOTE: This forces the action to be called locally, without any network dispatching, to allow
   * for proxy-side communication.
   */
  public void checkOutstandingTimeout(BValue ddfRequest) { invoke(checkOutstandingTimeout,ddfRequest,Context.commit); }
  
////////////////////////////////////////////////////////////////
// BComponent
////////////////////////////////////////////////////////////////

  /**
   * Although this Niagara AX callback is declared as 'final', 
   * descendants may override the 'transactionMgrStarted' method. 
   */
  public final void started() throws Exception
  {
    transactionMgrStarted();
    super.started();
  }

  /**
   * Although this Niagara AX callback is declared as 'final', 
   * descendants may override the 'transactionMgrStopped' method. 
   */
  public final void stopped() throws Exception
  {
    transactionMgrStopped();
  }
  
////////////////////////////////////////////////////////////////
// BIDdfTransactionMgr
////////////////////////////////////////////////////////////////
  
  /**
   * The communicator calls this method immediately prior to transmitting to allow
   * the transaction manager to prepare to receive the response or timeout.
   *
   * @param BIDdfRequest the request that the communicator is about to transmit
   */
  public void processTransaction(BIDdfRequest req)
  {
    try
    {
      if (getDdfCommunicator().getLog().isTraceOn())
        getDdfCommunicator().getLog().trace(
            DdfDefaultCommLexicon.beginTransaction(req));
      
      beginTransaction(req);
      
      // If the response timeout is zero, then we don't wait for a response.
      // NOTE: The BDdfCommunicator.doCommunicate method will call 'notifyAll'
      // on the request in this scenario, just after the request is transmitted
      if (req.getResponseTimeout().equals(BRelTime.DEFAULT))
      {
        if (getDdfCommunicator().getLog().isTraceOn())
          getDdfCommunicator().getLog().trace(DdfDefaultCommLexicon.sendOnlyTransactionNotification+": "+req.toString());
      }
      else
      {
        scheduleToCheckForTimeout(req);
      }
    }
    catch (Exception e)
    {
      if (getDdfCommunicator().getLog().isTraceOn())
        getDdfCommunicator().getLog().trace("TransactionError",e);
    }
  }
  
  /**
   * @return by default, the Niagara AX Nav parent cast to BIDdfCommunicator
   */
  public BIDdfCommunicator getDdfCommunicator()
  {
    if (ddfCommunicator == null)
      ddfCommunicator = (BIDdfCommunicator)getParent();
    return ddfCommunicator;
  }  
  
  /**
   * Starts the transaction processor and receive threads. The
   * transaction manger owns the references to each of these.
   */
  public void startTransactionMgr()
  {
    if (stopped)
    {
      stopped=false;
      transactionProcessor = new TransactionProcessor();
      transactionProcessor.start();

      receiveThread = new ReceiveThread();
      receiveThread.start();
      getDdfCommunicator().getDdfReceiver().startReceiver();
    }
  }
  
  /**
   * Stops the transaction processor and receive threads. The
   * transaction manger owns the references to each of these.
   */
  public void stopTransactionMgr()
  {
    if (!stopped)
    {
      stopped=true;
      if (receiveThread!=null)
      {
        getDdfCommunicator().getDdfReceiver().stopReceiver();
        receiveThread.stopReceiving();
        receiveThread.interrupt();
        try
        { 
          // Waits for the receiveThread to stop
          receiveThread.join(1000);
        }
        catch (InterruptedException ie)
        {
          
        }
      }
      if (transactionProcessor!=null)
      {
        transactionProcessor.interrupt();
        transactionProcessor.stopProcessing();
        try
        {
          // Waits for the transactionProcessor to stop
          transactionProcessor.join(1000);
        }
        catch (InterruptedException ie)
        {
          
        }
      }
      ddfCommunicator=null;
    }
    
    // If the receive thread or transaction processor thread is alive, then let's
    // try a last ditch effort to kill them. They could still be alive if the developer's
    // logic catches the interrupted exception and never returns back to the corrsponding
    // thread's run method in this java file.
    if ((receiveThread!=null && receiveThread.isAlive()) ||
        (transactionProcessor!=null && transactionProcessor.isAlive()))
    {
      // Spawns a thread that continually interrupts the receiveThread and the transactionProcessor
      new ReceiverAndTransactionMgrKiller(receiveThread, transactionProcessor).start();
    }
  }
  
////////////////////////////////////////////////////////////////
// BDdfTransactionMgr
////////////////////////////////////////////////////////////////
  
  /**
   * This method is called immediately before the given request is serialized
   * and transmitted. The descendant should make any preparations that are
   * necessary for it to determine whether a timeout occurs (from the
   * descendant's "doCheckOutstandingTimeout" method; or for it to
   * match received frames to the request (from the descendant's  frameReceived
   * method)
   *
   * Please note that there are two subclasses of this class that should be suitable
   * for all drivers. Therefore, Tridium's development team hopes that this class
   * should ever need to be extended anywhere else.
   */
  protected abstract void beginTransaction(BIDdfRequest req) throws Exception;
  
  /**
   * This method is called when the response timeout interval for the given
   * ddfRequest expires. The descendant class needs to verify that the given
   * request has not received its response and implement a retry mechanism.
   *
   * The ddfRequest has a property on it that indicates the number of remaining
   * retries. When retrying, the descendant should call forceTransmit on
   * the ddf communicator, pass the request, and decrement the value for the
   * property that indicates the number of remaining retires. If no retries
   * remain then the descendant should process the failure.
   *
   * Please note that there are two subclasses of this class that should suitable
   * for all drivers. Therefore, Tridium's development team hopes that this class
   * should ever need to be extended anywhere else.
   *
   * @param ddfRequest
   */
  protected abstract void doCheckOutstandingTimeout(BIDdfRequest ddfRequest);
  
  /**
   * This method is called on the transaction processor thread for all frames
   * that the ReceiveThread receives. The descendant needs to match up the frame
   * with any outstanding requests and perform all response processing. If the
   * frame does not match up then the descendant needs to pass the frame to
   * the unsolicited manager.
   *
   * Please note that there are two subclasses of this class that should be suitable
   * for all drivers. Therefore, Tridium's development team hopes that this class
   * should never need to be extended anywhere else.
   */
  protected abstract void frameReceived(IDdfDataFrame ddfReceiveFrame);
  
  /**
   * This is equivalent to the Niagara AX 'started' callback but
   * is available for descendants.
   */
  protected void transactionMgrStarted()
  {
  }
  
  /**
   * This is equivalent to the Niagara AX 'stopped' callback but
   * is available for descendants.
   */
  protected void transactionMgrStopped()
  {
  }

  
  /**
   * This method is called when the communicator's 'Reset Statistics'
   * action is invoked
   */
  public void resetStatistics()
  {
  }

  /**
   * Implements the action 'checkOutstandingTimeout'.
   *
   * NOTE: The method 'scheduleToCheckForTimeout' is called when a request
   * is transmitted and does a BDdfScheduler.INSTANCE.schedule call, that expires
   * when the request's timeout interval occurs - then calls this method
   * and passes in the request.
   *
   * This method casts the given arg to a BIDdfRequest and places it on the
   * transaction processor thread where in a matter of moments, it will be
   * processed further (on another thread), without any chance of backing up the BDdfScheduler
   * thread (on which BDdfScheduler.INSTANCE.schedule relies).
   *
   * @param arg - The arg that was passed to BDdfScheduler.INSTANCE.schedule in the method
   * scheduleToCheckForTimeout. This is the the BIDdfRequest.
   */
  public final void doCheckOutstandingTimeout(BValue arg)
  {
    transactionProcessor.enqueue((BIDdfRequest)arg);
  }
  
  /**
   * Arranges for the doCheckOutstandingTimeout method to be called and
   * passed the given req after the req's responseTimeout interval expires.
   * 
   * @param req
   * 
   * @throws DdfNoResponseExpectedException in the even that req.getResponseTimeout().getMillis() returns 0 -- that
   * would cause the req to not be checked for timeout. In which case, the transaction will
   * need to be cleaned up.
   */
  protected void scheduleToCheckForTimeout(BIDdfRequest req)
  {
    // Arranges for the checkOutstandingTimeout method to be called after the response timeout
    Ticket requestTicket =
      BDdfScheduler.INSTANCE.schedule(this, req.getResponseTimeout(), checkOutstandingTimeout, (BValue)req);
    requestTickets.put(req,requestTicket);
  }

  /**
   * This method is called when a data frame is received that does not match up
   * to a request that was transmitted.
   *  
   * @param ddfReceiveFrame a receive frame that does not match up to a reqeuest
   * that was transmitted. This receive frame is passed to the driver's
   * unsolicited manager.
   */
  protected void routeToUnsolicited(IDdfDataFrame ddfReceiveFrame)
  {
    BIDdfUnsolicitedMgr unsolicitedMgr = getDdfCommunicator().getDdfUnsolicitedMgr();
    if (unsolicitedMgr!=null)
      unsolicitedMgr.enqueueUnsolicitedFrame(ddfReceiveFrame);
  }

  /**
   * Descendants should consider calling this method only if a
   * BIDdfMultiFrameResponse is received for a BIDdfRequest
   * and the response is not yet complete. This resets the
   * timeout check and causes the framework to wait for a fresh
   * interval of the given request.getResponseTimeout before
   * retrying the transaction or possibly timing out.
   *
   * @param req a request whose response is a BIDdfMultiFrameResponse
   */
  protected void reScheduleToCheckForTimeout(BIDdfRequest req)
  { 
    // Locks the given req so that it cannot be rescheduled for
    // A fresh timeout interval while the doCheckOutstandingMethod is
    // Processing a timeout. This allows the doCheckOutstandingMethod to
    // Always know whether or not the request was rescheduled. It is most
    // Ideal, however, if the descendant locks all response processing
    // On the request.
    synchronized(req)
    {
      // Arranges for the checkOutstandingTimeout method to be called after the response timeout
      if (req.getResponseTimeout().getMillis()>=0)
      {
        // Gets the existing ticket for the request
        Ticket requestTicket = requestTickets.get(req);
        if (requestTicket!=null)
          requestTicket.cancel();
        requestTicket =
          BDdfScheduler.INSTANCE.schedule(this, req.getResponseTimeout(), checkOutstandingTimeout, (BValue)req);
        requestTickets.put(req,requestTicket);
      }
    }
  }

  /**
   * This method is called by the receive processing logic to
   * determine if a BIDdfFrameResponse is "complete". All
   * BIDdfFrameResponse's are always considered "complete" unless
   * they are an instance of BIDdfMultiFrameResponse. In that case,
   * BIMultiFrameResponse's are only considered complete if their
   * isComplete method returns true.
   *
   * @param ddfResponse the BIDdfResponse that was just returned
   * by the processReceive method of a BIDdfRequest.
   *
   * @return Please see the method's description.
   */
  protected boolean isCompletedResponse(BIDdfResponse ddfResponse)
  {
    if (ddfResponse instanceof BIDdfMultiFrameResponse)
    {
      return ((BIDdfMultiFrameResponse)ddfResponse).isComplete();
    }
    else
      return true;
  }

  /**
   * By default, the parent must be an instance of BIDdfCommunicator.
   */
  public boolean isParentLegal(BComponent parent)
  {
    return parent instanceof BIDdfCommunicator;
  }


  /**
   * This method handles a very special case when a request receives a response
   * and wishes to transmit a message back to the field-device as a result.
   *
   * @param xmitAckRsp the BIDdfTransmitAckResponse the result of whose getDdfAckBytes
   * method will be immediately transmitted onto the fieldbus.
   */
  protected void transmitRspAckBytes(final IDdfTransmitAckResponse xmitAckRsp)
  {
    try
    { 
      // Gets the ack bytes to be immediately transmitted
      byte[] xmitAckBytes = xmitAckRsp.getBytes();
      
      if (xmitAckBytes!=null)
      { 
        // Prints some trace
        if (getDdfCommunicator().getLog().isTraceOn())
          getDdfCommunicator().getLog().trace(DdfDefaultCommLexicon.ackReplyTransmit(xmitAckRsp));
        
        // Force-transmits a BDdfRawTransmitRequest that wraps the bytes. Please note that this
        // Calls "ddfForceTransmit" directly on the transmitter. It does not call "communicate" on the
        // Communicator. This bypasses tranaction handling for the ddf force transmit. This is what we
        // Want. The ddf ack does not get a response. Any response frame can be passed back to the
        // Original request provided that the given xmitAckRsp also implements BIDdfMultiFrameResponse
        getDdfCommunicator().getDdfTransmitter().forceTransmit(
            new BDdfRawTransmitRequest(xmitAckBytes));
      }
    }
    catch (Exception e)
    {
      getDdfCommunicator().getLog().error(DdfDefaultCommLexicon.ackReplyTransmitError(xmitAckRsp),e);
    }
  }

  /**
   * Gets the suffix that is placed onto the end of the ReceiveThread
   * and TransactionProcessor threads.
   *
   * @return hopefully an underscore plus the name of the ddf communicator's
   * Niagara AX parent.
   */
  protected String getThreadSuffix()
  {
    BIDdfCommunicator ddfCommunicator = getDdfCommunicator();
    
    if (ddfCommunicator instanceof BDdfCommunicator)
    {
      return ((BDdfCommunicator)ddfCommunicator).getWorkerThreadName();
    }
    else
      return "";

  }
  
////////////////////////////////////////////////////////////////
// ReceiverAndTransactionMgrKiller
////////////////////////////////////////////////////////////////

  /**
   * This thread is started when the transaction mgr needs to shut down.
   * It continually calls the 'interrupt' method on the receive thread
   * and on the transaction mgr. This is necessary in case the developer's
   * implementation of these threads is stuck in tight loop that blocks
   * to read from the field-bus.
   * 
   * @author lperkins
   */
  private class ReceiverAndTransactionMgrKiller
    extends Thread
  {
    ReceiveThread receiveThread;
    TransactionProcessor transactionProcessor;
    
    ReceiverAndTransactionMgrKiller(ReceiveThread receiveThread, TransactionProcessor transactionProcessor)
    {
      this.receiveThread=receiveThread;
      this.transactionProcessor=transactionProcessor;
    }
    public void run()
    {
      while ( (receiveThread!=null && receiveThread.isRunning) || (transactionProcessor!=null && transactionProcessor.isRunning) )
      {
        try
        {
          Thread.sleep(100);
        }
        catch (InterruptedException ie)
        {
         
        }
        
        if (receiveThread!=null && receiveThread.isRunning)
          receiveThread.interrupt();
        if (transactionProcessor!=null && transactionProcessor.isRunning)
          transactionProcessor.interrupt();
      }
    }
  }

////////////////////////////////////////////////////////////////
// ReceiveThread
////////////////////////////////////////////////////////////////
  private class ReceiveThread
    extends Thread
  {
    ReceiveThread()
    {

      super("Receive:"+getThreadSuffix());
      isReceiveThreadStopped=false;
      isRunning=false;
    }
    
    void stopReceiving()
    {
      isReceiveThreadStopped=true;
    }
    
    /**
     * This method is called on its own thread.
     */
    public void run()
    {
      try
      {
        isRunning=true;
        
        while (!isReceiveThreadStopped)
        {
          try
          { 
            // Asks the receiver to receive a frame
            IDdfDataFrame receivedFrame = getDdfCommunicator().getDdfReceiver().receiveFrame();
            
            if (receivedFrame!=null)
            {
              // Enqueues a copy of the received frame for processing on the TransactionProcessor thread.
              transactionProcessor.enqueue(receivedFrame.getFrameCopy());
            }
          }
          catch (Throwable t)
          { 
            // Logs the exception
            if (!isReceiveThreadStopped)
            {
              getDdfCommunicator().getLog().error(t.toString(),t);
              try{Thread.sleep(100);}catch(Exception ee){} // Prevents the thread from spinning.
            }
          }
        }
        getDdfCommunicator().getLog().message("Receive thread successfully stopped");
      }
      finally
      {
        isRunning=false;
      }
    }
    
    boolean isReceiveThreadStopped = false;
    boolean isRunning=false;
  }
  
////////////////////////////////////////////////////////////////
// TransactionProcessor
////////////////////////////////////////////////////////////////
  
  /**
   * The run method of this class processes received frames and
   * time outs for outstanding requests.
   *
   * The doCheckOutstandingTimeout(BValue) method casts its argument
   * to a BIDdfRequest and passes it to the enqueue method of an
   * instance of this class.
   *
   * The run method of the instance of the inner, Receiver class calls
   * this method and passes all received frames.
   *
   * @author lperkins
   */
  private class TransactionProcessor
    extends Thread
  {
    TransactionProcessor()
    {
      super("Transaction:"+getThreadSuffix());
      isTransactionProcessorStopped=false;
    }
    /**
     * The ReceiveThread passes a copy of all received frames here for
     * processing.
     * @param receivedFrame
     */
    void enqueue(IDdfDataFrame receivedFrame)
    {
      itemsToProcess.enqueue(receivedFrame);
    }
    /**
     * The doCheckOutstandingTimeout passes all BIDdfRequests here for processing
     * of request timeouts or retries.
     */
    void enqueue(BIDdfRequest checkOutstandingTimeoutArg)
    {
      itemsToProcess.enqueue(checkOutstandingTimeoutArg);
    }

    private void process(IDdfDataFrame processReceivedFrame)
    {
      frameReceived(processReceivedFrame);
    }

    private void process(BIDdfRequest processRequestCheckForTimeout)
    { 
      // Gets the Ticket that was created back when the 'scheduleToCheckForTimeout'
      // Called BDdfScheduler.INSTANCE.schedule for the same request that this method is now processing
      Ticket requestTicket = requestTickets.get(processRequestCheckForTimeout);
      
      // This if statement was added to allow proxy-client-side communications
      if (requestTicket == null)
      {
        doCheckOutstandingTimeout(processRequestCheckForTimeout);
      }
      // Verifies that the request's Ticket is expired (and isn't an instance of another
      // Ticket that has been rescheduled after the call to BDdfScheduler.INSTANCE.schedule). This could
      // Happen as a result of receiving a frame for a BIDdfMultiFrameResponse;
      else if (requestTicket.isExpired())
      {
        requestTickets.remove(processRequestCheckForTimeout); // Removes the request's ticket from our ticket reference table
        doCheckOutstandingTimeout(processRequestCheckForTimeout);
      }
    }
    
    void stopProcessing()
    {
      isTransactionProcessorStopped=true;
    }
    
    /**
     * This method is called on its own thread.
     */
    public void run()
    {
      try
      {
        isRunning=true;
        
        while (!isTransactionProcessorStopped)
        {
          try
          {
            // Pulls an item off of the 'itemsToProcess' queue. Blocks if necessary
            // until there is an item in the queue to process
            Object nextItemToProcess = itemsToProcess.dequeue(-1);
            
            if (nextItemToProcess instanceof IDdfDataFrame)     // Processes received frames
              process((IDdfDataFrame)nextItemToProcess);
            else if (nextItemToProcess instanceof BIDdfRequest) // Processes request timeouts
              process((BIDdfRequest)nextItemToProcess);
          }
          catch (InterruptedException ie)
          {
            // Does nothing but allows the while(!stopped) statement to be re-evaluated
          }
          catch (Throwable t)
          { 
            // Logs the exception
            if (!isTransactionProcessorStopped)
            {
              getDdfCommunicator().getLog().error(t.toString(),t);
              try{Thread.sleep(100);}catch(Exception ee){} // Prevents the thread from spinning.
            }
          }
          
        }
        getDdfCommunicator().getLog().message("Transaction mgr thread successfully stopped");
        
      }
      finally
      {
        isRunning=false;
      }
    }
    
    boolean isRunning=false;
    boolean isTransactionProcessorStopped=false;
    Queue itemsToProcess = new Queue();
    
  }
  
////////////////////////////////////////////////////////////////
// Attributes
////////////////////////////////////////////////////////////////
  
  /*
   * A cached communicator is necessary to permit client-side communications, which is required
   * for the video driver.
   */
  private BIDdfCommunicator ddfCommunicator;

  // The receiveThread and transactionProcessor check this through each loop iteration. This is
  // initialized to false in the 'startTransactionMgr' method.
  boolean stopped=true; 
  
  /**
   * This is the thread that receives incoming data for the driver. This thread
   * makes callbacks into the driver's BDdfReceiver to parse data frames out of
   * the raw data that is received from the field-bus.
   */
  ReceiveThread receiveThread;
  
  /**
   * This is a thread and a queue that processes transactions. The callbacks to
   * BIDdfRequests that occur when the request times out or receives its reply
   * are made on this thread. This thread also performs retries.
   */
  TransactionProcessor transactionProcessor;
  
  /**
   * This maps all outstanding requests to their corresponding Ticket that they
   * were assigned as a result of calling BDdfScheduler.schedule. This is part
   * of the mechanism that is used to process a timeout.
   */
  Hashtable<BIDdfRequest, Ticket> requestTickets = new Hashtable<>();
}
