/*
 * @copyright 2005 Tridium Inc.
 */
package com.tridium.ddf.comm.defaultComm;

import java.util.Hashtable;
import java.util.Iterator;
import java.util.Vector;

import javax.baja.log.Log;
import javax.baja.space.BComponentSpace;
import javax.baja.spy.SpyWriter;
import javax.baja.sys.Action;
import javax.baja.sys.BComplex;
import javax.baja.sys.BComponent;
import javax.baja.sys.BajaRuntimeException;
import javax.baja.sys.Flags;
import javax.baja.sys.Property;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;
import javax.baja.sys.Clock.Ticket;
import javax.baja.nre.util.Array;
import javax.baja.util.BTypeSpec;
import javax.baja.util.BWorker;
import javax.baja.util.Queue;
import javax.baja.util.Worker;

import com.tridium.ddf.clock.BDdfScheduler;
import com.tridium.ddf.comm.BIDdfCommunicating;
import com.tridium.ddf.comm.BIDdfCommunicator;
import com.tridium.ddf.comm.BIDdfReceiver;
import com.tridium.ddf.comm.BIDdfTransactionMgr;
import com.tridium.ddf.comm.BIDdfTransmitter;
import com.tridium.ddf.comm.BIDdfUnsolicitedMgr;
import com.tridium.ddf.comm.req.BDdfRawTransmitRequest;
import com.tridium.ddf.comm.req.BIDdfRequest;
import com.tridium.ddf.comm.req.BIDdfWriteRequest;
import com.tridium.ddf.comm.req.IDdfWritable;
import com.tridium.ddf.poll.BDdfPollScheduler;
import com.tridium.ddf.poll.BIDdfPollScheduler;

/**
 * BDdfCommunicator - Handles transmitting and receiving messages on behalf of a component
 * that implements BIDdfCommunicating such as BDdfCommDevice or BDdfCommNetwork.
 *
 * This is a raw implementation of BDdfCommunicator. It immediately places BIDdfRequests on
 * a worker queue and pulls them off, one-by-one (on a dedicated thread), and passes them to
 * the transaction manager (for timeout/retry scheduling and response matching) and then to
 * the transmitter for the first transmission attempt.
 *
 * The 'devSerialDriver' and 'devIpDriver' modules provide nearly fully-functional default
 * implementations of BDdfCommunicator for use in drivers.
 *
 * @see com.tridium.ddf.comm.multipleTransaction.BDdfMultipleTransactionCommunicator
 * @see com.tridium.ddf.comm.singleTransaction.BDdfSingleTransactionCommunicator
 *
 * @author    lperkins
 * @creation  Oct 9, 2006
 * @version   $Revision$ $Date$
 * @since     Niagara 3.0
 */
public class BDdfCommunicator
  extends BWorker
  implements BIDdfCommunicator
{
  /*-
   class BDdfCommunicator
   {
     properties
     {
       transmitter : BDdfTransmitter
         -- This item transmits messages for the communicator.
         default{[ new BDdfNullTransmitter() ]}
       receiver : BDdfReceiver
         -- This is a place-holder for the receiver.
         default{[ new BDdfNullReceiver() ]}
       transactionManager : BDdfTransactionMgr
         -- This is the mechanism that processes outstanding transactions
         default{[ new BDdfNullTransactionMgr() ]}
       pollScheduler : BDdfPollScheduler
         -- Most of the time communicators will require a poll scheduler.
         default{[ new BDdfPollScheduler()]}
       unsolicitedMgr : BDdfUnsolicitedMgr
         -- The simplest of drivers will not need unsolicited support
         default{[new BDdfNullUnsolicitedMgr()]}
     }
     actions
     {
       resetStatistics()
         -- Resets the various statistics that are on the transmitter
         -- and the receiver
        flags { confirmRequired }
       requeue()
         flags{hidden}
         -- This action is used internally.
     }
   }
   -*/
/*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
/*@ $com.tridium.ddf.comm.defaultComm.BDdfCommunicator(357627002)1.0$ @*/
/* Generated Thu Oct 25 11:30:22 EDT 2007 by Slot-o-Matic 2000 (c) Tridium, Inc. 2000 */

////////////////////////////////////////////////////////////////
// Property "transmitter"
////////////////////////////////////////////////////////////////

  /**
   * Slot for the <code>transmitter</code> property.
   * This item transmits messages for the communicator.
   * @see com.tridium.ddf.comm.defaultComm.BDdfCommunicator#getTransmitter
   * @see com.tridium.ddf.comm.defaultComm.BDdfCommunicator#setTransmitter
   */
  public static final Property transmitter = newProperty(0, new BDdfNullTransmitter(),null);

  /**
   * Get the <code>transmitter</code> property.
   * @see com.tridium.ddf.comm.defaultComm.BDdfCommunicator#transmitter
   */
  public BDdfTransmitter getTransmitter() { return (BDdfTransmitter)get(transmitter); }

  /**
   * Set the <code>transmitter</code> property.
   * @see com.tridium.ddf.comm.defaultComm.BDdfCommunicator#transmitter
   */
  public void setTransmitter(BDdfTransmitter v) { set(transmitter,v,null); }

////////////////////////////////////////////////////////////////
// Property "receiver"
////////////////////////////////////////////////////////////////

  /**
   * Slot for the <code>receiver</code> property.
   * This is a place-holder for the receiver.
   * @see com.tridium.ddf.comm.defaultComm.BDdfCommunicator#getReceiver
   * @see com.tridium.ddf.comm.defaultComm.BDdfCommunicator#setReceiver
   */
  public static final Property receiver = newProperty(0, new BDdfNullReceiver(),null);

  /**
   * Get the <code>receiver</code> property.
   * @see com.tridium.ddf.comm.defaultComm.BDdfCommunicator#receiver
   */
  public BDdfReceiver getReceiver() { return (BDdfReceiver)get(receiver); }

  /**
   * Set the <code>receiver</code> property.
   * @see com.tridium.ddf.comm.defaultComm.BDdfCommunicator#receiver
   */
  public void setReceiver(BDdfReceiver v) { set(receiver,v,null); }

////////////////////////////////////////////////////////////////
// Property "transactionManager"
////////////////////////////////////////////////////////////////

  /**
   * Slot for the <code>transactionManager</code> property.
   * This is the mechanism that processes outstanding transactions
   * @see com.tridium.ddf.comm.defaultComm.BDdfCommunicator#getTransactionManager
   * @see com.tridium.ddf.comm.defaultComm.BDdfCommunicator#setTransactionManager
   */
  public static final Property transactionManager = newProperty(0, new BDdfNullTransactionMgr(),null);

  /**
   * Get the <code>transactionManager</code> property.
   * @see com.tridium.ddf.comm.defaultComm.BDdfCommunicator#transactionManager
   */
  public BDdfTransactionMgr getTransactionManager() { return (BDdfTransactionMgr)get(transactionManager); }

  /**
   * Set the <code>transactionManager</code> property.
   * @see com.tridium.ddf.comm.defaultComm.BDdfCommunicator#transactionManager
   */
  public void setTransactionManager(BDdfTransactionMgr v) { set(transactionManager,v,null); }

////////////////////////////////////////////////////////////////
// Property "pollScheduler"
////////////////////////////////////////////////////////////////

  /**
   * Slot for the <code>pollScheduler</code> property.
   * Most of the time communicators will require a poll
   * scheduler.
   * @see com.tridium.ddf.comm.defaultComm.BDdfCommunicator#getPollScheduler
   * @see com.tridium.ddf.comm.defaultComm.BDdfCommunicator#setPollScheduler
   */
  public static final Property pollScheduler = newProperty(0, new BDdfPollScheduler(),null);

  /**
   * Get the <code>pollScheduler</code> property.
   * @see com.tridium.ddf.comm.defaultComm.BDdfCommunicator#pollScheduler
   */
  public BDdfPollScheduler getPollScheduler() { return (BDdfPollScheduler)get(pollScheduler); }

  /**
   * Set the <code>pollScheduler</code> property.
   * @see com.tridium.ddf.comm.defaultComm.BDdfCommunicator#pollScheduler
   */
  public void setPollScheduler(BDdfPollScheduler v) { set(pollScheduler,v,null); }

////////////////////////////////////////////////////////////////
// Property "unsolicitedMgr"
////////////////////////////////////////////////////////////////

  /**
   * Slot for the <code>unsolicitedMgr</code> property.
   * The simplest of drivers will not need unsolicited support
   * @see com.tridium.ddf.comm.defaultComm.BDdfCommunicator#getUnsolicitedMgr
   * @see com.tridium.ddf.comm.defaultComm.BDdfCommunicator#setUnsolicitedMgr
   */
  public static final Property unsolicitedMgr = newProperty(0, new BDdfNullUnsolicitedMgr(),null);

  /**
   * Get the <code>unsolicitedMgr</code> property.
   * @see com.tridium.ddf.comm.defaultComm.BDdfCommunicator#unsolicitedMgr
   */
  public BDdfUnsolicitedMgr getUnsolicitedMgr() { return (BDdfUnsolicitedMgr)get(unsolicitedMgr); }

  /**
   * Set the <code>unsolicitedMgr</code> property.
   * @see com.tridium.ddf.comm.defaultComm.BDdfCommunicator#unsolicitedMgr
   */
  public void setUnsolicitedMgr(BDdfUnsolicitedMgr v) { set(unsolicitedMgr,v,null); }

////////////////////////////////////////////////////////////////
// Action "resetStatistics"
////////////////////////////////////////////////////////////////

  /**
   * Slot for the <code>resetStatistics</code> action.
   * Resets the various statistics that are on the transmitter
   * and the receiver
   * @see com.tridium.ddf.comm.defaultComm.BDdfCommunicator#resetStatistics()
   */
  public static final Action resetStatistics = newAction(Flags.CONFIRM_REQUIRED,null);

  /**
   * Invoke the <code>resetStatistics</code> action.
   * @see com.tridium.ddf.comm.defaultComm.BDdfCommunicator#resetStatistics
   */
  public void resetStatistics() { invoke(resetStatistics,null,null); }

////////////////////////////////////////////////////////////////
// Action "requeue"
////////////////////////////////////////////////////////////////

  /**
   * Slot for the <code>requeue</code> action.
   * @see com.tridium.ddf.comm.defaultComm.BDdfCommunicator#requeue()
   */
  public static final Action requeue = newAction(Flags.HIDDEN,null);

  /**
   * Invoke the <code>requeue</code> action.
   * @see com.tridium.ddf.comm.defaultComm.BDdfCommunicator#requeue
   */
  public void requeue() { invoke(requeue,null,null); }

////////////////////////////////////////////////////////////////
// Type
////////////////////////////////////////////////////////////////

  public Type getType() { return TYPE; }
  public static final Type TYPE = Sys.loadType(BDdfCommunicator.class);

/*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

////////////////////////////////////////////////////////////////
// BComplex
////////////////////////////////////////////////////////////////
  /**
   * This must be under a BIDdfCommunicating in the navigation tree of the Niagara AX station.
   */
  public boolean isParentLegal(BComponent parent)
  {
    return parent instanceof BIDdfCommunicating;
  }

////////////////////////////////////////////////////////////////
// BComponent
////////////////////////////////////////////////////////////////
  private void loadClientSideSupport()
  {
    // Synchronizes the communicator for limited, direct, client-side
    // communications
    loadForClientComm(this, "UnableToSyncProxySpace - communicator");

    // Gives the transaction manager a chance to cache this as its communicator
    getDdfTransactionMgr().getDdfCommunicator();

    // Gives the transmitter a chance to cache this as its communicator
    getDdfTransmitter().getDdfCommunicator();

    // Gives the receiver a chance to cache this as its communicator
    getDdfReceiver().getDdfCommunicator();

    // Gives the unsolicited mgr a chance to cache this as its communicator
    getUnsolicitedMgr().getDdfCommunicator();

    // Caches the communicating parent now while it is hopefully available (it
    // can become null on the client-side)
    findCommunicatingParent();
  }

  /**
   * This prevents drivers that extend the ddf from accidentally overridding
   * the started method and forgetting to call super.started().
   */
  public final void started() throws Exception
  {
    loadClientSideSupport();

    // This has to be done before calling super.started because super.started
    // starts the communicator's worker, whose thread attempts to access the
    // 'communicatingParent'...we wouldn't want this to happen before the
    // 'communicatingParent' was initialized
    findCommunicatingParent();

    super.started();
    communicatorStarted();
  }



  /**
   * Any descendants that override this method should call super.descendantsStarted
   * since this calls the 'startCommunicating' method.
   */
  public void descendantsStarted() throws Exception
  {
    startCommunicating();
    super.descendantsStarted();
  }

  /* (non-Javadoc)
   * @see javax.baja.sys.BComponent#stopped()
   */
  public final void stopped() throws Exception
  {
    stopCommunicating();
    communicatorStopped();
    super.stopped();
  }

  /**
   * Keeps the communicator from showing up in the Niagara AX navigation tree (usually on the left side of Workbench)
   */
  public boolean isNavChild()
  {
    return false;
  }

////////////////////////////////////////////////////////////////
// BWorker
////////////////////////////////////////////////////////////////

  /**
   * Starts the communicator's worker thread, provided that the closest Nav ancestor network
   * or device is neither in fatalFault nor disabled.
   */
  protected void startWorker()
  {
    if (isCommEnabled())
    {
      super.startWorker();
    }
  }

  /**
   * Get the thread name used to start the worker.
   */
  protected String getWorkerThreadName()
  {
    String threadName= "Communicator";

    Type t = getType();
    if (t!=null)
    {
      BTypeSpec ts = t.getTypeSpec();

      if (ts!=null)
      {
        String tn = ts.getTypeName();

        if (tn!=null)
        {
          // Reassigns thread name to be the real class name instead of 'Communicator'
          threadName = tn;
        }
      }
    }

    Object h = getHandle();

    if (h!=null)
    {
      threadName += ":h" + h;
    }

    return threadName;
  }

  /**
   * This prevents drivers that extend the ddf from accidentally overriding
   * the stopWorker method and forgetting to call super.stopWorker().
   */
  protected final void stopWorker()
  {
    super.stopWorker();
  }

  /**
   * This satisfies BWorker's abstract 'getWorker' requirement.
   *
   *  @return a reference to the internal 'communicationWorker'
   */
  public Worker getWorker()
  {
    return  communicationWorker;
  }

////////////////////////////////////////////////////////////////
// Spy
////////////////////////////////////////////////////////////////
  public void spy(SpyWriter out) throws Exception
  {
    out.startProps("Ddf Communicator");
    out.endProps();
    getReceiver().spy(out);
    super.spy(out);
  }

////////////////////////////////////////////////////////////////
// BIDdfCommunicator
////////////////////////////////////////////////////////////////
  /**
   * Returns the current value of the 'transmitter' property cast as a BIDdfTransmitter.
   */
  public BIDdfTransmitter getDdfTransmitter()
  {
    if (ddfTransmitter==null)
    {
      // Caches the ddfTransmitter. This is important for client-side
      // video stream communications. Otherwise, this object's getParent()
      // method will eventually return null as the Niagara AX core will
      // eventually attempt to cleanup its proxy
      ddfTransmitter = (BIDdfTransmitter)getTransmitter();

      // Ensures that the ddfTransmitter is leased on the client-side
      loadForClientComm(ddfTransmitter, "Unable to Sync Transmitter");
    }

    // Returns the cached ddfTransmitter reference
    return ddfTransmitter;
  }

  /**
   * Returns the current value of the 'receiver' property cast as a BIDdfReceiver.
   */
  public BIDdfReceiver getDdfReceiver()
  {
    if (ddfReceiver==null)
    {
      // Caches the ddfReceiver. This is important for client-side
      // video stream communications. Otherwise, this object's getParent()
      // method will eventually return null as the Niagara AX core will
      // eventually attempt to cleanup its proxy
      ddfReceiver = (BIDdfReceiver)getReceiver();

      // Ensures that the ddfReceiver is leased on the client-side
      loadForClientComm(ddfReceiver, "Unable to Sync Receiver");
    }

    // Returns the cached ddfReceiver reference
    return ddfReceiver;
  }


  /**
   * Returns the current value of the 'transactionManager' property cast as a BIDdfTransactionMgr.
   */
  public BIDdfTransactionMgr getDdfTransactionMgr()
  {
    if (ddfTransactionMgr==null)
    {
      // Caches the ddfTransactionMgr. This is important for client-side
      // video stream communications. Otherwise, this object's getParent()
      // method will eventually return null as the Niagara AX core will
      // eventually attempt to cleanup its proxy
      ddfTransactionMgr = (BIDdfTransactionMgr)getTransactionManager();

      // Ensures that the ddfTransactionMgr is leased on the client-side
      loadForClientComm(ddfTransactionMgr, "Unable to Sync Transaction Manager");
    }

    // Returns the cached ddfTransactionMgr reference
    return ddfTransactionMgr;
  }

  /**
   * Returns a log with the log name of "module.typeName.comm".
   */
  public Log getLog()
  {
    String logName = getType().getTypeSpec().getTypeName() + ".h" + getHandle();

    return Log.getLog(logName);
  }

  /**
   * Returns the current value of the 'pollScheduler' property cast as a BIDdfPollScheduler.
   */
  public BIDdfPollScheduler getDdfPollScheduler()
  {
    return (BIDdfPollScheduler)getPollScheduler();
  }

  /**
   * Returns the current value of the 'unsolicitedMgr' property cast as a BIDdfUnsolicitedMgr.
   */
  public BIDdfUnsolicitedMgr getDdfUnsolicitedMgr()
  {
    return (BIDdfUnsolicitedMgr)getUnsolicitedMgr();
  }

  /**
   * This is the primary means of initiating i/o for the driver. The given request is placed onto the communicator's
   * queue where it will be processed in the order that it was placed on the queue.
   *
   * Note that if the the driver's subclass of BDdfCommDevice or BDdfCommNetwork is disabled or in fault
   * then this method does nothing (other than return false).
   *
   * Note that if the given request is an instance of BIDdfWriteRequest and another BIDdfWriteRequest, with an equivalent
   * writeParameters structure as the given request, already exists in the communicator queue, then the two requests are
   * combined into one. In which case, this method returns false but adds any unique writable source objects from the
   * given request to the writable source array of the equivalent request that is already in the communicator's queue.
   *
   * The previous paragraph handles the case where a single request from the driver's protocol is used to set the values
   * for more than one data point in the field-device.
   *
   * If none of the previously mentioned notes apply, then this places the given ddfRequest on the communicator's queue
   * and returns true.
   *
   * NOTE: This is a non-blocking call. If a driver developer needs to process the response or timeout condition, then he
   * or she should declare that the class of the given request implements com.tridium.ddf.comm.req.BIDdfCustomRequest. By doing so,
   * the request itself will receive a callback when either the response is received or a timeout occurs. The callback
   * will be made from the transaction manager thread. @see com.tridium.ddf.comm.req.BIDdfCustomRequest for more details about
   * the callback mechanism.
   *
   * @param BIDdfRequest ddfRequest - an instance of one of the driver's request classes. In general, the instance will be
   * placed on the communicator's queue, the communicator's thread will soon transmit the request, the communicator's
   * transaction manager thread and receive thread will parse incoming data and attempt to match up the received response
   * with the request.
   *
   * @throw BajaRuntimException if the 'isCommEnabled' method returns false (typically indicating that it is either
   * disabled or in fault)
   */
  public void communicate(final BIDdfRequest ddfRequest)
  {
    if (isCommEnabled())
    {
      // Ensures that the component and its child helpers (transmitter, receiver, etc)
      // are synchronized and loaded. This is especially necessary for retrieving
      // video streams directly on the client-side WB.
      loadClientSideSupport();

      // By default, we do not coalesce
      boolean coalesce = false;

      // We might coalesce write requests but only if configured to do so and
      // If we already have a write request queued that is the same exact type
      // And has the equivalent deviceId and writeId
      if ((ddfRequest instanceof BIDdfWriteRequest) &&
          (coalesceWrites((BIDdfWriteRequest)ddfRequest)))
      {
        coalesce = coalesceWrite((BIDdfWriteRequest)ddfRequest);
      }
      if (!coalesce)
      {
        infiniteQueue.enqueue(new Communication(ddfRequest));
      }
    }
    else
    {
      if (getLog().isTraceOn())
        getLog().trace(DdfDefaultCommLexicon.noCommunicationsAvailableWhileInvaldStatus);

      throw new BajaRuntimeException(DdfDefaultCommLexicon.noCommunicationsAvailableWhileInvaldStatus);
    }
  }

  /**
   * The general rule is that this method is called to partially stop the communicator.
   * More specifically, this method is called to stop the portions of the communicator
   * that actually write to ,read from, or otherwise manage the field-bus.
   *
   * The communicating parent (BDdfCommDevice or BDdfCommNetwork -- whichever is
   * the case) calls this when the communicating parent's status changes
   * to fault or disabled.
   *
   * The communicator's stopped method also calls this method.
   *
   * In addition, the serial version of the communicator calls this
   * method after the user changes the serial portName in order
   * to shut down any i/o that was for the previous serial port.
   *
   * This stops the communicator's own thread that processes the communictor's queue. Next,
   * this clears the communicator's queue and notifies any requests that were waiting in
   * the communicator's queue. Next, this stops the unsolicited manager thread. Next, this
   * stops the transaction manager thread. Finally, this stops the receive thread.
   *
   * Note that this does not stop the poll scheduler's thread. The poll scheduler's thread
   * starts when the station starts and stops when the station stops. Any poll attempts
   * that are made while the communicator is disabled will short-circuit.
   */
  public void stopCommunicating()
  {
    synchronized (commSynchronizer)
    {
      // Stops the communicator's own thread
      stopWorker();

      // Cleans up any outstanding entries that are in the communication
      // Queue
      clearQueue();

      // NOTE: Stopping is done in the opposite order as starting
      // so that the transaction manager is disabled last, giving
      // anything else the chance to gracefully shut down first.

      // Stops any threads, etc. that are being used by the unsolicited mgr
      getUnsolicitedMgr().stopUnsolicitedMgr();

      // Stops any threads, etc. that are being used by the transaction mgr
      getTransactionManager().stopTransactionMgr();

      // Stops any threads, etc. that are being used by the transmitter
      getTransmitter().stopTransmitter();

      // Stops any threads, etc. that are being used by the receiver
      getReceiver().stopReceiver();
    }

  }

  /**
   * This method is called to start any threads that the communicator or any
   * of its Niagara AX Nav-descendants requires in order to perform input and
   * output processing for the driver. In other words, this method is called to start the
   * portions of the communicator that actually write to ,read from, or otherwise
   * manage the field-bus.
   *
   * The communicating parent (BDdfCommDevice or BDdfCommNetwork -- whichever is
   * the case) calls this when the communicating parent's status changes
   * such that it is no longer fault or disabled.
   *
   * Also, the communicator's Niagara AX started callback calls this method
   *
   * In addition, in the serial version of the communicator if
   * the serial port is changed to a valid serial port, then this method
   * is called (after first calling the 'stopCommunicating' method to stop
   * i/o on the previous serial port).
   *
   * NOTE: If the communicating Nav parent is disabled or in fault then this
   * method does not do anything.
   *
   */
  public  void startCommunicating()
  {
    // If the current settings allow for communications to be enabled
    if (isCommEnabled())
    {
      synchronized(commSynchronizer)
      {
        loadClientSideSupport();

        // Starts any threads, etc. that are used by the receiver
        getReceiver().startReceiver();

        // Starts any threads, etc. that are being used by the transmitter
        getTransmitter().startTransmitter();

        // Starts any threads, etc. that are used by the transaction mgr
        getTransactionManager().startTransactionMgr();

        // Starts any threads, etc. that are used by the unsolicited mgr
        getUnsolicitedMgr().startUnsolicitedMgr();

        // Starts the communicator's own thread that processes the communicator's
        // infiniteQueue. Note that the 'startWorker' method is called automatically
        // during station start-up. This method is called to accommodate changes
        // to the communicator's settings. Nothing really re-starts in the event
        // that the worker is already started.
        startWorker();
      }
    }
  }

////////////////////////////////////////////////////////////////
// BDdfCommunicator API
////////////////////////////////////////////////////////////////
  /**
   * This method is called from the Niagara AX 'started' callback
   * which is called by the Niagara AX core framework when the
   * component is initialized in a running station. Components
   * are initialized in a running station when the station is
   * started or if they are added to the station database while
   * the station is running.
   */
  public void communicatorStarted()
    throws Exception
  {

  }

  /**
   * This method is called from the Niagara AX 'stopped' callback
   * which is called by the Niagara AX core framework when the
   * component completely shuts down. Components shut down if the
   * station is stopped or if they are removed from the station
   * database.
   */
  public void communicatorStopped()
    throws Exception
  {
  }

  /**
   * Returns the most immediate Niagara AX Nav ancestor that implements
   * BIDdfCommunicating.
   *
   * @return the most immediate Niagara AX Nav ancestor that implements
   * BIDdfCommunicating.
   */
  public BIDdfCommunicating getDdfCommunicatingParent()
  {
    // Caches the communicatingParent. This is important for client-side
    // video stream communications. Otherwise, this object's getParent()
    // method will eventually return null as the Niagara AX core will
    // eventually attempt to cleanup its proxy
    findCommunicatingParent();

    // Returns the cached communicatingParent reference
    return communicatingParent;
  }

  // METHOD: loadForClientComm
  // PURPOSE: To get a fresh copy of all property values for the
  //          given Object that is hopefully a BComponent. Also,
  //          this asks the Niagara core to keep all property
  //          values on the given object updated until further
  //          notice. This also asks the Niagara core to do
  //          likewise for all child components on the given
  //          object that is hopefully a component. Perhaps
  //          most importantly, this ensures that the call to
  //          'getParent' on the given object/BComponent will
  //          return the actual and not null, which could happen
  //          in the client-side JVM if the 'lease' and 'sync' that
  //          this method performs were to be omitted.
  // NOTE: This is necessary to allow limited, direct, client-side
  //       communications for video streams
  void loadForClientComm(Object o, String errorLogSummary)
  {
    BComponentSpace componentSpace;
    if ( (componentSpace = getComponentSpace()) != null && componentSpace.isProxyComponentSpace())
    {
      if (o instanceof BComponent)
      {
        if ( ((BComponent)o).getParent()==null || (!((BComponent)o).isSubscribed()))
        {
          BComponent c = (BComponent)o;
          c.lease(Integer.MAX_VALUE, Integer.MAX_VALUE);
          try
          {
            c.getComponentSpace().sync();
          }
          catch (Exception e)
          {
            getLog().error(errorLogSummary,e);
          }
        }
      }
    }
  }

  /**
   * This is an override point for descendants to easily implement their driver's communication requirements.
   *
   * NOTE: The override point is called from a safe worker-communicator thread.
   *
   * @param ddfRequest the request to transmit
   *
   * @return Returns immediately after the request is transmitted. The request object will receive a callback
   * when the response is received or when the response times out.
   */
  public void doCommunicate(BIDdfRequest ddfRequest)
  {
    if (grantAccess(ddfRequest)) // Allows the developer to temporarily deny access to requests
    {
      // Assigns a retry count and response time-out to the particular request, IF and only IF
      // The particular request does not specifically define it
      ddfRequest.setRemainingRetryCount(getTransmitter().getMaxRetryCount(ddfRequest));
      ddfRequest.setResponseTimeout(getReceiver().getResponseTimeout(ddfRequest));

      if (ddfRequest.getType().is(BDdfRawTransmitRequest.TYPE))
      {
        try
        { // Force-transmits the BDdfRawTransmitRequest bytes. Please note that this
          // Calls "doForceTransmit" directly on the transmitter.  This bypasses
          // Tranaction handling for the ddf force transmit. The BDdfRawTransmitRequest
          // Does not get a response.
          getDdfTransmitter().forceTransmit(ddfRequest);
        }
        catch (Exception e)
        {
          getLog().error("RawTransmitError",e);
        }
      }
      else
      {
        try
        {
          // Synchronizes the communicator for limited, direct, client-side
          // communications
          loadForClientComm(this, "UnableToSyncProxySpace - communicator");

          BIDdfTransmitter transmitter = getDdfTransmitter();

          // Synchronizes the transmitter for limited, direct, client-side
          // communications
          loadForClientComm(transmitter, "UnableToSyncProxySpace - transmitter");


          transmitter.getDdfCommunicator(); // Another HACK to load communicator for client-side video streams

          BIDdfTransactionMgr transactionMgr = getDdfTransactionMgr();

          // Synchronizes the transaction manager for limited, direct, client-side
          // communications
          loadForClientComm(transactionMgr, "UnableToSyncProxySpace - transaction manager");
          transactionMgr.getDdfCommunicator(); // Another HACK to load communicator for client-side video streams

          // Synchronizes the receiver for limited, direct, client-side
          // communications
          BIDdfReceiver receiver = getDdfReceiver();
          loadForClientComm(receiver, "UnableToSyncProxySpace - receiver");

          // Processes the request as a full transaction
          transactionMgr.processTransaction(ddfRequest);
          transmitter.transmitRequest(ddfRequest); // NOTE: Responses are handled in the transaction manager. This returns immediately.
        }
        finally
        {
          if (ddfRequest.getResponseTimeout().getMillis()<=0) // If the request's responseTimeout millis is 0, then there is no expected response
          {
            synchronized(ddfRequest)                          // Since there is no expected response in this case, then
            {                                                 // Let's notify anything that is waiting on the request's
              ddfRequest.notifyAll();                         // Transaction to complete.
            }
          }
        }
      }
    }
    else
    {
      requeue(ddfRequest);
    }
  }

  /**
   * Implements the 'requeue' action. The 'requeue' action is scheduled
   * from the doCommunicate method if the request passed into the doCommunicate
   * method is not granted access (in other words, the grantAccess method returned
   * false for that particular request).
   *
   * This method re-enqueues all requests that are in the requeueList.
   */
  public void doRequeue()
  {
    synchronized(requeueList)
    {
      Iterator<BIDdfRequest> i = requeueList.iterator();

      while (i.hasNext()) // Loops through all items that are in the requeueList
      {                   // Passes each item to the 'communicate' method.
        BIDdfRequest ddfRequest = i.next();

        i.remove(); // Cleans the item out of the requeueList

        communicate(ddfRequest);
      }
    }
  }

  /**
   * Calls the 'resetStatistics' method on the transmitter,
   * receiver, poll scheduler, transaction manager, and
   * unsolicited manager.
   */
  public void doResetStatistics()
  {
    getTransmitter().resetStatistics();

    getReceiver().resetStatistics();

    getPollScheduler().resetStatistics();

    getTransactionManager().resetStatistics();

    getUnsolicitedMgr().resetStatistics();
  }

  /**
   * The 'doCommunicate' method calls this method in the event that the 'grantAccess'
   * method returns false for the given request. This method arranges for the
   * given ddfRequest to be passed back to the 'doCommunicate' method in ten seconds.
   * This is done without blocking the calling thread.
   *
   * This is part of the mechanism that allows driver developers to temporarily grant
   * exclusive access to certain requests over other request. This method handles the
   * case were a particular request does not currently have exclusive access to the
   * driver's field-bus.
   *
   * @param ddfRequest see the description for this method for more details.
   */
  protected void requeue(BIDdfRequest ddfRequest)
  {
    synchronized(requeueList)
    {
      // WARNING:> I believe doing this could overwhelm the BDdfScheduler:
      //          BDdfScheduler.INSTANCE.schedule(this, BRelTime.makeSeconds(1), requeue, (BValue)ddfRequest );
      // INSTEAD:> I'll keep of list of requests that need re-enqueued and enqueue then
      //          When the requeue action fires
      requeueList.addElement(ddfRequest);

      // If the 'requeue' action has not been scheduled, then this asks the
      // BDdfScheduler to call the 'requeue' action in ten seconds. The requeue
      // action's implementation (doRequeue) will pull all requests off the
      // requeueList and 'communicate' them again, thereby re-enqueueing them all.
      if (requeueTicket==null || requeueTicket.isExpired())
        requeueTicket =  BDdfScheduler.INSTANCE.schedule(this,10000,requeue,null);
    }
  }

  /**
   * The 'communicate' method calls this method if its passed a 'BIDdfWriteRequests'.
   * This method reviews the contents of the communicator's queue and if there
   * is already a request in the queue for the equivalent 'writeParams' then the
   * given request's unique writableSource is simply added to the writableSource
   * for the request that already exists.
   *
   * @param ddfWriteRequest  a BIDdfWriteRequest to possibly coalesce with another
   * BIDdfWriteReqeust that is already in the communicator's queue with an equivalent
   * writeParams structure.
   *
   * @return true if the given request was coalesced (combined) into another request
   * that already exists in the communicator queue and that happens to write the
   * same data point(s) in the field-device. The returns false if the given request
   * does not become coalesced (combined) with another write-request that is already
   * in the queue. In the case where false is returned, the 'communicate' method
   * will place the write request on the queue. In the case where true is returned,
   * the 'communicate' method will not place the write request on the queue (since
   * an equivalent request already exists in the queue and since the two equivalent
   * requests were coalesced (combined) during the logic of this method)
   */
  protected boolean coalesceWrite(final BIDdfWriteRequest ddfWriteRequest)
  {
    // TODO: Consider making this public, and having it take a proxy ext, and
    // Having it called from the proxyExtension's doWrite logic before
    // Calling the communicate method
    synchronized(infiniteQueue)
    {
      Object[] communicatorObjects = infiniteQueue.toArray();

      boolean coalesce = false;

      // Loops through all communication requests in the queue, checks if there is already
      // A write request of the same exact type, with the equivalent deviceId, and the equivalent
      // Write parameters. If so, then this sets coalesce to true, which terminates the loop.
      for (int i=0; i<communicatorObjects.length && !coalesce; i++)
      {
        if (communicatorObjects[i] instanceof Communication)
        {
          Communication commObject = (Communication)communicatorObjects[i];

          // If any request is a write request
          if (commObject.ddfRequest instanceof BIDdfWriteRequest)
          {
            // If any write request's type matches the given request exactly
            if (commObject.ddfRequest.getType() == ddfWriteRequest.getType())
            {
              // If any write request's device id is equivalent to the the given one's device id
              if (commObject.ddfRequest.getDeviceId().equivalent(ddfWriteRequest.getDeviceId())) // If any write request's deviceId is equivalent to the given request's device id
              {
                // If any write request's writeParameters is equivalent to the given one's writeParameters
                if (((BIDdfWriteRequest)commObject.ddfRequest).getWriteParameters().equivalent(ddfWriteRequest.getWriteParameters())) // If any write request's writeParameters is equivalent to the given request's writeParameters
                {
                  boolean groupable = ((BIDdfWriteRequest)commObject.ddfRequest).isGroupable();

                  boolean sameWritableSource = sameWritableSource((BIDdfWriteRequest)commObject.ddfRequest,ddfWriteRequest);

                  if (groupable || // If the write request is "groupable" -> In other words, if the writable source for the write requests simply needs combined with the writable source for the write request in the communicator queue
                      sameWritableSource)// Or if the writable source for the given write request is exactly the same as the writable source for the write request in the communicator queue
                  {
                    coalesce = true; // NOTE: This causes the for loop to exit (on next for loop eval).

                    if (!sameWritableSource)
                    {
                      // Adds the IDdfWritables from the request that was passed to this method into
                      // The request that is already in the queue that matches the request that was
                      // Passed to this method.
                      ensureInWritableSource((BIDdfWriteRequest)commObject.ddfRequest, ddfWriteRequest.getWritableSource());
                    }
                  }
                }
              }
            }
          }
        }
      }
      // Returns whether or not the given write request should simply be coalesced
      return coalesce;
    }
  }

  /**
   * The 'coalesceWrite' method calls this method to combine the unique writeable source
   * from a new request in with the writable source for another, equivalen request that
   * is already in the communicator's queue.
   *
   * @param writeReqInQueue a reference to a BIDdfWriteRequest that is already in the
   * communicator's queue. Any unique objects among the given 'writableSourceToCoalesce'
   * will be added to this request's 'writableSource' array.
   *
   * @param writableSourceToCoalesce the 'writableSource' objects from the new request,
   * which will not be placed in the communicator's queue, but instead will be combined
   * with the given 'writeReqInQueue'.
   */
  protected void ensureInWritableSource(BIDdfWriteRequest writeReqInQueue, IDdfWritable[] writableSourceToCoalesce)
  {
    Array<IDdfWritable> writableSourceInQueue = new Array<>(writeReqInQueue.getWritableSource());

    // Loops through all writable source that needs to be coalesced onto the writeReqInQueue request
    for (int i=0; i<writableSourceToCoalesce.length; i++)
    {
      // If any particular writable source reference is not in the array of the writable source that is
      // Already in writeReqInQueue, then this adds it.
      if (!(writableSourceInQueue.contains(writableSourceToCoalesce[i])))
        writableSourceInQueue.add(writableSourceToCoalesce[i]);
    }
    writeReqInQueue.setWritableSource(writableSourceInQueue.trim());
  }

  /**
   * The 'communicate' method calls this method before attempting any coalescing logic
   * for write requests.
   *
   * Descendant classes can override this and return false to disable write coalescing for
   * a particular message or for all messages.
   *
   * @param a write request that is passed to the 'communicate' method.
   *
   * @return true to have the 'communicate' method consider coalescing
   * the given write-request with any write request for the equivalent
   * field-point that is already in the communicator's queue. If there
   * is no such equivalent request then the write request will be placed on
   * the communicator queue. Or return false to have the given
   * write request placed onto the communicator's queue regardless
   * of whether there is  already another request to write the equivalent
   * field-point in the communicator queue.
   */
  protected boolean coalesceWrites(BIDdfWriteRequest ddfRequest)
  {
    return true;
  }

  /**
   * This method is called on the communicator's worker thread for each
   * request that is dequeued.
   *
   * Developers may override this method to grant exclusive access
   * to certain requests by temporarily denying access to some requests
   * while at the same time, granting access to other requests.
   *
   * If the developer returns true then the request will be processed
   * immediately (transmitted and scheduled for response checking). If
   * the developer returns false then the request will be placed back
   * in the back of the communicator queue.
   *
   * @param ddfRequest this is a request that was just pulled off of the
   * communicator queue. However, this request has not yet been processed.
   * This request will be processed immediately if this method returns
   * true. This request will be processed later if this method returns
   * false.
   *
   * @return this default version of the checkAccess method returns true
   * all of the time. Developers may customize this, as described.
   */
  protected boolean grantAccess(BIDdfRequest ddfRequest)
  {
    return true;
  }

  /**
   * Communications are considered to be enabled if the communicating parent
   * is neither disabled nor fault.
   */
  boolean isCommEnabled()
  {
    findCommunicatingParent();
    return !(communicatingParent==null || communicatingParent.isDisabled() || communicatingParent.isFault());
  }



  /**
   * This method serves in the 'coalescing' process. Its purpose is to check
   * if all of the writable source for one request is the same as the writable
   * source for another request.
   *
   * @param reqA a write request to check if all of its writableSource objects
   * are included among the writableSource for the given 'reqB'
   *
   * @param reqB a write request to check if it includes all of the 'writableSource'
   * objects from the given 'reqA'.
   *
   * @return true if each writableSource object for reqA is also included among the
   * writableSource for reqB.
   */
  private boolean sameWritableSource(final BIDdfWriteRequest reqA, final BIDdfWriteRequest reqB)
  {
    // Gets all 'writables' for reqA and reqB
    IDdfWritable reqAWritables[] = reqA.getWritableSource();
    IDdfWritable reqBWritables[] = reqB.getWritableSource();

    if (reqAWritables==null || reqBWritables==null ||   // If no 'writables' for either reqeust
        reqAWritables.length!=reqBWritables.length ||   // Or different nbr of 'writables' in each
        reqAWritables.length == 0)                      // Or same nbr 'writables' in each (as
    {                                                   //  verified by the previous line of code)
      return false;                                     //  but the reqA's size (and also, implicitly
    }                                                   //  the reqB's size) = 0
    else
    {
      // Hashes all reqBWritables to their object reference
      // NOTE: We are hashing to avoid an O(n^2) pair of nested loops.
      // HASHING is O(n) instead -> O(nb + na) to be exact
      Hashtable<IDdfWritable, IDdfWritable> reqBHash = new Hashtable<>(reqBWritables.length*2); // Doubles the capacity of the hash to ensure that no rehashing takes places

      for (int b=0; b<reqBWritables.length; b++)
        reqBHash.put(reqBWritables[b], reqBWritables[b]);

      // Reviews all reqAWritables
      for (int a=0; a<reqAWritables.length; a++)
      {
        // If any reqAWritable is not directly in the reqBWritables hash then reqA has a 'writable;
        // That is not in reqB so we return false
        if (reqBHash.get(reqAWritables[a])==null)
          return false;
      }

      // If the logic makes it this far, then all 'writables' among reqA were also found among 'reqB'
      return true;
    }
  }

  /**
   * Clears all items from the queue by dequeuing them all and
   * calling 'notifyAll' on each communication's request to wake
   * up any threads that are waiting on the request to complete.
   *
   * Note that if we did not notify any threads that are waiting on
   * the request, then those threads would deadlock because by
   * clearing this queue, the communication objects will not be
   * serviced. Not servicing the communication objects results in
   * no timeout or success. No timeout or success means that the
   * request is not notified. If the request is not notified then
   * the DdfRequestUtil.communicateSync method will get stuck.
   */
   private void clearQueue()
   {
     // Dequeues the next item from the queue
     Object queueItem = infiniteQueue.dequeue();

     // Loops through all items in the queue
     while (queueItem!=null)
     {
       // It the item is a 'Communication' then we need to notify
       // all threads waiting on the request for the communication.
       if (queueItem instanceof Communication)
       {
         Communication queueCommItem = (Communication)queueItem;

         // Locks the request so that it can be notfiied
         synchronized (queueCommItem.ddfRequest)
         {
           // Notfies all threads waiting on the request.
           queueCommItem.ddfRequest.notifyAll();
         }
       }
       // Pulls the next item off of the queue
       queueItem = infiniteQueue.dequeue();
     }

   }

  /**
   * This method finds the first BDdfCommNetwork or BDdfCommDevice
   * that is above this communicator.
   */
  private void findCommunicatingParent()
  {
    // Looks up the ancestry for the first device or network and
    // Keys off of its status
    BComplex parent = getParent();

    while (parent!=null)
    {
      if (parent instanceof BIDdfCommunicating)
      {
        this.communicatingParent=(BIDdfCommunicating)parent;
        return;
      }
      parent = parent.getParent();
    }
  }

////////////////////////////////////////////////////////////////
// Communication
////////////////////////////////////////////////////////////////

  /**
   * Communication - Whenever a request is passed to the communicate method,
   * in instance of this is really placed on the communicator's queue wrapping
   * the given request (since the queue has to take a runnable).
   *
   * @author   LPerkins
   * @creation Aug 24, 2007
   * @version  $Revision$ $Aug 24, 2007 4:01:27 PM$
   * @since    Niagara 3.2
   */
  public class Communication
    implements Runnable
  {
    Communication(BIDdfRequest ddfRequest)
    {
     this.ddfRequest=ddfRequest;
    }

    public void run()
    {
      doCommunicate(ddfRequest);
    }

    BIDdfRequest ddfRequest;
  }

////////////////////////////////////////////////////////////////
// Attributes
////////////////////////////////////////////////////////////////

  /**
   * This is used in conjunction with the 'requeue' action.
   */
  protected Vector<BIDdfRequest> requeueList = new Vector<>(); // See the 'requeue' method

  /**
   * This is used in conjunction with the 'requeue' action.
   */
  protected Ticket requeueTicket = null; // See the 'requeue' method

  /**
   * This is a direct reference to the communicator's internal queue.
   */
  protected Queue infiniteQueue = new Queue();

  /**
   * This is a direct reference to the communicator's internal worker.
   *
   * This wraps the communicator's own thread.
   */
  protected Worker communicationWorker = new Worker(infiniteQueue);

  // This is a reference to the communicating parent that owns this communicator
  private BIDdfCommunicating communicatingParent;

  // This is a reference to the receiver, cached for performance optimization and
  // to permit client-side, direct communications (a required change to support
  // video streams)
  private BIDdfReceiver ddfReceiver;

  // This is a reference to the transaction manager, cached for performance optimization and
  // to permit client-side, direct communications (a required change to support
  // video streams)
  private BIDdfTransactionMgr ddfTransactionMgr;

  private BIDdfTransmitter ddfTransmitter;

  // Creates a special class for the commSynchronizer object. This will make
  // it easier to identify instances of the CommSynchonizer object in any
  // deadlock traces by the VM
  class CommSynchronizer
  {

  }

  private Object commSynchronizer = new CommSynchronizer();


}
