/*
 * @copyright 2005 Tridium Inc.
 */
package com.tridium.ddf.comm;

import javax.baja.sys.BInterface;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;

import com.tridium.ddf.IDdfFacetConst;
import com.tridium.ddf.comm.req.BIDdfRequest;

/**
 * The BDdfCommunicator uses an instance of this to transmit a BIDdfRequest.
 * 
 * 
 * Instead of implementing this directly, there are some implementations that can be extended.
 * These are in the packages:
 * <OL>
 *   <LI>com.tridium.ddf.comm.defaultComm
 *   <LI>com.tridium.ddf.comm.multipleTransaction
 *   <LI>com.tridium.ddf.comm.singleTransaction
 * </OL>
 * 
 * In addition, there are reasonable serial versions of these in the devSerialDriver module.
 * 
 * There are also reasonable Tcp/Ip and Udp/Ip versions of these in the devIpDriver module.
 *
 * @author    lperkins
 * @creation  Oct 10, 2006
 * @version   $Revision$ $Date$
 * @since     Niagara 3.0
 */
public interface BIDdfTransmitter
    extends BInterface, IDdfFacetConst
{
  /**
   * This method should call forceTransmit and also update 
   * any statistical values, such as a 'transmissionCount'
   * property.
   * 
   * @param ddfRequest the driver's request to serialize (convert to a byte array)
   * and transmit.
   */
  public void transmitRequest(BIDdfRequest ddfRequest);
  
  /**
   * This method is called to blindly transmit the byte representation of the given request
   * 
   * This method should only be called internally be a BIDdfTransactionMgr.
   * 
   * @param ddfRequest. The driver's request to serialize (convert to a byte array)
   * and transmit.
   */
  public void forceTransmit(BIDdfRequest ddfRequest) throws Exception;
  
  /**
   * Gets the communicator that owns this transmitter.
   * @return the communicator that owns this transmitter.
   */
  public BIDdfCommunicator getDdfCommunicator();
  
  /**
   * Gets the maximum number of retries that is permitted per request, unless
   * the individual request defines a different value, before timing out.
   * 
   * @return the maximum number of retries that is permitted per request, unless
   * the individual request defines a different value, before timing out.
   */
  public int getMaxRetryCount();
  
  /**
   * This is a 'setter' for a 'retransmissionCount' property. 
   * By default, this framework will automatically call this
   * method each time that it attempts to retry a transmission.
   */
  public void setRetransmissionCount(long retransmissionCount);

  /**
   * This is a 'getter' for a 'retransmissionCount' property.
   *  
   * By default, this framework will automatically call the
   * corresponding 'setter' each time that it attempts to retry
   * a transmission.
   */
  public long getRetransmissionCount();
  
  /**
   * Starts any threads that are required to accommodate the drivers
   * transmitter.
   */
  public void startTransmitter();
  
  /**
   * Stops any threads that were required to accommodate the drivers
   * transmitter.
   */
  public void stopTransmitter();
  
  
////////////////////////////////////////////////////////////////
// Type
////////////////////////////////////////////////////////////////
  
  public static final Type TYPE = Sys.loadType(BIDdfTransmitter.class);
}
