/*
 * @copyright 2005 Tridium Inc.
 */
package com.tridium.ddf.comm;

import javax.baja.log.Log;
import javax.baja.sys.BInterface;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;

import com.tridium.ddf.IDdfFacetConst;
import com.tridium.ddf.comm.req.BIDdfRequest;
import com.tridium.ddf.poll.BIDdfPollScheduler;

/**
 * The BIDdfCommunicator is fundamtental to the developer driver framework.
 * It combines the transmitter, receiver, transaction manager, poll
 * scheduler, and unsolicited receive handler for a field-bus.
 * 
 * Instead of implementing this directly, there are a wealth of implementations that can be extends.
 * These are in the packages:
 * <OL>
 *   <LI>com.tridium.ddf.comm.defaultComm
 *   <LI>com.tridium.ddf.comm.multipleTransaction
 *   <LI>com.tridium.ddf.comm.singleTransaction
 * </OL>
 * In additon, there reasonable serial versions of these in the devSerialDriver module.
 * 
 * There are reasonable Tcp/Ip versions of these in the devTcpDriver module.
 * @author lperkins
 */
public interface BIDdfCommunicator
    extends BInterface, IDdfFacetConst
{
  /**
   * The implementing object should transmit this BIDdfRequest object. Then depending on the
   * BIDdfRequest type, the object should make the proper arrangements for the BIDdfRequest
   * object's processReceive methods to be called for incoming frames that match up to the 
   * request.
   * 
   * @param ddfRequest
   */
  public void communicate(BIDdfRequest ddfRequest);
  
  /**
   * The BIDdfCommunicator uses the BIDdfTransmitter to transmit the byte
   * array representation of the BIDdfRequests.
   */
  public BIDdfTransmitter getDdfTransmitter();
  
  /**
   * The BIDdfCommunicator uses the BIDdfReceiver to receive data frames
   * (glorified byte arrays) from the field-bus.
   */
  public BIDdfReceiver getDdfReceiver();
  /**
   * The BIDdfCommunicator uses the BIDdfTransactionMgr to match received
   * data from to previously transmitted requests or to the BIDdfUnsolicitedMgr.
   */
  public BIDdfTransactionMgr getDdfTransactionMgr();
  
  /**
   * The BIDdfCommunicator sends Trace information, errors, and other messages
   * to this Log.
   */
  public Log getLog();
  /**
   * The BIDdfCommunicator places poll groups (of IDdfReadables -- proxies) into
   * the BIDdfPollScheduler for periodic updates.
   * 
   * @throws IllegalAccessException if the BIDdfCommunicator does not fundamentally
   * feature a developer poll scheduler.
   */
  public BIDdfPollScheduler getDdfPollScheduler() throws IllegalAccessException;
  /**
   * The BIDdfCommunicator's transaction manager sends data frames that it receives,
   * but cannot match to previously transmitted responses, to the BIDdfUnsolicitedMgr.
   */
  public BIDdfUnsolicitedMgr getDdfUnsolicitedMgr();
  
  /**
   * This method is called to disable communications. 
   */
  public void stopCommunicating();

  /**
   * This method is called to enable communications. 
   */
  public void startCommunicating();
  
////////////////////////////////////////////////////////////////
//Type
////////////////////////////////////////////////////////////////
  
  public static final Type TYPE = Sys.loadType(BIDdfCommunicator.class);
  
}
