/*
 * @copyright 2005 Tridium Inc.
 */
package com.tridium.ddf;

import javax.baja.agent.AgentInfo;
import javax.baja.agent.AgentList;
import javax.baja.agent.BPxView;
import javax.baja.registry.TypeInfo;
import javax.baja.security.BPermissions;
import javax.baja.sys.BIcon;
import javax.baja.sys.BObject;
import javax.baja.sys.Context;
import javax.baja.sys.Sys;
import javax.baja.util.Lexicon;
import javax.baja.nre.util.TextUtil;

/**
 * This is used to rename the Ddf Device Manager and the Ddf Point Manager based on the name
 * of the driver that is built off of this framework.
 *
 * For example, instead of being named 'Ddf Device Manager', the device manager is named
 * 'Your Driver Device Manager' 'Your Driver' is the name of a driver that inherits from
 * this framework.
 *
 * This basically wraps another AgentInfo object but provides its own implementation of
 * the 'getDisplayName' method.
 *
 * @author lperkins
 *
 */
public class DdfAgentInfo
 implements AgentInfo
{
  /**
   * Creates an instance of this class, that wraps the given AgentInfo, yet provides
   * its own implementation of the 'getDisplayName(Context)' method.
   *
   * @param nAgentInfo the default instance of AgentInfo for a BComponet
   *
   * @param driverModuleName the name of the driver that builds off of this framework
   *
   * @param displayNameKey the name of the key in the driver's Lexicon from which to lookup
   * the text for 'Device Manager'
   *
   * @param displayNameDefault the default text to use for the String 'Device Manager' in
   * the event that the driver's Lexicon does not define the 'displayNameKey'
   */
  public DdfAgentInfo(AgentInfo nAgentInfo, String driverModuleName, String displayNameKey, String displayNameDefault)
  {
    this.nAgentInfo=nAgentInfo;
    this.driverModuleName=driverModuleName;
    this.displayNameKey=displayNameKey;
    this.displayNameDefault=displayNameDefault;
  }

////////////////////////////////////////////////////////////////////////////
// AgentInfo
////////////////////////////////////////////////////////////////////////////

  /**
   * Get the agent id which is used to uniquely
   * identify it in AgentLists.
   */
  public String getAgentId()
  {
    return nAgentInfo.getAgentId();
  }

  /**
   * Get an array of the TypeInfos which this
   * agent is directly registered on.
   */
  public TypeInfo[] getAgentOn()
  {
    return nAgentInfo.getAgentOn();
  }

  /**
   * Get the TypeInfo for the instance which would be
   * returned by the <code>getInstance()<code>.
   */
  public TypeInfo getAgentType()
  {
    return nAgentInfo.getAgentType();
  }

  /**
   * Customized the 'displayName' as described in the Java Doc for the class
   * definition.
   */
  public String getDisplayName(Context cx)
  {
    if (driverModuleName!=null) // Sanity Check
    {
      // Gets name of driver module
      String driverDisplayName = TextUtil.toFriendly(driverModuleName);

      if (driverDisplayName!=null) // Sanity Check
      {
        // Gets suffix name of the manager
        Lexicon driverLex = Lexicon.make(driverModuleName);
        String suffix = displayNameDefault;
        if (driverLex!=null) // Sanity Check
        {
          suffix = driverLex.get(displayNameKey,displayNameDefault);
        }

        // Returns the name of the driver module concat'ed with the name of the manager
        return driverDisplayName + ' '+suffix;
      }
    }
    return nAgentInfo.getDisplayName(cx);
  }

  /**
   * Get the icon for the agent.
   */
  public BIcon getIcon(Context cx)
  {
    return nAgentInfo.getIcon(cx);
  }

  /**
   * Create an instance of the agent.
   */
  public BObject getInstance()
  {
    return nAgentInfo.getInstance();
  }

  /**
   * Get the permissions required to use this agent.
   */
  public BPermissions getRequiredPermissions()
  {
    return nAgentInfo.getRequiredPermissions();
  }

////////////////////////////////////////////////////////////////////////////
// Util
////////////////////////////////////////////////////////////////////////////

  /**
   * The BDdfCommNetwork.getAgents method calls this to rename the ddf
   * device manager to be the "Your Driver" Device Manager where "Your Driver"
   * is the name of the driver that extends the ddf.
   *
   * @param agents The result of calling super.getAgents(cx) on the component that calls this method.
   *
   * @param driverModuleName The name of the module that inherits from the ddf.
   *
   * @param mgrTypeSpec the type spec of the ddf manager whose name will be changed
   *
   * @param mgrSuffixKey the lexicon key to look up for the suffix, for example, "DeviceManager",
   *        in case the driver developer wishes to customize the manager's name (he or she can do
   *        so from his or her driver's Lexicon)
   *
   * @param mgrSuffixDefault the default suffix. This is used if the driver developer does not
   *        provide a 'mgrSuffixKey' in his or her driver's Lexicon to customize the manager's
   *        name (for example, "Device Manager")
   *
   * @return the AgentList with the one particular agent renamed as prescribed.
   */
  public static AgentList getAgentsHelp(AgentList agents, String driverModuleName, String mgrTypeSpec, String mgrSuffixKey, String mgrSuffixDefault)
  {
    TypeInfo ddfManagerTypeInfo = Sys.getRegistry().getType(mgrTypeSpec);
    AgentInfo ddfManagerAgentInfo =
      new DdfAgentInfo(
          ddfManagerTypeInfo.getAgentInfo(),
          driverModuleName,
          mgrSuffixKey,
          mgrSuffixDefault);

    boolean containsExtendedVersionOfMgr = false;
    boolean renamedAgent = false;

    // Determines if the list of agents contains an agent whose type is a descendant of the agent type identified
    // by the given mgrTypeSpec. If so, then we need to assure that the descendant agent shows up instead of the
    // default agent that is defined in the ddf.
    for(int i=0; i<agents.size(); ++i)
    {
      AgentInfo agentInfo = agents.get(i);
      TypeInfo agentTypeInfo = agentInfo.getAgentType();
      if (agentTypeInfo.is(ddfManagerTypeInfo) &&  !agentTypeInfo.equals(ddfManagerTypeInfo))
      {
        containsExtendedVersionOfMgr = true;
      }
    }

    // Loops through the given list of agents...
    for(int i=0; i<agents.size(); ++i)
    {
      // If an agent from the agent list matches the agent as identified by the given mgrTypeSpec
      if (agents.get(i).getAgentType().equals(ddfManagerTypeInfo))
      {
        agents.remove(i); // Removes the AgentInfo that the Niagara AX core added automatically
        if (!containsExtendedVersionOfMgr)  // If no modules provide an overridden version of the agent
        {                                   // Then this adds the AgentInfo right back to the list but
          agents.add(ddfManagerAgentInfo);  // changes the name, as is the purpose of this method
          renamedAgent = true;
        }
      }
    }

    // If the for loop did not rename an AgentInfo object, then that means that somehow the Niagara AX
    // core did not place that the corresponding agent type into the agent list. If that is the case,
    // and if no other module defines an extended version of the agent, then this adds the appropriate
    // AgentInfo to the AgentList with given name.
    if (!renamedAgent && !containsExtendedVersionOfMgr)
      agents.add(ddfManagerAgentInfo);

    // Issue 12302 Do not place the default Device Manager or Point Manager above Px Pages.
    for (int i=agents.size()-1; i>=0; i--)
    {
      if (agents.get(i).getAgentType().is(BPxView.TYPE))
      {
        agents.toTop(i);
      }
    }

    return agents;

  }

  /**
   * This method is required for AgentInfo in baja 3.1. For backwards compatibility,
   * it simply returns null.
   */
  public String getAppName()
  {
    return null;
  }

////////////////////////////////////////////////////////////////
// Attributes
////////////////////////////////////////////////////////////////

  /**
   * This is the AgentInfo object that this object wraps. See the Java Doc with the
   * class description for more details.
   */
  protected AgentInfo nAgentInfo;

  /**
   * This is the driverModuleName that is passed to the constructor.
   */
  protected String driverModuleName;

  /**
   * This is the displayNameKey that is passed to the constructor.
   *
   * This is used to look up text in the Lexicon for a driver that is built
   * on this framework.
   */
  protected String displayNameKey;

  /**
   * This is the displayNameKey that is passed to the constructor.
   *
   * This is used by default, if the driver that is built on this framework
   * does not define a displayNameKey in its lexicon.
   */
  protected String displayNameDefault;
}
