/*
 * Copyright 2000 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.control.ext;

import javax.baja.control.BNumericPoint;
import javax.baja.control.BPointExtension;
import javax.baja.control.enums.BTotalizationInterval;
import javax.baja.status.BStatus;
import javax.baja.status.BStatusNumeric;
import javax.baja.status.BStatusValue;
import javax.baja.sys.Action;
import javax.baja.sys.BAbsTime;
import javax.baja.sys.BComponent;
import javax.baja.sys.BFacets;
import javax.baja.sys.BRelTime;
import javax.baja.sys.Clock;
import javax.baja.sys.Context;
import javax.baja.sys.Flags;
import javax.baja.sys.Property;
import javax.baja.sys.Slot;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;

/**
 * BNumericTotalizerExt is a standard point extension used to
 * for integrating a numeric point value over time.  
 *
 * For example, a totalizer with a minutely totalization
 * interval can convert an instantaneous flow reading 
 * in cubic feet per minute (cfm) into a value representing
 * total cubic feet consumed.
 *
 * @author    Dan Giorgis
 * @creation  9 Nov 00
 * @version   $Revision: 31$ $Date: 6/23/09 10:22:28 AM EDT$
 * @since     Baja 1.0
 */

public class BNumericTotalizerExt
  extends BPointExtension
{ 

  /*-
  
  class BNumericTotalizerExt
  {
    properties
    {
      totalFacets: BFacets
        -- These facets are applied against the total property.
        default {[ BFacets.makeNumeric() ]}

      propagateFlags: BStatus
      -- defines which input status flags will be propagated from
      -- input to output.
        default{[ BStatus.make(BStatus.FAULT | BStatus.DOWN | BStatus.DISABLED | BStatus.STALE) ]}
      invalidValueFlags: BStatus
      -- defines which input status flags will denote invalid input 
      -- values that should not be included in the total
        default{[ BStatus.make(BStatus.FAULT | BStatus.DOWN | BStatus.DISABLED) ]}
      
      total: BStatusNumeric
        -- Total accumulated value since last reset
        flags { readonly }
        default {[ new BStatusNumeric(0) ]}
      timeOfTotalReset: BAbsTime
        -- Time of last reset 
        flags { readonly }
        default {[ BAbsTime.make() ]}
      totalizationInterval: BTotalizationInterval
        -- Interval over which to accumulate values
        default {[ BTotalizationInterval.minutely ]}

    }

    actions
    {
      timerExpired()
      resetTotal()
      -- Reset the total to zero
    }
  }

  -*/
/*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
/*@ $javax.baja.control.ext.BNumericTotalizerExt(1804875717)1.0$ @*/
/* Generated Mon Jun 22 14:23:30 EDT 2009 by Slot-o-Matic 2000 (c) Tridium, Inc. 2000 */

////////////////////////////////////////////////////////////////
// Property "totalFacets"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>totalFacets</code> property.
   * These facets are applied against the total property.
   * @see javax.baja.control.ext.BNumericTotalizerExt#getTotalFacets
   * @see javax.baja.control.ext.BNumericTotalizerExt#setTotalFacets
   */
  public static final Property totalFacets = newProperty(0, BFacets.makeNumeric(),null);
  
  /**
   * Get the <code>totalFacets</code> property.
   * @see javax.baja.control.ext.BNumericTotalizerExt#totalFacets
   */
  public BFacets getTotalFacets() { return (BFacets)get(totalFacets); }
  
  /**
   * Set the <code>totalFacets</code> property.
   * @see javax.baja.control.ext.BNumericTotalizerExt#totalFacets
   */
  public void setTotalFacets(BFacets v) { set(totalFacets,v,null); }

////////////////////////////////////////////////////////////////
// Property "propagateFlags"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>propagateFlags</code> property.
   * defines which input status flags will be propagated
   * from input to output.
   * @see javax.baja.control.ext.BNumericTotalizerExt#getPropagateFlags
   * @see javax.baja.control.ext.BNumericTotalizerExt#setPropagateFlags
   */
  public static final Property propagateFlags = newProperty(0, BStatus.make(BStatus.FAULT | BStatus.DOWN | BStatus.DISABLED | BStatus.STALE),null);
  
  /**
   * Get the <code>propagateFlags</code> property.
   * @see javax.baja.control.ext.BNumericTotalizerExt#propagateFlags
   */
  public BStatus getPropagateFlags() { return (BStatus)get(propagateFlags); }
  
  /**
   * Set the <code>propagateFlags</code> property.
   * @see javax.baja.control.ext.BNumericTotalizerExt#propagateFlags
   */
  public void setPropagateFlags(BStatus v) { set(propagateFlags,v,null); }

////////////////////////////////////////////////////////////////
// Property "invalidValueFlags"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>invalidValueFlags</code> property.
   * defines which input status flags will denote invalid
   * input values that should not be included in the total
   * @see javax.baja.control.ext.BNumericTotalizerExt#getInvalidValueFlags
   * @see javax.baja.control.ext.BNumericTotalizerExt#setInvalidValueFlags
   */
  public static final Property invalidValueFlags = newProperty(0, BStatus.make(BStatus.FAULT | BStatus.DOWN | BStatus.DISABLED),null);
  
  /**
   * Get the <code>invalidValueFlags</code> property.
   * @see javax.baja.control.ext.BNumericTotalizerExt#invalidValueFlags
   */
  public BStatus getInvalidValueFlags() { return (BStatus)get(invalidValueFlags); }
  
  /**
   * Set the <code>invalidValueFlags</code> property.
   * @see javax.baja.control.ext.BNumericTotalizerExt#invalidValueFlags
   */
  public void setInvalidValueFlags(BStatus v) { set(invalidValueFlags,v,null); }

////////////////////////////////////////////////////////////////
// Property "total"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>total</code> property.
   * Total accumulated value since last reset
   * @see javax.baja.control.ext.BNumericTotalizerExt#getTotal
   * @see javax.baja.control.ext.BNumericTotalizerExt#setTotal
   */
  public static final Property total = newProperty(Flags.READONLY, new BStatusNumeric(0),null);
  
  /**
   * Get the <code>total</code> property.
   * @see javax.baja.control.ext.BNumericTotalizerExt#total
   */
  public BStatusNumeric getTotal() { return (BStatusNumeric)get(total); }
  
  /**
   * Set the <code>total</code> property.
   * @see javax.baja.control.ext.BNumericTotalizerExt#total
   */
  public void setTotal(BStatusNumeric v) { set(total,v,null); }

////////////////////////////////////////////////////////////////
// Property "timeOfTotalReset"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>timeOfTotalReset</code> property.
   * Time of last reset
   * @see javax.baja.control.ext.BNumericTotalizerExt#getTimeOfTotalReset
   * @see javax.baja.control.ext.BNumericTotalizerExt#setTimeOfTotalReset
   */
  public static final Property timeOfTotalReset = newProperty(Flags.READONLY, BAbsTime.make(),null);
  
  /**
   * Get the <code>timeOfTotalReset</code> property.
   * @see javax.baja.control.ext.BNumericTotalizerExt#timeOfTotalReset
   */
  public BAbsTime getTimeOfTotalReset() { return (BAbsTime)get(timeOfTotalReset); }
  
  /**
   * Set the <code>timeOfTotalReset</code> property.
   * @see javax.baja.control.ext.BNumericTotalizerExt#timeOfTotalReset
   */
  public void setTimeOfTotalReset(BAbsTime v) { set(timeOfTotalReset,v,null); }

////////////////////////////////////////////////////////////////
// Property "totalizationInterval"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>totalizationInterval</code> property.
   * Interval over which to accumulate values
   * @see javax.baja.control.ext.BNumericTotalizerExt#getTotalizationInterval
   * @see javax.baja.control.ext.BNumericTotalizerExt#setTotalizationInterval
   */
  public static final Property totalizationInterval = newProperty(0, BTotalizationInterval.minutely,null);
  
  /**
   * Get the <code>totalizationInterval</code> property.
   * @see javax.baja.control.ext.BNumericTotalizerExt#totalizationInterval
   */
  public BTotalizationInterval getTotalizationInterval() { return (BTotalizationInterval)get(totalizationInterval); }
  
  /**
   * Set the <code>totalizationInterval</code> property.
   * @see javax.baja.control.ext.BNumericTotalizerExt#totalizationInterval
   */
  public void setTotalizationInterval(BTotalizationInterval v) { set(totalizationInterval,v,null); }

////////////////////////////////////////////////////////////////
// Action "timerExpired"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>timerExpired</code> action.
   * @see javax.baja.control.ext.BNumericTotalizerExt#timerExpired()
   */
  public static final Action timerExpired = newAction(0,null);
  
  /**
   * Invoke the <code>timerExpired</code> action.
   * @see javax.baja.control.ext.BNumericTotalizerExt#timerExpired
   */
  public void timerExpired() { invoke(timerExpired,null,null); }

////////////////////////////////////////////////////////////////
// Action "resetTotal"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>resetTotal</code> action.
   * Reset the total to zero
   * @see javax.baja.control.ext.BNumericTotalizerExt#resetTotal()
   */
  public static final Action resetTotal = newAction(0,null);
  
  /**
   * Invoke the <code>resetTotal</code> action.
   * @see javax.baja.control.ext.BNumericTotalizerExt#resetTotal
   */
  public void resetTotal() { invoke(resetTotal,null,null); }

////////////////////////////////////////////////////////////////
// Type
////////////////////////////////////////////////////////////////
  
  public Type getType() { return TYPE; }
  public static final Type TYPE = Sys.loadType(BNumericTotalizerExt.class);

/*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

  public BFacets getSlotFacets(Slot slot)
  {
    if(slot.equals(total))
      return getTotalFacets();
    return super.getSlotFacets(slot);
  }

////////////////////////////////////////////////////////////////
//  PointExtension
////////////////////////////////////////////////////////////////

  public boolean requiresPointSubscription()
  {
    return true;
  }

////////////////////////////////////////////////////////////////
//  Parent checking
////////////////////////////////////////////////////////////////  

  /**
   * Parent must be NumericPoint.
   */
  public boolean isParentLegal(BComponent parent)
  {                        
    if (!super.isParentLegal(parent)) return false;
    return parent instanceof BNumericPoint;
  }

////////////////////////////////////////////////////////////////
//  Initialization
////////////////////////////////////////////////////////////////

  /**
   * Callback for when the extension is started.
   */
  public void started()
    throws Exception
  {
    super.started();
    // Initialize state variables
    lastExecuteTime = Clock.ticks();
    ticket = Clock.schedulePeriodically(this, BRelTime.makeSeconds(10), timerExpired, null);
  }
  
  public void stopped()
  {
    if(ticket!=null)
      ticket.cancel();
  }

////////////////////////////////////////////////////////////////
// Update Methods
////////////////////////////////////////////////////////////////

  /**
   * Callback for timer expired.
   */
  public void doTimerExpired()
  {
    onExecute(getParentPoint().getOutStatusValue(), null);
  }

  /** 
   * Called when either me or my parent control 
   * point is updated.
   */ 
  public void onExecute(BStatusValue o, Context cx)
  {
    BStatusNumeric out = (BStatusNumeric)o;

    BStatus status = out.getStatus();
    double value = out.getValue();
    long now = Clock.ticks();
    
    double interval;
    if (getTotalizationInterval().equals(BTotalizationInterval.minutely))
      interval = 60000;  // milliseconds 
    else 
      interval = (60 * 60000);
    
    // Only include in total if input is valid
    if((status.getBits() & getInvalidValueFlags().getBits())==0 && !Double.isNaN(value))
    {
      // Only integrate if our previous value was also valid
      if((lastStatus.getBits() & getInvalidValueFlags().getBits())==0)
      {
        //  Calculate the totalization since the last
        //  time we updated.    
        //  (area under rectangle : deltaT * lastValue)    
        double thisTotal = (value * (now - lastExecuteTime)) / interval;
    
        getTotal().setValue(getTotal().getValue()+thisTotal);
      }
    } 
      
    if (Double.isNaN(value))
      getTotal().setStatus(BStatus.make(BStatus.FAULT | (status.getBits() & getPropagateFlags().getBits())));
    else
      getTotal().setStatus(BStatus.make(status.getBits() & getPropagateFlags().getBits()));
    
    lastStatus = out.getStatus(); 
    lastExecuteTime = now;    
  }

  public void changed(Property property, Context context)
  {    
    if(isRunning() && property.equals(totalizationInterval))
      doResetTotal();
    super.changed(property, context);
  }  
  
////////////////////////////////////////////////////////////////
// Actions
////////////////////////////////////////////////////////////////

  public void doResetTotal()
  {
    getTotal().setValue(0);
    setTimeOfTotalReset(BAbsTime.now());
  }

////////////////////////////////////////////////////////////////
// Attributes
////////////////////////////////////////////////////////////////  

  private Clock.Ticket ticket;

  /* last time the totalizer updated */
  private long lastExecuteTime;   

  /* last status from parent object */
  private BStatus lastStatus = BStatus.ok;
}