/*
 * Copyright 2002 Tridium, Inc. All Rights Reserved.
 */
package com.tridium.basicdriver.serial;

import java.io.InputStream;
import java.io.OutputStream;
import javax.baja.serial.BISerialPort;
import javax.baja.serial.BISerialService;
import javax.baja.sys.BRelTime;
import javax.baja.sys.Clock;
import javax.baja.sys.Sys;
import com.tridium.basicdriver.comm.Comm;
import com.tridium.basicdriver.comm.CommReceiver;
import com.tridium.basicdriver.comm.CommTransactionManager;
import com.tridium.basicdriver.comm.CommTransmitter;
import com.tridium.basicdriver.message.Message;
import com.tridium.basicdriver.message.ReceivedMessage;
import com.tridium.basicdriver.util.BasicException;

/**
 * SerialComm is used to synchronize access to a
 * serial network and handles the synchronization of the serial
 * communication.
 * @author    Scott Hoye
 * @creation  26 Mar 02
 * @version   $Revision: 1$ $Date: 03/26/02 12:47:14 PM$
 * @since     Niagara 3.0 basicdriver 1.0
 */

public class SerialComm
  extends Comm
{

/////////////////////////////////////////////////////////////////
// Constructors
/////////////////////////////////////////////////////////////////

 /**
  * Constructor - initializes the SerialComm with a specified BSerialNetwork
  * and CommReceiver (receive driver).  At a minimum, these two instances
  * must always be supplied.  Uses the default CommTransmitter 
  * (transmit driver) and CommTransactionManager.
  */
  public SerialComm(BSerialNetwork serialNetwork, CommReceiver rDriver)
  {
    super(serialNetwork, rDriver);
  }

 /**
  * Constructor - initializes the SerialComm with a specified BSerialNetwork,
  * CommReceiver (receive driver), and CommTransmitter (transmit driver).
  * Uses the default CommTransactionManager.
  */
  public SerialComm(BSerialNetwork serialNetwork, CommReceiver rDriver, CommTransmitter tDriver)
  {
    super(serialNetwork, rDriver, tDriver);
  }

 /**
  * Constructor - initializes the SerialComm with a specified BSerialNetwork,
  * CommReceiver (receive driver), CommTransmitter (transmit driver), and
  * CommTransactionManager.
  */
  public SerialComm(BSerialNetwork serialNetwork, CommReceiver rDriver, CommTransmitter tDriver, CommTransactionManager manager)
  {
    super(serialNetwork, rDriver, tDriver, manager);
  }

/////////////////////////////////////////////////////////////////
// Comm port management
/////////////////////////////////////////////////////////////////

  /**  
   * Starts the serial transmit/receive drivers. Returns true if successfully started, false otherwise.
   * Opens the serial port and extracts the input and output streams,
   * and also starts the receive driver thread.
   */
  protected boolean started()
    throws Exception
  {
    try
    {                       
      BISerialService platSvc = (BISerialService)Sys.getService(BISerialService.TYPE);

      serialPort = ((BSerialNetwork)getNetwork()).getSerialPortConfig().open(getNetwork().getName());
  
      serialPort.enableReceiveTimeout(platSvc.getMinTimeout());

      in = serialPort.getInputStream();
      out = serialPort.getOutputStream();
    }
    catch(Exception e)
    {
      String errMsg = "Error opening and configuring the serial port";
      getNetwork().getLog().error(errMsg, e);
      
      if (in != null)
      {
        try
        {
          in.close();
        }
        catch (Exception x)
        {
          getNetwork().getLog().error("Unable to close serial input stream.", x);
        }
      }

      if (out != null)
      {
        try
        {
          out.close();
        }
        catch (Exception x)
        {
          getNetwork().getLog().error("Unable to close serial output stream.", x);
        }
      }

      if(serialPort != null) serialPort.close();
      
      throw e;
    }

    // Set the receive/transmit drivers streams and
    // start the receive thread
    getCommReceiver().setInputStream(in);
    getCommTransmitter().setOutputStream(out);
    rxThread = new Thread(getCommReceiver(), "SerialRcv:" + getNetwork().getName());
    getCommReceiver().setAlive(true);
    rxThread.start();
    rxThread.setPriority(Thread.NORM_PRIORITY);

    return true;
  }

  /**
   * Stops the serial transmit/receive drivers.
   * Closes the serial port and the input and output streams.
   */
  protected void stopped()
    throws Exception
  {
    getCommReceiver().setAlive(false);
    if((getCommReceiver()!=null) && (rxThread!=null)) rxThread.interrupt();

    if (in != null)
    {
      try
      {
        in.close();
      }
      catch (Exception e)
      {
        getNetwork().getLog().error("Unable to close serial input stream.", e);
      }
    }

    if (out != null)
    {
      try
      {
        out.close();
      }
      catch (Exception e)
      {
        getNetwork().getLog().error("Unable to close serial output stream.", e);
      }
    }

    if(serialPort != null)
    {
      serialPort.disableReceiveTimeout();
      serialPort.close();
    }
    in  = null;
    out = null;
  }

////////////////////////////////////////////////////////////
//  BasicDriver Api Overrides
////////////////////////////////////////////////////////////

  /**
   * Send a message using the message request/response service to
   * the communication medium.  Block the calling thread
   * until the response is obtained or the transaction times out.
   * Overridden here to enforce the inter message delay.
   *
   * @param msg a network request (in message form) to be
   *    sent to the output stream
   * @param responseTimeout the timeout to wait for a response for
   *    this request.
   * @param retryCount the number of retries to perform if the request
   *    fails (a timeout occurs).
   * @return Message the response received for the sent message
   *    if successful (or null if no response expected),
   *    otherwise an exception is thrown (i.e. timeout).
   */
  public Message transmit(Message msg, BRelTime responseTimeout, int retryCount)
    throws BasicException
  {
    if (msg == null)
      return null;

    // May not need this check - keep for now for legacy purposes
    if(!msg.getResponseExpected())
    {
      transmitNoResponse(msg);
      return null;
    }

    performInterMessageDelay();
    
    return super.transmit(msg, responseTimeout, retryCount);
  }

  /**
   * Send a message to the transmit driver and do not expect or wait
   * for a response from the receive driver.
   * Overridden here to enforce the inter message delay.
   *
   * @param msg a message to be sent to the output stream
   */
  public void transmitNoResponse(Message msg)
    throws BasicException
  {
    if(msg == null)
      return;

    if(!isCommStarted()) throw new BasicException("Communication handler service not started.");

    performInterMessageDelay();
    
    super.transmitNoResponse(msg);
  }

  /**
   * This is the access point for the receive driver to
   * pass its received unsolicited messages and/or 
   * response messages up to the communications handler for
   * processing.  Overridden here to enforce the Inter Message
   * Delay.
   *
   * @param msg the response/unsolicited message received 
   *    from the input stream.
   */
  public void receive(ReceivedMessage msg)
  {
    if (msg == null) return; // By default, don't do anything if received a null message.
    
    setReceivedMessageTicks(Clock.ticks());
    
    super.receive(msg);
  }

////////////////////////////////////////////////////////////
//  SerialComm methods
////////////////////////////////////////////////////////////

  /**
   * This method is called just before a message transmit in order to 
   * checks to see when the last time a message was received.
   * It waits the necessary inter message delay time if needed before returning.
   */
  protected void performInterMessageDelay()
  {
    long minDelay = ((BSerialNetwork)getNetwork()).getInterMessageDelay().getMillis();
    if (minDelay <= 0L) return; // short circuit
    
    long difference = Clock.ticks() - lastRecvMessageTicks;
    
    if (difference >= minDelay) return;
    
    // Otherwise sleep the appropriate time
    long sleepTime = Math.max((minDelay - difference), MIN_SLEEP_TIME);
    try { Thread.sleep(sleepTime); } catch (Exception e) { e.printStackTrace(); }
  }
  
  /**
   * Should be called with the current time ticks whenever a message
   * is received.
   */
  protected void setReceivedMessageTicks(long ticks)
  {
    lastRecvMessageTicks = ticks;
  }

/////////////////////////////////////////////////////
// Serial port convenience methods
//////////////////////////////////////////////////////
  
  /**
   * Returns the serial port used by this SerialComm
   */
  public BISerialPort getSerialPort()
  {
    return serialPort;
  }

////////////////////////////////////////////////////////////
//  Attributes of SerialComm
////////////////////////////////////////////////////////////
  private static final long MIN_SLEEP_TIME = 10L;
  
  private BISerialPort serialPort;
  private InputStream in;
  private OutputStream out;
  private Thread rxThread;
  private long lastRecvMessageTicks = 0L;

}