/*
 * Copyright 2002 Tridium, Inc. All Rights Reserved.
 */
package com.tridium.basicdriver.message;

import java.io.OutputStream;

/**
 * The Message class is the super class for all basic messages.
 * It usually boils down to an array of bytes.
 *
 * @author    Scott Hoye
 * @creation  25 Mar 02
 * @version   $Revision: 1$ $Date: 03/25/02 12:47:14 PM$  
 * @since     Niagara 3.0 basicdriver 1.0
 */
public abstract class Message
{
////////////////////////////////////////////////////////////
//  Constructor
////////////////////////////////////////////////////////////
  
 /**
  * Empty constructor
  */
  public Message ()
  {
  }
  

////////////////////////////////////////////////////////////
//  Access methods
////////////////////////////////////////////////////////////
  
 /** 
  * Sets the tag associated with this message.  
  * Used to match transactions (i.e. request/response transactions). 
  * Note to developers - if this tag value is kept the same (default is an Integer of -1) for 
  * all messages (both requests and responses), then the basicdriver 
  * will default and work one transaction at a time (request/immediate response).
  * Only change from default Integer -1 if you want to support multiple simultaneous 
  * transactions in which case this tag is used for request/response matching.
  */
  public void setTag(Object t) { tag = t; }
  
  /** 
  * Returns the tag associated with this message.  
  * Used to match transactions (i.e. request/response transactions). 
  * Note to developers - if this tag value is kept the same (default Integer -1) for 
  * all messages (both requests and responses), then the basicdriver 
  * will default and work one transaction at a time (request/immediate response).
  * Only change from default Integer -1 if you want to support multiple simultaneous 
  * transactions in which case this tag is used for request/response matching.
  */
  public Object getTag() { return tag; }
  
 /**
  * Returns true if a response is expected for this message, false if not.
  */  
  public boolean getResponseExpected() { return isResponseExpected; }

 /**
  * Sets whether a response is expected for this message.  True if a response
  * is expected, false if not.
  */    
  public void setResponseExpected(boolean expected)
  {
    isResponseExpected = expected;
  }
  
  /** 
  * Sets whether the response was successful.  True for successful, false
  * for unsuccessful.
  */
  public void setSuccessfulResponse(boolean b)  { success = b; }
  
  /** 
  * Returns whether the response was successful.  True for successful, false
  * for unsuccessful.
  */
  public boolean getSuccessfulResponse()    { return success; }
      
////////////////////////////////////////////////////////////
//  Override methods
////////////////////////////////////////////////////////////

  /**
  * Write the message to the given output stream. 
  * Must be overridden by subclasses. 
  */
  public abstract void write(OutputStream out);

  /**
  * Converts a basic response ReceivedMessage to a Message.
  * Normally, the received message passed in
  * just contains the raw bytes of a received message
  * that need to be interpreted.
  * Must be overridden by subclasses.
  */
  public abstract Message toResponse(ReceivedMessage resp);
  
  /**
   * Return a debug string for this message.
   */
  public String toDebugString()
  {
    StringBuffer sb = new StringBuffer();
    sb.append("Message = " + toString());
    sb.append("\n  Tag = " + getTag());
    sb.append("\n  Response Expected = " + getResponseExpected());
    sb.append("\n  Received Successful Response = " + getSuccessfulResponse());
    return sb.toString();
  }
  
///////////////////////////////////////////////////////////
//  Constants
////////////////////////////////////////////////////////////
  public static final Object DEFAULT_TAG = new Integer(-1);

////////////////////////////////////////////////////////////
//  Attributes
////////////////////////////////////////////////////////////
  private boolean isResponseExpected = true;
  private boolean success = true;
  private Object tag = DEFAULT_TAG;  // if keep this tag value the same (-1) for 
                                     // all messages (both requests and responses,
                                     // then the basicdriver will default and work
                                     // one transaction at a time (request/immediate response)
                                     // ONLY CHANGE FROM -1 if you want to support multiple
                                     // simultaneous transactions in which case the tag is used for
                                     // request/response matching.
}