/*
 * Copyright 2002 Tridium, Inc. All Rights Reserved.
 */
package com.tridium.basicdriver.comm;

import java.io.OutputStream;
import javax.baja.nre.util.ByteArrayUtil;
import com.tridium.basicdriver.message.Message;

/**
 * CommTransmitter is the basic driver for access (writing)
 * to the output stream.  Only one transaction can be
 * processed at a time.
 *
 * @author    Scott Hoye
 * @creation  26 Mar 02
 * @version   $Revision: 1$ $Date: 03/26/02 12:47:14 PM$
 * @since     Niagara 3.0 basicdriver 1.0
 */

public class CommTransmitter
  extends Comm.CommSupport
{

 /**
  * Default Constructor
  */
  public CommTransmitter() { }

////////////////////////////////////////////////////////////////
//  Public Access Methods
////////////////////////////////////////////////////////////////

 /**
  * Sets the output stream associated with this
  * transmit driver for sending messages.
  */
  public void setOutputStream(OutputStream out)
  {
    this.out = out;
  }

 /**
  * Gets the output stream associated with this
  * transmit driver for sending messages.
  */
  protected OutputStream getOutputStream()
  {
    return out;
  }

  /**
  * Provides support to write a Message
  * to the output stream.  Calls
  * Message.write(OutputStream out) with
  * the current output stream.
  */
  public void writeMessage(Message message)
  {
    try
    {
      synchronized(out)
      {
        if ((getComm().getNetwork() != null) && (getComm().getNetwork().getLog().isTraceOn()))
          getComm().getNetwork().getLog().trace ("**** Sending Message [" + getComm().getNetwork().getName() + "]: " + message.toDebugString());

        getComm().getCommReceiver().initReceiveState(message);
        writeBytesStart(out);
        message.write(out);
        writeBytesEnd(out);
      }
    }
    catch (Exception e)
    {
      getComm().handleFailedTransmit(message, e);
    }
  }

  /**
  * Method to be called if special processing
  * is required at the start of sending bytes to the output stream.
  * Should be overridden by subclasses as needed.  The default
  * is to do nothing.
  */
  public void writeBytesStart(OutputStream out)
    throws Exception
  {}

  /**
  * Method to be called if special processing
  * is required at the end of sending bytes to the output stream.
  * Should be overridden by subclasses as needed.  The default
  * is to do nothing.
  */
  public void writeBytesEnd(OutputStream out)
    throws Exception
  {}

  /**
  * Method to send a byte array to the output stream.
  */
  public void writeBytes(byte[] a)
    throws Exception
  {
    synchronized(out)
    {
      if ((getComm().getNetwork() != null) && (getComm().getNetwork().getLog().isTraceOn()))
      {
        System.out.println("****** Sending Bytes [" + getComm().getNetwork().getName() + "]: *******");
        ByteArrayUtil.hexDump(a);
        System.out.println("****************************************");
      }

      getComm().getCommReceiver().initReceiveState(a);
      writeBytesStart(out);
      if(a.length > 0)
        out.write(a);
      writeBytesEnd(out);
    }
  }

  /**
  * Method to write a single byte (expressed as an integer parameter) to the output stream.
  */
  public void writeByte(int value)
    throws Exception
  {
    byte[] a = { (byte)value };
    writeBytes(a);
  }

////////////////////////////////////////////////////////////////
//  Attributes of CommTransmitter
////////////////////////////////////////////////////////////////
  private OutputStream out;
}

