/*
 * Copyright 2002 Tridium, Inc. All Rights Reserved.
 */
package com.tridium.basicdriver.comm;

import java.util.Hashtable;
import com.tridium.basicdriver.message.Message;

/**
 * This class will handle transactions for the communication
 * handler (Comm).
 *
 * @author    Scott Hoye (taken from Robert Adams' LonTransactionManager)
 * @creation  01 Apr 02
 * @version   $Revision: 1$ $Date: 04/01/02 12:47:14 PM$
 * @since     Niagara 3.0 basicdriver 1.0
 */
public class CommTransactionManager
  extends Comm.CommSupport
{

////////////////////////////////////////////////////////////
//  Constructor
////////////////////////////////////////////////////////////

 /**
  * Constructor.
  */
  public CommTransactionManager()
  {
    CommTransaction transaction;
    for( int index = 0; index < MAX_TRANSACTIONS; index++)
    {
      transaction = new CommTransaction(index);
      transactionBuffers[index] = transaction;
    }
  }

////////////////////////////////////////////////////////////
//  Access methods
////////////////////////////////////////////////////////////

 /**
  * Sets the last transaction identifier used.
  */
  private synchronized void setLastCommTransactionId(int transactionId)
  {
    lastCommTransactionId = transactionId;
  }

 /**
  * Gets the last transaction identifier used.
  */
  private synchronized int getLastCommTransactionId()
  {
    return lastCommTransactionId;
  }

  /**
   * Get a reference to a CommTransaction object and lock its usage
   * until freeCommTransaction is called.  Use the given Message
   * as the request message for the CommTransaction.
   *
   * If no free CommTransaction object is available, then this method
   * will cause the calling thread to block until one becomes available.
   *
   * @return first available CommTransaction
   */
  public CommTransaction getCommTransaction(Message msg)
  {
    int startIndex;
    int buffersChecked = 0;

    if ( getLastCommTransactionId() >= (MAX_TRANSACTIONS - 1))
      startIndex = 0;
    else
      startIndex = getLastCommTransactionId() + 1;

    while (true)
    {
      CommTransaction sameTagCommTransaction = getCommTransactionMatch(msg.getTag());
      if ( (sameTagCommTransaction == null) || (!sameTagCommTransaction.isUsed()) )
      {

        for ( int index = startIndex; index < MAX_TRANSACTIONS; index++) // Check this, used to be MAX_TRANSACTIONS - 1
        {
          synchronized ( transactionBuffers[index])
          {
            if ( !transactionBuffers[index].isUsed())
            {
              tagsToCommTransactionIds.put(msg.getTag(),transactionIds[index]);
              setLastCommTransactionId(index);
              initCommTransaction(index, msg);
              return( transactionBuffers[index]);
            }
            else
            {
              buffersChecked++;
            }
          }
        }

        for ( int index = 0; index < startIndex; index++)
        {
          synchronized ( transactionBuffers[index])
          {
            if ( !transactionBuffers[index].isUsed())
            {
              tagsToCommTransactionIds.put(msg.getTag(),transactionIds[index]);
              setLastCommTransactionId(index);
              initCommTransaction(index,msg);
              return( transactionBuffers[index]);
            }
            else
            {
              buffersChecked++;
            }
          }
        }

        getComm().getNetwork().getLog().warning("CommTransactionManager.getCommTransaction " +
                "All transactions are used. Number checked is " + buffersChecked + "!!!");
      }

      synchronized ( waitObject)
      {
        try
        {
          waitObject.wait();
        }
        catch( InterruptedException e) {}

      }
    }
  }

 /**
  * Initializes a transaction with the given request message.
  */
  private void initCommTransaction(int index, Message msg)
  {
    CommTransaction transaction = transactionBuffers[index];

    transaction.setUsed(true);
    transaction.setRequestMessage(msg);
  }

  /**
   * Free a CommTransaction object for another transaction
   * to use.
   *
   * @param transaction the CommTransaction object to free.
   */
  public void freeCommTransaction(CommTransaction transaction)
  { // 05/13/2005 - LP - Responses for recycled tags are matching up incorrectly.
    tagsToCommTransactionIds.remove( transaction.getRequestMessage().getTag() );
    synchronized(transaction)
    {
      transaction.setRequestMessage(null);
      transaction.setResponseMessage(null);
      transaction.setUsed(false);
      transaction.setComplete(false);
    }

    synchronized( waitObject)
    {
      waitObject.notify();
    }
  }

  /**
   * Get a reference to a CommTransaction object by 
   * matching the given tag with the request message of
   * the appropriate CommTransaction.
   *
   * @param tag the tag used for matching request/response
   *    transactions. 
   */
  public CommTransaction getCommTransactionMatch(Object tag)
  {
    Integer transactionId = tagsToCommTransactionIds.get(tag);
    if (transactionId != null)
      return( transactionBuffers[transactionId.intValue()]);
    else
      return null;
  }

  /**
   * Cancels all waiting (outstanding) CommTransactions immediately
   */
  public void cancelAllOutstandingCommTransactions()
  {
    CommTransaction transaction;

    for( int index = 0; index < MAX_TRANSACTIONS; index++)
    {
      transaction = transactionBuffers[index];
      synchronized(transaction)
      {
        if ((transaction != null) && (transaction.isUsed()))
        {
          transaction.setResponseMessage( null);
          transaction.setComplete(true);
          transaction.notify();
        }
      }
    }
  }

////////////////////////////////////////////////////////////
//  Attributes of CommTransactionManager
////////////////////////////////////////////////////////////

  private static final int MAX_TRANSACTIONS = 15;
  private Object waitObject = new Object();
  private CommTransaction[] transactionBuffers = new CommTransaction[MAX_TRANSACTIONS];
  private Hashtable<Object, Integer> tagsToCommTransactionIds = new Hashtable<>(32);
  private int lastCommTransactionId = MAX_TRANSACTIONS - 1;
  private static final Integer[] transactionIds = { new Integer(0), new Integer(1),
                                                    new Integer(2), new Integer(3),
                                                    new Integer(4), new Integer(5),
                                                    new Integer(6), new Integer(7),
                                                    new Integer(8), new Integer(9),
                                                    new Integer(10), new Integer(11),
                                                    new Integer(12), new Integer(13),
                                                    new Integer(14) };
}
