/*
 * Copyright 2000 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.ui.event;

import javax.baja.sys.*;
import javax.baja.ui.*;

/**
 * BInputEvent is the base class for widget
 * events which are triggered by user input.
 *
 * @author    Brian Frank
 * @creation  21 Nov 00
 * @version   $Revision: 7$ $Date: 3/3/09 10:19:17 AM EST$
 * @since     Baja 1.0
 */
public abstract class BInputEvent
  extends BWidgetEvent
{ 

////////////////////////////////////////////////////////////////
// Type
////////////////////////////////////////////////////////////////  

  public Type getType() { return TYPE; }
  public static final Type TYPE = Sys.loadType(BInputEvent.class);

////////////////////////////////////////////////////////////////
// Modifiers
////////////////////////////////////////////////////////////////

  public static final int ALT_MASK     = 8;    //InputEvent.ALT_MASK
  public static final int ALT_GRAPH_MASK = 32; //InputEvent.ALT_GRAPH_MASK
  public static final int BUTTON1_MASK = 16;   //InputEvent.BUTTON1_MASK
  public static final int BUTTON2_MASK = 8;    //InputEvent.BUTTON2_MASK
  public static final int BUTTON3_MASK = 4;    //InputEvent.BUTTON3_MASK
  public static final int CTRL_MASK    = 2;    //InputEvent.CTRL_MASK
  public static final int META_MASK    = 4;    //InputEvent.META_MASK
  public static final int SHIFT_MASK   = 1;    //InputEvent.SHIFT_MASK

////////////////////////////////////////////////////////////////
// Constructors
////////////////////////////////////////////////////////////////

  /**
   * Construct a new widget event for the
   * specified source widget.
   */
  public BInputEvent(int id, BWidget source, int modifiers)
  {   
    super(id, source);
    this.modifiers = modifiers;
    this.when = Clock.millis();
  }

  public BInputEvent(int id, BWidget source, long when, int modifiers)
  {   
    super(id, source);
    this.modifiers = modifiers;
    this.when = when;
  }

  /**
   * No arg constructor
   */
  public BInputEvent()
  {
  }

////////////////////////////////////////////////////////////////
// Methods
////////////////////////////////////////////////////////////////  

  /**
   * Get the modifier mask for this event.
   */
  public int getModifiers()
  {
    return modifiers;
  }

  /**
   * Get the timestamp in millis when the event occurred
   * @return
   */
  public long getWhen()
  {
    return when;
  }
  
  /**
   * Returns whether the alt modifier is set.
   */
  public boolean isAltDown()
  {
    return (modifiers & ALT_MASK) != 0;
  }

  /**
   * Returns whether the alt graph modifier is set.
   */
  public boolean isAltGraphDown()
  {
    return (modifiers & ALT_GRAPH_MASK) != 0;
  }

  /**
   * Returns whether the control modifier is set.
   */
  public boolean isControlDown()
  {
    return (modifiers & CTRL_MASK) != 0;
  }

  /**
   * Returns whether the meta modifier is set.
   */
  public boolean isMetaDown()
  {
    return (modifiers & META_MASK) != 0;
  }

  /**
   * Returns whether the shift modifer is set.
   */
  public boolean isShiftDown()
  {
    return (modifiers & SHIFT_MASK) != 0;
  }

  /**
   * Get a debug string for the modifiers set.
   */
  public String modifiersToString()
  {                      
    String s = "";
    if (isAltDown())      s += "Alt ";
    if (isAltGraphDown()) s += "AltGraph ";
    if (isControlDown())  s += "Ctrl ";
    if (isMetaDown())     s += "Meta ";
    if (isShiftDown())    s += "Shift ";
    return s.trim();
  }

  /**
   * Return true if this key event was consumed.
   */
  public boolean isConsumed()
  {
    return consumed;
  }
  
  /**
   * This method is called to consume the key event
   * so that it is not propogated to other controls 
   * such as focus traversal, accelerators, or 
   * default buttons.
   */
  public void consume()
  {
    consumed = true;
  }               
  
////////////////////////////////////////////////////////////////
// Attributes
////////////////////////////////////////////////////////////////  
  
  private int modifiers;
  private boolean consumed;
  private long when;
}
