/*
 * Copyright 2005 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.security;

import java.io.*;

import javax.baja.category.*;
import javax.baja.sys.*;

import javax.baja.nre.util.TextUtil;

/**
 * BPermissionsMap is used to grant permissions to categories.
 *
 * @author    Brian Frank
 * @creation  19 Feb 05
 * @version   $Revision: 12$ $Date: 2/16/09 9:24:11 AM EST$
 * @since     Baja 1.0
 */
public final class BPermissionsMap
  extends BSimple
  implements BIUnlinkable
{

////////////////////////////////////////////////////////////////
// Factory
////////////////////////////////////////////////////////////////
  
  /**
   * Make map where category number maps to array index.  
   * The first item (permissions[0]) is unused.  If any
   * items in the array are null, then BPermissions.none
   * is assumed.
   */
  public static BPermissionsMap make(BPermissions[] permissions)
  {                  
    // short circuit for empty map
    if (permissions.length <= 1)
      return DEFAULT;
                          
    // make safe copy of array
    permissions = permissions.clone();
    
    // construct new instance
    return (BPermissionsMap)(new BPermissionsMap(permissions, null).intern());
  }

  /**
   * Create a new BPermissionsMap from the bitwise OR of this BPermissionsMap
   * instance and the specified BPermissionsMap.
   *
   * @since Niagara 3.5
   */  
  public BPermissionsMap or(BPermissionsMap other)
  {
    // short circuit for super user
    if (this.isSuperUser() || other.isSuperUser())
      return BPermissionsMap.SUPER_USER;

    BPermissions[] newPermissions =
      new BPermissions[Math.max(this.permissions.length, other.permissions.length)];
    
    for (int i = 0; i < newPermissions.length; i++)
    {
      BPermissions p1 = i < this.permissions.length ? this.permissions[i] : BPermissions.none;
      BPermissions p2 = i < other.permissions.length ? other.permissions[i] : BPermissions.none;
      
      if (p1 == null) p1 = BPermissions.none;
      if (p2 == null) p2 = BPermissions.none;
      newPermissions[i] = p1.or(p2);
    }
    
    return make(newPermissions);
  }
  
  /**
   * Create a new BPermissionsMap from the bitwise AND of this BPermissionsMap
   * instance and the specified BPermissionsMap.
   *
   * @since Niagara 3.5
   */  
  public BPermissionsMap and(BPermissionsMap other)
  {
    // short circuit for super user
    if (this.isSuperUser() && other.isSuperUser())
      return BPermissionsMap.SUPER_USER;
    
    BPermissions[] newPermissions =
      new BPermissions[Math.max(this.permissions.length, other.permissions.length)];
    
    for (int i = 0; i < newPermissions.length; i++)
    {
      BPermissions p1 = i < this.permissions.length ? this.permissions[i] : BPermissions.none;
      BPermissions p2 = i < other.permissions.length ? other.permissions[i] : BPermissions.none;
      
      // if super user, use all permissions in mask
      if (this.isSuperUser()) p1 = BPermissions.all;
      if (other.isSuperUser()) p2 = BPermissions.all;
      
      if (p1 == null) p1 = BPermissions.none;
      if (p2 == null) p2 = BPermissions.none;
      newPermissions[i] = p1.and(p2);
    }
    
    return make(newPermissions);
  }

////////////////////////////////////////////////////////////////
// Constructor
////////////////////////////////////////////////////////////////
  
  /**
   * Private constructor.
   */
  private BPermissionsMap(BPermissions[] permissions, String string)
  {
    this.permissions = permissions;                                
    this.string = string;
  }

////////////////////////////////////////////////////////////////
// Access
////////////////////////////////////////////////////////////////  
  
  /**
   * Return if this is the super user permission map which
   * is automatically granted all permissions in all categories.
   */
  public boolean isSuperUser()
  {
    return this == SUPER_USER;
  }                       
  
  /**
   * Get the number of category/permissions mappings.
   */
  public int size()
  { 
    if (size < 0)
    {           
      int x = permissions.length-1;
      while(x > 0)
      {
        BPermissions p = permissions[x];
        if (p != null && p.getMask() != 0) break;
        x--;
      }
      size = x + 1;
    }
    return size;
  }

  /**
     * Given a category index, return the mapped BPermissions
     * or return BPermissions.none if no mapping configured.
     * If this is the super user, then always return BPermissions.all.
     */
  public BPermissions getPermissions(int categoryIndex)
  {                                   
    // no such thing as category 0
    if (categoryIndex < 1) throw new IllegalArgumentException("Category index < 1");
   
    // super is always all
    if (this == SUPER_USER) return BPermissions.all;
    
    // out of range is always none
    if (categoryIndex >= permissions.length)return BPermissions.none;
    
    // otherwise map to zero
    BPermissions p = permissions[categoryIndex]; 
    if (p == null) return BPermissions.none;
    return p;
  }

  /**
   * Given a category mask, compute the applied permissions which
   * are a bitwise "OR" of the permissions assigned to each union
   * category and a bitwise "AND" of the permissions assigned to
   * each intersection category from the specified mask.  If this
   * is the super user instance then always return BPermissions.all.
   */
  public BPermissions getCategoryPermissions(BCategoryMask categoryMask)
  {                                     
    if (this == SUPER_USER) return BPermissions.all;
    BCategoryService catService = null;
    try
    {
      catService = (BCategoryService)Sys.getService(BCategoryService.TYPE);
    }
    catch(ServiceNotFoundException ex)
    {
    }
    catch(NotRunningException ex)
    {
    }
    
    int orMask = 0;
    int andMask = 0xFFFFFFFF;
    boolean applyAnd = false;
    boolean applyOr = false;
    
    int len = permissions.length;
    for (int i=1; i<len; ++i)
    {
      if (categoryMask.get(i))
      {
        BCategoryMode mode = null;
        if (catService != null)
        {
          BCategory cat = null;
          try
          {
            cat = catService.getCategory(i);
          }
          catch (NotRunningException e) 
          {
          }
          if (cat != null) mode = cat.getMode();
        }
        
        BPermissions p = permissions[i];

        if ((mode == null) || (mode == BCategoryMode.union))
        {
          if (p == null) continue;
          applyOr = true;
          orMask |= p.getMask();
        }
        else
        {
          // If an intersection category is assigned to the component, but
          // the user has null permissions on that category, then we can
          // return no permissions immediately
          if (p == null) return BPermissions.none;
          applyAnd = true;
          andMask &= p.getMask();
        }
      }
    }   
    
    if (applyAnd)
    {
      if (applyOr)
        return BPermissions.make(orMask & andMask);
      else
        return BPermissions.make(andMask);
    }
    else
      return BPermissions.make(orMask);
  }

////////////////////////////////////////////////////////////////
// BSimple
////////////////////////////////////////////////////////////////  

  /**
   * BPermissionsMap uses its encodeToString() value's hash code.
   * 
   * @since Niagara 3.4
   */
  public int hashCode()
  {
    try
    {
      if (hashCode == -1) 
        hashCode = encodeToString().hashCode();
      return hashCode;
    }
    catch(Exception e) 
    { 
      return System.identityHashCode(this);
    }
  }
  
  public boolean equals(Object obj)
  {                    
    if (obj instanceof BPermissionsMap)
    {                                 
      BPermissionsMap x = (BPermissionsMap)obj;
      
      // special super user check
      if (this.isSuperUser()) return x.isSuperUser();
      if (x.isSuperUser()) return this.isSuperUser();
      
      // figure out one with longer array 
      int max = Math.max(permissions.length, x.permissions.length);

      // compare indices
      for(int i=1; i<max; ++i)
        if (!getPermissions(i).equals(x.getPermissions(i))) return false;

      // at this point we know they are equal
      return true;
    }
    return false;
  }

  public void encode(DataOutput out)
    throws IOException
  {                      
    out.writeUTF(encodeToString());
  }
  
  public BObject decode(DataInput in)
    throws IOException
  {                       
    return decodeFromString(in.readUTF());
  }

  public String encodeToString()
  {         
    if (string == null)
    {       
      StringBuffer s = new StringBuffer();
      for(int i=1; i<permissions.length; ++i)
      {               
        BPermissions p = permissions[i];
        if (p == null || p.getMask() == 0) continue;
        if (s.length() > 0) s.append(';');
        s.append(i).append('=').append(p.encodeToString());          
      }
      string = s.toString();
    }
    return string;
  }
  
  public BObject decodeFromString(String s)
    throws IOException
  { 
    // special string encoding         
    if (s.equals("super")) return SUPER_USER;
    if (s.equals("")) return DEFAULT;

    // working array
    BPermissions[] permissions = new BPermissions[8];

    // parse "index=permission;" pairs
    String[] tokens = TextUtil.split(s, ';');
    for(int i=0; i<tokens.length; ++i)
    {                  
      // parse index and permissions
      String token = tokens[i];
      int eq = token.indexOf('=');
      int index = Integer.parseInt(token.substring(0, eq));
      BPermissions p = BPermissions.make(token.substring(eq+1));
      
      // check capacity             
      if (index >= permissions.length)
      {           
        BPermissions[] temp = new BPermissions[Math.max(index+4, permissions.length*2)];
        System.arraycopy(permissions, 0, temp, 0, permissions.length);
        permissions = temp;
      }
      
      // assign into specified index
      permissions[index] = p;
    }

    return new BPermissionsMap(permissions, s).intern();
  }  

////////////////////////////////////////////////////////////////
// Type
////////////////////////////////////////////////////////////////  
  
  /**
   * The SUPER_USER instance automatically is granted 
   * all permissions in any category.
   */
  public static final BPermissionsMap SUPER_USER = new BPermissionsMap(new BPermissions[0], "super");
  
  /**
   * This is default instance is the empty permissions map.
   */
  public static final BPermissionsMap DEFAULT = new BPermissionsMap(new BPermissions[0], "");

  public Type getType() { return TYPE; }
  public static final Type TYPE = Sys.loadType(BPermissionsMap.class);

////////////////////////////////////////////////////////////////
// Attributes
////////////////////////////////////////////////////////////////

  private BPermissions[] permissions;  // could have null items
  private int size = -1;               // cache for size()
  private String string;               // cached string encoding
  private int hashCode = -1;
   
}

