/*
 * Copyright 2015 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.security;

import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.Arrays;

import javax.baja.agent.BIAgent;
import javax.baja.nre.annotations.AgentOn;
import javax.baja.nre.annotations.NiagaraType;
import javax.baja.nre.util.ByteArrayUtil;
import javax.baja.nre.util.SecurityUtil;
import javax.baja.nre.util.TextUtil;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;
import com.tridium.nre.auth.NiagaraStationAlgorithmBundle;
import com.tridium.nre.auth.Pbkdf2;
import com.tridium.nre.auth.Pbkdf2AlgorithmBundle;
import com.tridium.nre.security.KeyDerivationAlgorithmBundle;
import com.tridium.nre.security.PBEEncodingKey;
import com.tridium.nre.security.SecretChars;

@NiagaraType(agent = @AgentOn(types = {"baja:Password"}))
public final class BPbkdf2HmacSha256PasswordEncoder
    extends BAbstractPasswordEncoder
    implements BIAgent
{
/*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
/*@ $javax.baja.security.BPbkdf2HmacSha256PasswordEncoder(119196451)1.0$ @*/
/* Generated Fri Apr 10 14:04:24 EDT 2015 by Slot-o-Matic (c) Tridium, Inc. 2012 */

////////////////////////////////////////////////////////////////
// Type
////////////////////////////////////////////////////////////////
  
  @Override
  public Type getType() { return TYPE; }
  public static final Type TYPE = Sys.loadType(BPbkdf2HmacSha256PasswordEncoder.class);

/*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/
////////////////////////////////////////////////////////////////
// Constructor
////////////////////////////////////////////////////////////////
  public BPbkdf2HmacSha256PasswordEncoder(){}

////////////////////////////////////////////////////////////////
// BAbstractPasswordEncoder
////////////////////////////////////////////////////////////////

  @Override
  public void encode(SecretChars password)
    throws Exception
  {
    byte[] salt = new byte[SALT_LENGTH];
    new SecureRandom().nextBytes(salt);
    this.salt = ByteArrayUtil.toHexString(salt);
    this.iterationCount = ITERATION_COUNT;
    byte[] key = Pbkdf2.deriveKey(salt, iterationCount, password.get(), ALGORITHM_BUNDLE);
    this.key = TextUtil.bytesToHexString(key);
  }

  @Override
  public void parse(String key) throws IllegalArgumentException
  {
    String[] data = ALGORITHM_BUNDLE.decode(key);
            
    salt = data[Pbkdf2AlgorithmBundle.SALT_INDEX];
    iterationCount = Integer.parseInt(data[Pbkdf2AlgorithmBundle.ITERATION_COUNT_INDEX]);
    this.key = data[Pbkdf2AlgorithmBundle.VALIDATION_HASH_INDEX];
  }

  @Override
  public String getValue()
  {
    String[] data = new String[ALGORITHM_BUNDLE.getDataElementCount()];
    data[Pbkdf2AlgorithmBundle.SALT_INDEX] = salt;
    data[Pbkdf2AlgorithmBundle.ITERATION_COUNT_INDEX] = String.valueOf(iterationCount);
    data[Pbkdf2AlgorithmBundle.VALIDATION_HASH_INDEX] = key;
    return ALGORITHM_BUNDLE.encode(data);
  }

  @Override
  public String getEncodingType()
  {
    return ENCODING_TYPE;
  }

  @Override
  public boolean isReversible()
  {
    return false;
  }

  @Override
  public boolean validate(SecretChars password)
  {
    try
    {
      byte[] key = Pbkdf2.deriveKey(ByteArrayUtil.hexStringToBytes(salt), iterationCount, password.get(), ALGORITHM_BUNDLE);
      String tkey = TextUtil.bytesToHexString(key);
  
      return SecurityUtil.equals(tkey, this.key);
    }
    catch(Exception e)
    {
      e.printStackTrace();
      return false;
    }
  }

  @Override
  public String getEncodedValue()
  {
    return getValue();
  }
  
////////////////////////////////////////////////////////////////
// BPbkdf2HmacSha256-specific
////////////////////////////////////////////////////////////////

  public void encode(String password, String salt, int iterationCount)
    throws Exception
  {
    this.salt = salt;
    this.iterationCount = iterationCount;
    byte[] hexSalt = ByteArrayUtil.hexStringToBytes(salt);
    byte[] key = Pbkdf2.deriveKey(hexSalt, iterationCount, password, ALGORITHM_BUNDLE);
    this.key = TextUtil.bytesToHexString(key);
  }

  public String getSalt()
  {
    return salt;
  }
  
  public int getIterationCount()
  {
    return iterationCount;
  }

  public byte[] getKey()
  {
    return ByteArrayUtil.hexStringToBytes(key);
  }

  public static BPbkdf2HmacSha256PasswordEncoder makeFake(String userName)
  {
    byte[] key = new byte[ALGORITHM_BUNDLE.getKeyLength()];
    new SecureRandom().nextBytes(key);
    BPbkdf2HmacSha256PasswordEncoder fake = new BPbkdf2HmacSha256PasswordEncoder();
    
    fake.key = TextUtil.bytesToHexString(key);
    MessageDigest md;
    try
    {
      md = MessageDigest.getInstance("Sha-256");
      byte[] strBytes = userName.getBytes(StandardCharsets.UTF_8);

      //NCCB-71466: adding Secure random bytes : FAKE_SALT_PREPEND_BYTES to the username before creating the salt
      byte[] combinedArray = ByteBuffer.allocate(SALT_LENGTH + strBytes.length).put(FAKE_SALT_PREPEND_BYTES).put(strBytes).array();

      byte[] salt = md.digest(combinedArray);
      fake.salt = TextUtil.bytesToHexString(Arrays.copyOf(salt, SALT_LENGTH));
      fake.iterationCount = ITERATION_COUNT;
      return fake;
    }
    catch (NoSuchAlgorithmException e)
    {
      throw new UnsupportedOperationException(e.getLocalizedMessage());
    }
  }
  
////////////////////////////////////////////////////////////////
// Attributes 
////////////////////////////////////////////////////////////////
  private String salt;
  private String key;
  private int iterationCount;
  
  public static final KeyDerivationAlgorithmBundle ALGORITHM_BUNDLE = NiagaraStationAlgorithmBundle.getInstance();
  public static final String ENCODING_TYPE = ALGORITHM_BUNDLE.getAlgorithmName();

  private static final int ITERATION_COUNT = PBEEncodingKey.DEFAULT_VALIDATION_ITERATION_COUNT;
  private static final int SALT_LENGTH = 16;
  private static final byte[] FAKE_SALT_PREPEND_BYTES = new byte[SALT_LENGTH];
  //NCCB-71466: Secure random bytes generator
  static
  {
    new SecureRandom().nextBytes(FAKE_SALT_PREPEND_BYTES);
  }
}
