/*
 * Copyright 2015 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.security;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.AccessController;
import java.security.InvalidParameterException;
import java.security.MessageDigest;
import java.security.PrivilegedExceptionAction;
import java.util.Base64;
import java.util.HashMap;
import java.util.Map;

import javax.baja.io.BIContextEncodable;

import javax.baja.nre.util.TextUtil;
import javax.baja.sys.BAbsTime;
import javax.baja.sys.BObject;
import javax.baja.sys.BSimple;
import javax.baja.sys.BajaRuntimeException;
import javax.baja.sys.Context;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;

/**
 * A history for storing and managing hashed password values.
 *
 * @author    Lee Adcock
 * @creation  28 Oct 2011
 * @version   $Revision: 46$ $Date: 5/14/08 3:14:00 PM EDT$
 * @since     Niagara 3.7
 */
public final class BPasswordHistory
  extends BSimple implements BIContextEncodable
{

  public static final BPasswordHistory DEFAULT = new BPasswordHistory();

////////////////////////////////////////////////////////////////
//Type
////////////////////////////////////////////////////////////////

  public Type getType() { return TYPE; }
  public static final Type TYPE = Sys.loadType(BPasswordHistory.class);

////////////////////////////////////////////////////////////////
//PasswordHistory
////////////////////////////////////////////////////////////////

  /**
   * Create an empty password history instance.
   */
  public BPasswordHistory()
  {
    this.passwords = new BPassword[0];
  }

  /**
   * Create a new BPasswordHistory with the contents of the current
   * history, plus the additional BPassword.  The size of the password
   * history will be increased by one.
   * @throws InvalidParameterException when the specified history is already in the history
   */
  public BPasswordHistory add(String username, BPassword password)
  {
    return add(username, password, passwords.length+1);
  }

  /**
   * Create a new BPasswordHistory with the contents of the current
   * history, plus the additional BPassword.  The size of the resulting
   * history will be no larger than the defined maximumHistorySize.  If
   * the size of the existing history plus one is larger than the
   * maximumHistorySize, the oldest histories will be truncated.
   * @throws InvalidParameterException when the specified history is already in the history
   */
  public BPasswordHistory add(String username, BPassword password, int maximumHistorySize)
  {
    // Sanity checks
    if (!password.getPasswordEncoder().isReversible())
    {
      throw new BajaRuntimeException("Cannot add " + password.getEncodingType() + " encoded passwords to BPasswordHistory.");
    }
    if(maximumHistorySize<0) throw new InvalidParameterException("History size must be zero or greater.");
    if(maximumHistorySize==0) return BPasswordHistory.DEFAULT;
    BPassword convertedPassword = getConvertedPassword(password);
    if(contains(username, password, convertedPassword, maximumHistorySize))
    {
      throw new BajaRuntimeException("Duplicate password.");
    }

    // Create a new password history instance to return
    BPasswordHistory newHistory = new BPasswordHistory();
    newHistory.passwords = new BPassword[Math.min(maximumHistorySize, passwords.length+1)];
    
    // Add the new history record
    newHistory.passwords[0] = convertedPassword;
    // Copy the password records from the old history to the new history,
    // checking to make sure the new password doesn't already exist.
    if(passwords.length>0)
    {
      System.arraycopy(passwords, 0, newHistory.passwords, 1, Math.min(maximumHistorySize - 1, passwords.length));
    }

    newHistory.initSaltAndIterationCount();
    return newHistory;
  }

  private String getOldPasswordHash(String username, BPassword password)
  {
    String value;
    try
    {
      value = AccessController.doPrivileged((PrivilegedExceptionAction<String>)password.getPasswordEncoder()::getValue);
    }
    catch (Exception e1)
    {
      throw new SecurityException();
    }
    try
    {
      MessageDigest md = MessageDigest.getInstance("SHA-1");
      md.update(value.getBytes());
      md.update((TextUtil.bytesToHexString(md.digest())+"H"+username).getBytes());
      return TextUtil.bytesToHexString(md.digest());
    }
    catch (Exception e)
    {
      // This should never happen
      return String.valueOf(value.hashCode());
    }
  }

  /**
   * Check whether the history contains the specified password.  Returns
   * true if the password exists, false if it does not.
   */
  public boolean contains(String username, BPassword password)
  {
    return contains(username, password, passwords.length);
  }

  /**
   * Check whether the most history contains the specified password within
   * the most recent passwords, as specified by the historySize parameter.
   * Returns true if the password exists within the specified number of
   * records, false if it does not.
   */
  public boolean contains(String username, BPassword password, int historySize)
  {
    return contains(username, password, null, historySize);
  }

  /**
   * Check whether the most history contains the specified password within
   * the most recent passwords, as specified by the historySize parameter.
   * The prehashed password parameter allows the caller to provide a hashed
   * version of password. The purpose of this is to avoid re-hashing a password
   * to compare it with existing hashes in the history, as this can be expensive.
   * Returns true if the password exists within the specified number of
   * records, false if it does not.
   * 
   */
  public boolean contains(String username, BPassword password, BPassword prehashedPassword, int historySize)
  {
    if (password.getPasswordEncoder().isReversible())
    {
      // Create the stored password hash
      String encodedValue = null;
      if (prehashedPassword != null)
      {
        encodedValue = prehashedPassword.getPasswordEncoder().getEncodedValue();
      }
      
      // Search for the stored value, return true if found.
      for(int i=0; i<Math.min(passwords.length, historySize); i++)
      {
        if (passwords[i].getPasswordEncoder().isReversible())
        {
          try
          {
            if (AccessController.doPrivileged((PrivilegedExceptionAction<String>)passwords[i].getPasswordEncoder()::getValue).equals(
                 getOldPasswordHash(username, password)))
            {
              return true;
            }
          }
          catch (Exception e)
          {
            throw new SecurityException();
          }
        }
        else //noinspection StatementWithEmptyBody
          if (passwords[i].getEncodingType().equals(BPbkdf2HmacSha256PasswordEncoder.ENCODING_TYPE))
        {
          if (encodedValue == null)
          {
            encodedValue = getConvertedPassword(password).getPasswordEncoder().getEncodedValue();
          }
          if (passwords[i].getPasswordEncoder().getEncodedValue().equals(encodedValue))
          {
            return true;
          }
        }
        else
        {
          //Unsupported Stored Password Encoding
        }
      }
    }
    else
    {
      //password is already hashed and we can't compare to anything in the history
      return true;
    }
    
    // Password not found in history
    return false;
  }

  /**
   * Get the size of the password history.  This will return the total
   * number of passwords records within the history.
   */
  public int getSize()
  {
    return passwords.length;
  }

////////////////////////////////////////////////////////////////
// Simple / BIContextEncodable
////////////////////////////////////////////////////////////////

  public void encode(DataOutput encoder) throws IOException
  {
    encode(encoder, null);
  }
  
  public void encode(DataOutput encoder, Context context) throws IOException
  {
    // Encode the history size
    encoder.writeInt(passwords.length);

    // Encode each password in the history
    for (BPassword password : passwords)
    {
      encoder.writeUTF(password.encodeToString(context));
    }
  }

  public BObject decode(DataInput decoder) throws IOException
  {
    return decode(decoder, null);
  }
  
  public BObject decode(DataInput decoder, Context context) throws IOException
  {
    // Create a new BPasswordHistory instance
    BPasswordHistory passwordHistory = new BPasswordHistory();

    // Initialize password storage
    passwordHistory.passwords = new BPassword[decoder.readInt()];

    // Populate the passwords
    for(int i=0; i<passwordHistory.passwords.length; i++)
    {
      passwordHistory.passwords[i] = (BPassword)BPassword.DEFAULT.decodeFromString(decoder.readUTF());
    }
    passwordHistory.initSaltAndIterationCount();
    
    return passwordHistory;
  }

  public String encodeToString() throws IOException
  {
    return encodeToString(null);
  }
  
  public String encodeToString(Context context) throws IOException
  {
    StringBuilder buffer = new StringBuilder();
    if(passwords.length>0)
    {
      // Encode each password in the history, separated by pipes,
      // with one trailing pipe.
      for (BPassword password : passwords)
      {
        buffer.append(password.encodeToString(context)).append("|");
      }
      //remove the trailing |
      buffer.deleteCharAt(buffer.length()-1);
    }
    return Base64.getEncoder().encodeToString(buffer.toString().getBytes());
  }

  public BObject decodeFromString(String s) throws IOException
  {
    return decodeFromString(s, null);
  }
  
  public BObject decodeFromString(String s, Context context) throws IOException
  {
    BPasswordHistory passwordHistory = new BPasswordHistory();
    String[] passwordStrs = TextUtil.split(new String(Base64.getDecoder().decode(s), StandardCharsets.UTF_8), '|');
    passwordHistory.passwords = new BPassword[passwordStrs.length];
    for (int i=0; i<passwordStrs.length; i++)
    {
      //check for old style password (which is a Hashed String, not cryptUtil'd)
      // some might be, some might not
      try
      {
        passwordHistory.passwords[i] = (BPassword)BPassword.DEFAULT.decodeFromString(passwordStrs[i]);
      }
      catch(IOException ioe)
      {
        passwordHistory.passwords[i] = BPassword.make(passwordStrs[i], context);
      }
    }
    passwordHistory.initSaltAndIterationCount();
    return passwordHistory;
  }

  public boolean equals(Object obj)
  {
    // Is this a BPasswordHistory instance? If not, these are
    // not equal.
    if(!(obj instanceof BPasswordHistory))
    {
      return false;
    }
    BPasswordHistory passwordHistory = (BPasswordHistory)obj;

    // Is the history size the same? If not, these are not equal.
    if(passwordHistory.passwords.length != this.passwords.length)
    {
      return false;
    }

    for(int i=0; i<passwordHistory.passwords.length; i++)
    {
      if (!passwordHistory.passwords[i].equals(this.passwords[i]))
      {
        return false;
      }
    }

    // We have a match!
    return true;
  }
  
  private void initSaltAndIterationCount()
  {
    if (salt == null)
    {
      for (BPassword password : passwords)
      {
        if (password.getEncodingType().equals(BPbkdf2HmacSha256PasswordEncoder.ENCODING_TYPE))
        {
          BPbkdf2HmacSha256PasswordEncoder encoder = 
            (BPbkdf2HmacSha256PasswordEncoder)password.getPasswordEncoder();
          
          salt = encoder.getSalt();
          iterationCount = encoder.getIterationCount();
          return;
        }
      }
    }
  }
  
  /**
   * Converts the password to a Pbdkdf2 password. Must be in a reversible format.
   */
  private BPassword getConvertedPassword(BPassword password)
  {
    try
    {
      // If we have a password with a salt in the history already, use the same salt and iteration count
      if (salt != null && iterationCount != 0)
      {
        BPbkdf2HmacSha256PasswordEncoder encoder = new BPbkdf2HmacSha256PasswordEncoder();
        encoder.encode(AccessController.doPrivileged((PrivilegedExceptionAction<String>)password.getPasswordEncoder()::getValue), salt, iterationCount);
        return BPassword.make(encoder);
      }
      // If we have an empty history, just create a new password with a random salt
      else
      {
        return BPassword.make(AccessController.doPrivileged((PrivilegedExceptionAction<String>)password.getPasswordEncoder()::getValue), BPbkdf2HmacSha256PasswordEncoder.ENCODING_TYPE);
      }
    }
    catch(Exception e)
    {
      throw new SecurityException();
    }    
  }
  
////////////////////////////////////////////////////////////////
//Attributes
////////////////////////////////////////////////////////////////

  private BPassword[] passwords;
  private String salt = null;
  private int iterationCount = 0;
}
