/*
 * Copyright 2018 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.security;

import java.security.AccessControlException;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.security.PrivilegedExceptionAction;
import java.security.SecureRandom;
import java.util.Objects;
import java.util.Optional;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.baja.nre.annotations.NiagaraType;
import javax.baja.nre.util.ByteArrayUtil;
import javax.baja.nre.util.SecurityUtil;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;

import com.tridium.crypto.core.bundle.CryptographicAlgorithmBundle;
import com.tridium.nre.security.AbstractAesAlgorithmBundle;
import com.tridium.nre.security.Aes256PasswordManager;
import com.tridium.nre.security.EncryptionAlgorithmBundle;
import com.tridium.nre.security.ISecretBytesSupplier;
import com.tridium.nre.security.ISecurityInfoProvider;
import com.tridium.nre.security.KeyRingPermission;
import com.tridium.nre.security.SecretBytes;
import com.tridium.nre.security.SecurityConstants;
import com.tridium.nre.security.SecurityInitializer;

/**
 * @author Patrick sager
 * @creation 1/3/2018
 * @since Niagara 4.6
 */
@NiagaraType
public abstract class BAbstractAes256PasswordEncoder
  extends BReversiblePasswordEncoder
{
/*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
/*@ $javax.baja.security.BAbstractAes256PasswordEncoder(2979906276)1.0$ @*/
/* Generated Wed Jan 03 12:45:58 EST 2018 by Slot-o-Matic (c) Tridium, Inc. 2012 */

////////////////////////////////////////////////////////////////
// Type
////////////////////////////////////////////////////////////////
  
  @Override
  public Type getType() { return TYPE; }
  public static final Type TYPE = Sys.loadType(BAbstractAes256PasswordEncoder.class);

/*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

////////////////////////////////////////////////////////////////
// BAbstractPasswordEncoder
////////////////////////////////////////////////////////////////

  @Override
  public void encode(SecretBytes passwordBytes) throws Exception
  {
    byte[] ivBytes = new byte[16];
    new SecureRandom().nextBytes(ivBytes);
    iv = ByteArrayUtil.toHexString(ivBytes);
    String aesTransformation = getAesTransformation();

    if (usesExternalEncryptionKey())
    {
      if (getEncryptionKey().isPresent())
      {
        cipher = ByteArrayUtil.toHexString(Aes256PasswordManager.encrypt(passwordBytes.get(), ivBytes, getEncryptionKey().get(), aesTransformation));
      }
      else
      {
        throw new MissingEncodingKeyException();
      }
    }
    else
    {
      cipher = AccessController.doPrivileged((PrivilegedExceptionAction<String>)() -> {
        ISecurityInfoProvider provider = SecurityInitializer.getInstance().getSecurityInfoProvider();
        return ByteArrayUtil.toHexString(Aes256PasswordManager.getManager(
          provider.getKeyRing(), keyAlias).encrypt(passwordBytes.get(), ivBytes, aesTransformation));
      });
    }
  }

  @Override
  public SecretBytes getSecretBytes() throws Exception
  {
    String aesTransformation = getAesTransformation();
    if (usesExternalEncryptionKey())
    {
      if (getEncryptionKey().isPresent())
      {
        return Aes256PasswordManager.decryptSecret(getEncryptionKey().get(), ByteArrayUtil.hexStringToBytes(cipher), ByteArrayUtil.hexStringToBytes(iv), aesTransformation);
      }
      else
      {
        throw new MissingEncodingKeyException();
      }
    }
    else
    {
      ISecurityInfoProvider provider = AccessController.doPrivileged((PrivilegedAction<ISecurityInfoProvider>)() ->
        SecurityInitializer.getInstance().getSecurityInfoProvider());

      try
      {
        return Aes256PasswordManager.getManager(provider.getKeyRing(), keyAlias).decryptSecret(cipher, iv, aesTransformation);
      }
      catch (AccessControlException e)
      {
        // we don't have permissions for the alias, check if we have permissions for the alternateOwner
        String alternateOwner = getAlternateOwner();
        if (alternateOwner != null)
        {
          try
          {
            SecurityManager sm = System.getSecurityManager();
            if (sm != null)
            {
              sm.checkPermission(new KeyRingPermission(alternateOwner));
            }
            return AccessController.doPrivileged((PrivilegedExceptionAction<SecretBytes>)() -> {
              return Aes256PasswordManager.getManager(provider.getKeyRing(), keyAlias).decryptSecret(cipher, iv, aesTransformation);
            });
          }
          catch (AccessControlException ignored) { }
        }
        throw e;
      }
    }
  }

  @Override
  public boolean validate(SecretBytes secret) throws Exception
  {
    return AccessController.doPrivileged((PrivilegedExceptionAction<Boolean>) () -> {
      try (SecretBytes myBytes = getSecretBytes())
      {
        return SecurityUtil.equals(secret.get(), myBytes.get());
      }
      catch (SecurityException e)
      {
        return false;
      }
      catch (MissingEncodingKeyException meke)
      {
        return false;
      }
    });
  }

  @Override
  public void transcode(String encodedValue, Optional<ISecretBytesSupplier> key)
    throws Exception
  {
    try
    {
      String encodingType = CryptographicAlgorithmBundle.extractName(encodedValue);
      BAbstractAes256PasswordEncoder encoder = (BAbstractAes256PasswordEncoder) BAbstractPasswordEncoder.make(encodingType);
      encoder.parse(encodedValue);

      String[] data = getAlgorithmBundle().decode(encodedValue);
      Objects.requireNonNull(data);

      byte[] newIvBytes = new byte[16];
      new SecureRandom().nextBytes(newIvBytes);
      String newIv = ByteArrayUtil.toHexString(newIvBytes);
      String oldIv = encoder.iv;
      byte[] oldIvBytes = ByteArrayUtil.hexStringToBytes(oldIv);
      String oldCipher = encoder.cipher;
      byte[] oldCipherBytes = ByteArrayUtil.hexStringToBytes(oldCipher);
      byte[] cipherBytes = null;
      keyAlias = encoder.keyAlias;
      // Attempt transcode
      // We have four basic options:
      //  - We use external and they use external
      //  - We use external and they use keyring
      //  - We use keyring and they use exernal
      //  - We use keyring and they use keyring
      // If we both use keyring, no transcode is necessary
      // If we both use the same key, no transcode is necessary
      // NCCB-71465: If we both use keyring, we just want to change the initialization vector.
      if (!key.isPresent() && !usesExternalEncryptionKey())
      {
        try
        {
          cipherBytes = Aes256PasswordManager.transcode(
            oldCipherBytes,
            oldIvBytes,
            newIvBytes,
            getEncryptionKey().get(),
            getEncryptionKey().get(),
            encoder.getAesTransformation(),
            getAesTransformation()
          );
        }
        catch (Exception e)
        {
          // NCCB-71465: In the event that the transcode doesn't work, we fall back to our
          // old behavior: overwrite our stuff and bail
          LOG.log(Level.FINE, "Failed to transcode password, using original cipher data", e);
          cipher = ByteArrayUtil.toHexString(oldCipherBytes);
          iv = oldIv;
          return;
        }
      }
      // NCCB-71465: If we both use the same key, we just want to change the initialization vector.
      else if (usesExternalEncryptionKey() && key.isPresent() && validateExternalEncryptionKey(key))
      {
        try
        {
          cipherBytes = Aes256PasswordManager.transcode(
            oldCipherBytes,
            oldIvBytes,
            newIvBytes,
            key.get().get().get(),
            key.get().get().get(),
            encoder.getAesTransformation(),
            getAesTransformation()
          );
        }
        catch (Exception e)
        {
          // NCCB-71465: In the event that the transcode doesn't work, we fall back to our
          // old behavior: overwrite our stuff and bail
          LOG.log(Level.FINE, "Failed to transcode password, using original cipher data", e);
          cipher = ByteArrayUtil.toHexString(oldCipherBytes);
          iv = oldIv;
          return;
        }
      }
      // We both use keys
      else if (key.isPresent() && usesExternalEncryptionKey() && getEncryptionKey().isPresent())
      {
        cipherBytes = Aes256PasswordManager.transcode(oldCipherBytes,
                                                      oldIvBytes,
                                                      newIvBytes,
                                                      key.get().get().get(),
                                                      getEncryptionKey().get(),
                                                      encoder.getAesTransformation(),
                                                      getAesTransformation());
      }
      // At least one of us uses a keyring
      else
      {
        ISecurityInfoProvider provider = AccessController.doPrivileged((PrivilegedAction<ISecurityInfoProvider>)() ->
          SecurityInitializer.getInstance().getSecurityInfoProvider());

        Aes256PasswordManager manager = Aes256PasswordManager.getManager(provider.getKeyRing(), keyAlias);
        // I use a key, they use a keyring
        if (!key.isPresent() && usesExternalEncryptionKey() && getEncryptionKey().isPresent())
        {
          cipherBytes = manager.transcodeFromKeyring(oldCipherBytes,
                                                     oldIvBytes,
                                                     newIvBytes,
                                                     getEncryptionKey().get(),
                                                     encoder.getAesTransformation(),
                                                     getAesTransformation());
        }
        // I use a keyring, they use a key
        else if (key.isPresent() && !usesExternalEncryptionKey())
        {
          cipherBytes = manager.transcodeToKeyring(oldCipherBytes,
                                                   oldIvBytes,
                                                   newIvBytes,
                                                   key.get().get().get(),
                                                   encoder.getAesTransformation(),
                                                   getAesTransformation());

        }
      }
      if (cipherBytes == null)
      {
        throw new SecurityException();
      }
      cipher = ByteArrayUtil.toHexString(cipherBytes);
      iv = newIv;
    }
    catch (AccessControlException e)
    {
      throw e;
    }
    catch (Exception e)
    {
      throw new SecurityException(e);
    }
  }

  public String getKeyAlias()
  {
    return keyAlias;
  }

  /**
   * Get the alternate owner of this password encoder. Any caller with KeyRingPermission on the alias
   * of the alternate owner is allowed to decrypt the password even if they do not have the KeyRingPermission
   * for the alias that this password is encrypted with.
   *
   * @since Niagara 4.6
   */
  public String getAlternateOwner()
  {
    return null;
  }
  
  protected abstract EncryptionAlgorithmBundle getAlgorithmBundle();

  /**
   * Get the aes transformation to use for this encoder's EncryptionAlgorithmBundle
   *
   * @return the aes transformation
   * @since Niagara 4.9
   */
  private String getAesTransformation()
  {
    EncryptionAlgorithmBundle bundle = getAlgorithmBundle();
    if (bundle instanceof AbstractAesAlgorithmBundle)
    {
      return ((AbstractAesAlgorithmBundle) bundle).getAesTransformation();
    }
    else
    {
      return SecurityConstants.AES_TRANSFORMATION;
    }
  }

////////////////////////////////////////////////////////////////
// Attributes
////////////////////////////////////////////////////////////////
  protected String cipher = null;
  protected String iv = null;
  protected String keyAlias = null;

  private static final Logger LOG = Logger.getLogger("encoder");
}
