/*
 * Copyright 2003 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.security;

import javax.baja.sys.*;
import javax.baja.util.*;

import com.tridium.authn.LoginFailureCause;

/**
 * AuthenticationException is thrown when an operation fails
 * due to invalid authentication credentials.  If an instance
 * of AuthenticationException is created with an AuthenticationRealm,
 * then the application layer can catch this exception and prompt
 * the user to enter the credentials.
 *
 * @author    Brian Frank
 * @creation  28 Jan 03
 * @version   $Revision: 4$ $Date: 2/26/05 3:56:54 PM EST$
 * @since     Baja 1.0
 */
public class AuthenticationException
  extends BajaRuntimeException
  implements Localizable
{

  /**
   * Create with everything.
   */
  public AuthenticationException(String msg, Throwable cause, AuthenticationRealm authenticationRealm)
  {
    super(msg, cause);
    this.authenticationRealm = authenticationRealm;
  }

  /**
   * Create with the specified AuthenticationRealm.
   */
  public AuthenticationException(AuthenticationRealm authenticationRealm)
  {
    this.authenticationRealm = authenticationRealm;
  }

  /**
   * Create with the specified AuthenticationRealm.
   */
  public AuthenticationException(AuthenticationRealm authenticationRealm, Throwable cause)
  {
    super(cause);
    this.authenticationRealm = authenticationRealm;
  }

  /**
   * Create with the specified AuthenticationRealm and LoginFailureCause
   */
  public AuthenticationException(AuthenticationRealm authenticationRealm, LoginFailureCause loginFailureCause)
  {
    super(loginFailureCause.getDefaultFailureMessage());
    this.authenticationRealm = authenticationRealm;
    this.loginFailureCause = loginFailureCause;
  }
    
  /**
   * Create with specified message.
   */
  public AuthenticationException(String msg)
  {
    super(msg);
  }

  /**
   * Create with no message.
   */
  public AuthenticationException()
  {
  }

////////////////////////////////////////////////////////////////
// Methods
////////////////////////////////////////////////////////////////

  public String toString(Context cx)
  {
    return Lexicon.make("baja", cx).getText("AuthenticationException");
  }

  public void setLoginFailureCause(LoginFailureCause loginFailureCause)
  {
    this.loginFailureCause = loginFailureCause;
  }

  public LoginFailureCause getLoginFailureCause()
  {
    return loginFailureCause;
  }
  
  /**
   * If this exception was created with an authentication
   * space, then return it.  Otherwise return null.
   */ 
  public AuthenticationRealm getAuthenticationRealm()
  {
    return authenticationRealm;
  }

////////////////////////////////////////////////////////////////
// Attributes
////////////////////////////////////////////////////////////////  
  
  private AuthenticationRealm authenticationRealm;
  private LoginFailureCause loginFailureCause;
}
