/*
 * Copyright 2003 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.file;

import java.io.*;
import javax.baja.sys.*;
import javax.baja.category.*;
import javax.baja.nav.*;
import javax.baja.space.*;
import javax.baja.security.*;

/**
 * <p>
 * BIFile is the interface implemented by BObjects which
 * have file like semantics.  Some files are directories
 * which organize a BFileSpace into a tree.  Some files
 * contain data which can streamed, read, and written.  A
 * BIFile doesn't necessarily exclude the possibility that
 * a file is both a directory and a data container.  In
 * general directory/data semantics should be defined by
 * implementing BIDataFile and/or BIDirectory.
 * </p>
 * <p>
 * The class hierarchy of BIFile is used to define file
 * content types such as BTextFile, BHtmlFile, etc.  This
 * provides the ability to use the registry for discovery
 * of user agents.
 * </p>
 * <p>
 * Every BIFile has a matching implementation of BIFileStore
 * which provides pluggable functionality for file I/O.  All
 * the methods which are implementated by BIFileStore are also
 * provided on BIFile as a convenience.
 * </p>
 *
 * @author    Brian Frank on 24 Jan 03
 * @version   $Revision: 16$ $Date: 9/17/07 3:23:22 PM EDT$
 * @since     Baja 1.0
 */
public interface BIFile
  extends BINavNode, BISpaceNode, BIComparable, BICategorizable, BIProtected
{

////////////////////////////////////////////////////////////////
// Methods
////////////////////////////////////////////////////////////////

  /**
   * Get the file's backing store.
   */
  BIFileStore getStore();

  /**
   * Set the file's backing store.
   */
  void setStore(BIFileStore store);

  /**
   * Return {@code getStore().getFileSpace()}.
   */
  BFileSpace getFileSpace();

  /**
   * Return {@code getStore().getFilePath()}.
   */
  FilePath getFilePath();

  /**
   * Return {@code getStore().getFileName()}.
   */
  String getFileName();

  /**
   * Return {@code getStore().getExtension()}.
   */
  String getExtension();

  /**
   * Return {@code getStore().isDirectory()}.
   */
  boolean isDirectory();

  /**
   * Get the mime type string for this file.
   */
  String getMimeType();

  /**
   * Return {@code getStore().getSize()}.
   */
  long getSize();

  /**
   * Return {@code getStore().getLastModified()}.
   */
  BAbsTime getLastModified();

  /**
   * Return {@code getStore().isReadonly()}.
   */
  boolean isReadonly();

  /**
   * Return {@code getStore().getInputStream()}.
   */
  InputStream getInputStream()
    throws IOException;

  /**
   * Return {@code getStore().read()}.
   */
  byte[] read()
    throws IOException;

  /**
   * Call {@code getFileSpace().delete(getFilePath())}.
   */
  void delete()
    throws IOException;

  /**
   * Return {@code getStore().getOutputStream()}.
   */
  OutputStream getOutputStream()
    throws IOException;

  /**
   * Call {@code getStore().write()}.
   */
  void write(byte[] content)
    throws IOException;

  /**
   * Return {@code getStore().equals()}.
   */
  boolean equals(Object object);

////////////////////////////////////////////////////////////////
// Type
////////////////////////////////////////////////////////////////

  /**
   * Type for "baja:IFile".
   */
  Type TYPE = Sys.loadType(BIFile.class);

}
