/*
 * Copyright 2000-2001 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.alarm;

import java.util.ArrayList;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.baja.sys.Action;
import javax.baja.sys.BAbsTime;
import javax.baja.sys.BComponent;
import javax.baja.sys.BIcon;
import javax.baja.sys.BLink;
import javax.baja.sys.Context;
import javax.baja.sys.Flags;
import javax.baja.sys.Property;
import javax.baja.sys.ServiceNotFoundException;
import javax.baja.sys.SlotCursor;
import javax.baja.sys.Sys;
import javax.baja.sys.Topic;
import javax.baja.sys.Type;
import javax.baja.util.BCompositeTopic;
import javax.baja.util.BDaysOfWeekBits;
import javax.baja.util.BTimeRange;

/**
 * BAlarmRecipient is the super-class of all alarm recipients
 * in the Baja framework.  Modeled after an entry in the
 * recipient list in a BACnet Notification class.
 *
 * @author    Dan Giorgis
 * @creation  19 Feb 01
 * @version   $Revision: 44$ $Date: 3/10/11 3:10:15 PM EST$
 * @since     Baja 1.0
 */

public abstract class BAlarmRecipient
  extends BComponent
{

  /*-

  class BAlarmRecipient
  {
    properties
    {
      timeRange: BTimeRange
        -- Time during which the recipient will receive alarms.
        default {[ new BTimeRange()]}
      daysOfWeek: BDaysOfWeekBits
        -- Days during which the recipient will receive alarms.
        default {[ BDaysOfWeekBits.DEFAULT ]}
      transitions: BAlarmTransitionBits
        -- Alarm transition types the recipient wishes to receive.
        default {[ BAlarmTransitionBits.ALL ]}
      routeAcks: boolean
        -- Route Alarm Acknowledgements
        default {[ true ]}
    }
    actions
    {
      routeAlarmAck(alarm: BAlarmRecord)
        -- Route an alarm ack request
        flags { hidden }
        default {[ new BAlarmRecord() ]}
      routeAlarm(alarm: BAlarmRecord)
        -- Route an alarm record
        flags { summary }
        default {[ new BAlarmRecord() ]}
    }
    topics
    {
      newUnackedAlarm: BAlarmRecord
        flags { hidden }
      -- Fired when a new unacked alarm is encountered
    }
  }

  -*/
/*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
/*@ $javax.baja.alarm.BAlarmRecipient(988581987)1.0$ @*/
/* Generated Thu Sep 11 10:14:02 EDT 2014 by Slot-o-Matic (c) Tridium, Inc. 2012 */

////////////////////////////////////////////////////////////////
// Property "timeRange"
////////////////////////////////////////////////////////////////

  /**
   * Slot for the {@code timeRange} property.
   * Time during which the recipient will receive alarms.
   * @see #getTimeRange
   * @see #setTimeRange
   */
  public static final Property timeRange = newProperty(0, new BTimeRange(),null);

  /**
   * Get the {@code timeRange} property.
   * Time during which the recipient will receive alarms.
   * @see #timeRange
   */
  public BTimeRange getTimeRange() { return (BTimeRange)get(timeRange); }

  /**
   * Set the {@code timeRange} property.
   * Time during which the recipient will receive alarms.
   * @see #timeRange
   */
  public void setTimeRange(BTimeRange v) { set(timeRange,v,null); }

////////////////////////////////////////////////////////////////
// Property "daysOfWeek"
////////////////////////////////////////////////////////////////

  /**
   * Slot for the {@code daysOfWeek} property.
   * Days during which the recipient will receive alarms.
   * @see #getDaysOfWeek
   * @see #setDaysOfWeek
   */
  public static final Property daysOfWeek = newProperty(0, BDaysOfWeekBits.DEFAULT,null);

  /**
   * Get the {@code daysOfWeek} property.
   * Days during which the recipient will receive alarms.
   * @see #daysOfWeek
   */
  public BDaysOfWeekBits getDaysOfWeek() { return (BDaysOfWeekBits)get(daysOfWeek); }

  /**
   * Set the {@code daysOfWeek} property.
   * Days during which the recipient will receive alarms.
   * @see #daysOfWeek
   */
  public void setDaysOfWeek(BDaysOfWeekBits v) { set(daysOfWeek,v,null); }

////////////////////////////////////////////////////////////////
// Property "transitions"
////////////////////////////////////////////////////////////////

  /**
   * Slot for the {@code transitions} property.
   * Alarm transition types the recipient wishes to receive.
   * @see #getTransitions
   * @see #setTransitions
   */
  public static final Property transitions = newProperty(0, BAlarmTransitionBits.ALL,null);

  /**
   * Get the {@code transitions} property.
   * Alarm transition types the recipient wishes to receive.
   * @see #transitions
   */
  public BAlarmTransitionBits getTransitions() { return (BAlarmTransitionBits)get(transitions); }

  /**
   * Set the {@code transitions} property.
   * Alarm transition types the recipient wishes to receive.
   * @see #transitions
   */
  public void setTransitions(BAlarmTransitionBits v) { set(transitions,v,null); }

////////////////////////////////////////////////////////////////
// Property "routeAcks"
////////////////////////////////////////////////////////////////

  /**
   * Slot for the {@code routeAcks} property.
   * Route Alarm Acknowledgements
   * @see #getRouteAcks
   * @see #setRouteAcks
   */
  public static final Property routeAcks = newProperty(0, true,null);

  /**
   * Get the {@code routeAcks} property.
   * Route Alarm Acknowledgements
   * @see #routeAcks
   */
  public boolean getRouteAcks() { return getBoolean(routeAcks); }

  /**
   * Set the {@code routeAcks} property.
   * Route Alarm Acknowledgements
   * @see #routeAcks
   */
  public void setRouteAcks(boolean v) { setBoolean(routeAcks,v,null); }

////////////////////////////////////////////////////////////////
// Action "routeAlarmAck"
////////////////////////////////////////////////////////////////

  /**
   * Slot for the {@code routeAlarmAck} action.
   * Route an alarm ack request
   * @see #routeAlarmAck(BAlarmRecord alarm)
   */
  public static final Action routeAlarmAck = newAction(Flags.HIDDEN,new BAlarmRecord(),null);

  /**
   * Invoke the {@code routeAlarmAck} action.
   * Route an alarm ack request
   * @see #routeAlarmAck
   */
  public void routeAlarmAck(BAlarmRecord alarm) { invoke(routeAlarmAck,alarm,null); }

////////////////////////////////////////////////////////////////
// Action "routeAlarm"
////////////////////////////////////////////////////////////////

  /**
   * Slot for the {@code routeAlarm} action.
   * Route an alarm record
   * @see #routeAlarm(BAlarmRecord alarm)
   */
  public static final Action routeAlarm = newAction(Flags.SUMMARY,new BAlarmRecord(),null);

  /**
   * Invoke the {@code routeAlarm} action.
   * Route an alarm record
   * @see #routeAlarm
   */
  public void routeAlarm(BAlarmRecord alarm) { invoke(routeAlarm,alarm,null); }

////////////////////////////////////////////////////////////////
// Topic "newUnackedAlarm"
////////////////////////////////////////////////////////////////

  /**
   * Slot for the {@code newUnackedAlarm} topic.
   * @see #fireNewUnackedAlarm
   */
  public static final Topic newUnackedAlarm = newTopic(Flags.HIDDEN,null);

  /**
   * Fire an event for the {@code newUnackedAlarm} topic.
   * @see #newUnackedAlarm
   */
  public void fireNewUnackedAlarm(BAlarmRecord event) { fire(newUnackedAlarm, event, null); }

////////////////////////////////////////////////////////////////
// Type
////////////////////////////////////////////////////////////////

  @Override
  public Type getType() { return TYPE; }
  public static final Type TYPE = Sys.loadType(BAlarmRecipient.class);

/*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

  /**
   * Check the timeRange and transition to see if this alarm should be routed,
   * and if so, send to the handleAlarm method.
   *
   * @param alarmRecord The alarm to route.
   */
  public final void doRouteAlarm(BAlarmRecord alarmRecord)
  {
    if (accept(alarmRecord))
      handleAlarm(alarmRecord);
  }

  /**
   * Route the alarm ack request back to the AlarmService.
   */
  public void doRouteAlarmAck(BAlarmRecord alarmRecord)
  {
    try
    {
      ((BAlarmService)Sys.getService(BAlarmService.TYPE)).doAckAlarm(alarmRecord);
    }
    catch(ServiceNotFoundException e)
    {
      logger.warning("BAlarmRecipient: cannot find AlarmService");
      e.printStackTrace();
    }
    catch(Exception e)
    {
      logger.warning("BAlarmRecipient: cannot resolve alarm source for: " + alarmRecord);
      e.printStackTrace();
    }
  }

  /**
   * Handle the AlarmRecipient specific routing of the alarm.
   */
  public abstract void handleAlarm(BAlarmRecord alarmRecord);

  /**
   * Get the AlarmClasses linked to this AlarmRecipient.
   */
  public String[] getSubscribedAlarmClasses()
  {
    String[] subscribedAlarmClasses = getSubscribedAlarmClasses("alarm");
    return subscribedAlarmClasses;
  }

  /**
   * Get the EscalatedAlarmClasses linked to this AlarmRecipient.
   * Valid levels are 1, 2 and 3.
   */
  public String[] getSubscribedEscalatedAlarmClasses(int level)
  {
    if (level < 1 || level > 3)
      return new String[0];

    return getSubscribedAlarmClasses("escalatedAlarm"+level);
  }

  /**
   * This method returns a list of subscribed alarm classes based on the
   * name of the source slot name of the links of this alarm recipient.
   * @param sourceSlotName The name of the source slot.
   * @return
   */
  private String[] getSubscribedAlarmClasses(String sourceSlotName)
  {
    ArrayList<String> classes = new ArrayList<>();
    SlotCursor<Property> c = this.getProperties();
    while (c.next(BLink.class))
    {
      try
      {
        BLink link = (BLink)c.get();
        if (!link.isActive())
          link.activate();

        BComponent srcComp = link.getSourceComponent();
        if (srcComp == null || srcComp.getName() == null)
        {
          continue;
        }

        if (srcComp instanceof BAlarmClass && link.getSourceSlotName().equals(sourceSlotName))
        {
          classes.add(srcComp.getName());
          // Remember, it's the name of the alarm class, not the alarm class
          // itself.  AlarmRecords store their alarm
          // class as a string, so we need to match that
        }
        else
        {
          //check if we're dealing with nested alarm class folders
          while( srcComp instanceof BIAlarmClassFolder )
          {
            //get source component as an Alarm Class Folder
            BAlarmClassFolder folder = (BAlarmClassFolder)srcComp;
            String folderSlotName = link.getSourceSlotName();
            //get the link to this folder
            link = ((BCompositeTopic)folder.get(folderSlotName)).getMirror().link;
            //check the source component of the link
            srcComp = link.getSourceComponent();
            if( srcComp instanceof BAlarmClass && link.getSourceSlotName().equals(sourceSlotName) )
            {
              classes.add(srcComp.getName());
              break;
            }
          }
        }
      }
      catch (Exception e)
      {
        logger.log(Level.SEVERE, "alarm:BAlarmRecipient:getSubscribedAlarmClasses " + e.getMessage(),e);
      }
    }
    
   return classes.toArray(EMPTY_STRING_ARRAY);
  }

  /**
   * Check to see if the alarm falls within the time and day ranges and the transitions.
   */
  public boolean accept(BAlarmRecord rec)
  {
    BAbsTime now = rec.getTimestamp();

    // Check day of week
    if (!getDaysOfWeek().includes(now.getWeekday()))
      return false;

    // Check time range
    if (!getTimeRange().includes(now))
      return false;

    //only send alarm if transitions are set
    if (!getTransitions().includes(rec.getSourceState()))
      return false;

    //do not send if last transition was an ack.
    /**
     * if !getRouteAcks()
     *   if isAcknoweldged()
     *     if getNormalTime() != null
     *       if getNormalTime().isBefore(getAckTime())
     *         return false;
     *       else continue
     *     else return false
     *   else continue
     * else continue;
     *
     */
    if (!getRouteAcks() && rec.isAcknowledged())
    {
      if (rec.getNormalTime().isNull())
        return false;
      else if (rec.getNormalTime().isBefore(rec.getAckTime()))
        return false;
    }

    return true;
  }

  @Override
  public String toString(Context cx)
  {
    if(isMounted())
      return getName();
    else
      return TYPE.getTypeName();
  }

  private static final Logger logger = Logger.getLogger("alarm");

  /**
   * Get the icon.
   */
  @Override
  public BIcon getIcon() { return icon; }
  private static final BIcon icon = BIcon.make("module://icons/x16/alarm/alarmRecipient.png");
  private static final String[] EMPTY_STRING_ARRAY = new String[0];
}
