/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 */

/**
 * API Status: **Private**
 * @module nmodule/alarm/rc/fe/EnumAlarmRangeEditor
 */
define(['baja!',
  'baja!alarm:AlarmAlgorithm,alarm:AlarmSourceExt,control:ControlPoint',
  'bajaux/Widget',
  'bajaux/commands/Command',
  'bajaux/util/CommandButtonGroup',
  'jquery',
  'Promise',
  'underscore',
  'nmodule/alarm/rc/util/alarmUtils',
  'nmodule/webEditors/rc/fe/CompositeEditor',
  'nmodule/webEditors/rc/fe/fe',
  'nmodule/webEditors/rc/fe/feDialogs',
  'nmodule/webEditors/rc/fe/baja/DisplayOnlyEditor',
  'nmodule/webEditors/rc/fe/baja/util/rangeUtils',
  'nmodule/webEditors/rc/fe/config/CompositeBuilder',
  'css!nmodule/alarm/rc/fe/alarmEditors'], function (baja,
                                                     types,
                                                     Widget,
                                                     Command,
                                                     CommandButtonGroup,
                                                     $,
                                                     Promise,
                                                     _,
                                                     alarmUtils,
                                                     CompositeEditor,
                                                     fe,
                                                     feDialogs,
                                                     DisplayOnlyEditor,
                                                     rangeUtils,
                                                     CompositeBuilder) {

  'use strict';

  var getPointFacets      = alarmUtils.getPointFacets,
      getEnumRangeDisplay = rangeUtils.getEnumRangeDisplay;

////////////////////////////////////////////////////////////////
// Support functions
////////////////////////////////////////////////////////////////

  //TODO: type ext
  /**
   * Horrible way of getting around the absence of the overridden
   * `getSlotFacets` in `BEnumChangeOfStateAlgorithm` and `BEnumFaultAlgorithm`.
   *
   * @inner
   * @param {baja.Complex} [algorithm]
   * @returns {Promise} promise to be resolved with parent point facets, or
   * `null` if could not be found
   */
  function getRangeFacet(algorithm) {
    return getPointFacets(algorithm)
      .then(function (facets) {
        return facets.get('range');
      });
  }

  function isNull(enumRange) {
    return !enumRange || enumRange.equals(baja.EnumRange.DEFAULT);
  }

////////////////////////////////////////////////////////////////
// Commands
////////////////////////////////////////////////////////////////

  var AddCommand = function AddCommand(ed) {
    Command.call(this, {
      module: 'alarm',
      lex: 'alarm.enumRangeFE.add',
      func: function () {
        return ed.read()
          .then(function (range) {
            var max = Math.max.apply(Math, range.getOrdinals()) + 1;
            return feDialogs.showFor({ value: baja.Integer.make(max), formFactor: 'mini' })
              .then(function (num) {
                if (num === null) {
                  return;
                }
                var ordinals = range.getOrdinals().concat([num]);
                return ed.doLoad(baja.EnumRange.make({
                  ordinals: ordinals,
                  tags: _.map(_.map(ordinals, String), baja.SlotPath.escape)
                }));
              });
          });
      }
    });
  };
  AddCommand.prototype = Object.create(Command.prototype);
  AddCommand.prototype.constructor = AddCommand;


////////////////////////////////////////////////////////////////
// EnumAlarmRangeEditor
////////////////////////////////////////////////////////////////

  /**
   * Editor for handling `EnumRange` values on `Enum`-based alarm algorithms.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/CompositeEditor
   * @alias module:nmodule/alarm/rc/fe/EnumAlarmRangeEditor
   */
  var EnumAlarmRangeEditor = function EnumAlarmRangeEditor(params) {
    if (params.formFactor === Widget.formfactor.mini) {
      DisplayOnlyEditor.apply(this, arguments);
      this.$makeDisplayOnly();
    } else {
      CompositeEditor.apply(this, arguments);
      this.getCommandGroup().add(new AddCommand(this));
    }
  };
  EnumAlarmRangeEditor.prototype = Object.create(CompositeEditor.prototype);
  EnumAlarmRangeEditor.prototype.constructor = EnumAlarmRangeEditor;

  /**
   * Get the command to add a new ordinal.
   *
   * @private
   * @returns {module:bajaux/commands/Command}
   */
  EnumAlarmRangeEditor.prototype.$getAddCommand = function () {
    return this.getCommandGroup().get(0);
  };

  /**
   * Make the editor display-only when in mini mode.
   */
  EnumAlarmRangeEditor.prototype.$makeDisplayOnly = function () {
    DisplayOnlyEditor.$mixin(this);
  };

  /**
   * Get the element containing the Add command button.
   *
   * @private
   * @returns {jQuery}
   */
  EnumAlarmRangeEditor.prototype.$getAddButtonElement = function () {
    return this.jq().children('.addButton');
  };

  /**
   * Get the element containing all boolean checkboxes.
   *
   * @private
   * @returns {jQuery}
   */
  EnumAlarmRangeEditor.prototype.$getEditorsElement = function () {
    return this.jq().children('.editors');
  };

  /**
   * Get the "base" enum range: the range from the algorithm's containing
   * EnumPoint. In Workbench world, this is returned to us by the overridden
   * `getSlotFacets` method on the algorithm class; since we don't have that
   * in BajaScript, we hack it.
   *
   * The retrieved range
   *
   * @private
   * @returns {Promise} promise to be resolved with a `baja:EnumRange`
   */
  EnumAlarmRangeEditor.prototype.$getBaseRange = function () {
    var that  = this,
        range = that.properties().getValue('range');

    if (!range) {
      return getRangeFacet(this.getComplex())
        .then(function (range) {
          range = range || baja.EnumRange.DEFAULT;

          that.properties().add('range', range);
          //TODO: ugh. PropertySheetRow should listen for property change events
          //the popout editor will wind up getting the facets from the
          //row, not the value editor, so we must propagate this up
          var row = that.jq().closest('.PropertySheetRow');
          if (row.length) {
            row.data('widget').properties().add('range', range);
          }

          return range;
        });
    }

    return Promise.resolve(range);
  };

  /**
   * When the editor is in display-only mode, just show a comma-separated list
   * of active enum values.
   *
   * @param {baja.EnumRange} enumRange
   * @returns {Promise} promise to be resolved with display string
   */
  EnumAlarmRangeEditor.prototype.valueToString = function (enumRange) {
    var ordinals = enumRange.getOrdinals(),
        displays = _.map(ordinals, function (o) {
          return getEnumRangeDisplay(o, enumRange);
        });

    return Promise.join(
      Promise.all(displays),
      this.$getBaseRange() //must be called to pass property to popout editor
    )
      .spread(function (tags) {
        return tags.join(', ');
      });
  };

  /**
   * The builder for an `EnumAlarmRangeEditor` will show one boolean editor for
   * each ordinal in our active range. The display string will either come
   * from the base range from the containing point, or will just be the ordinal
   * itself if no base range is present.
   *
   * @returns {module:nmodule/webEditors/rc/fe/config/CompositeBuilder}
   */
  EnumAlarmRangeEditor.prototype.makeBuilder = function () {
    var that    = this,
        builder = new CompositeBuilder();

    builder.getKeys = function () {
      return that.$getBaseRange()
        .then(function (range) {
          if (isNull(range)) {
            range = builder.getDataSource();
          }
          return _.map(range.getOrdinals(), String);
        });
    };

    builder.getDomFor = function (key) {
      return $('<div data-ordinal="' + key + '"/>')
        .appendTo(that.$getEditorsElement());
    };

    builder.getConfigFor = function (key) {
      return that.$getBaseRange()
        .then(function (range) {
          return Promise.resolve(isNull(range) ?
              key :
              getEnumRangeDisplay(parseInt(key, 10), range)
          )
            .then(function (display) {
              return {
                properties: { trueText: display, falseText: display },
                value: false,
                formFactor: 'mini'
              };
            });
        });
    };

    builder.getValueFor = function (key) {
      return !!this.getDataSource().isOrdinal(parseInt(key, 10));
    };

    return builder;
  };

  /**
   * Initializes a command button for the "add ordinal" command.
   *
   * @param {JQuery} dom
   * @returns {Promise}
   */
  EnumAlarmRangeEditor.prototype.doInitialize = function (dom) {
    dom.addClass('EnumAlarmRangeEditor');
    dom.html('<div class="editors"/><div class="addButton editor"/>');
    return Promise.join(
      CompositeEditor.prototype.doInitialize.apply(this, arguments),
      fe.buildFor({
        dom: dom.children('.addButton'),
        type: CommandButtonGroup,
        value: this.getCommandGroup()
      })
    );
  };

  /**
   * Loads in the enum range. If a base range is present, the "add ordinal"
   * button will be hidden.
   *
   * @param {baja.EnumRange} enumRange
   * @returns {Promise}
   */
  EnumAlarmRangeEditor.prototype.doLoad = function () {
    var that = this;
    return Promise.join(
      that.$getBaseRange(),
      CompositeEditor.prototype.doLoad.apply(that, arguments)
    )
      .spread(function (baseRange) {
        if (baseRange && !baseRange.equals(baja.EnumRange.DEFAULT)) {
          that.$getAddButtonElement().hide();
        }
      });
  };

  /**
   * Read the individual boolean editors and assemble them into an `EnumRange`
   * containing all selected ordinals.
   *
   * @returns {Promise} promise to be resolved with a `baja:EnumRange` value
   */
  EnumAlarmRangeEditor.prototype.doRead = function () {
    var kids = this.getChildEditors(this.$getEditorsElement());
    return Promise.join(kids.readAll(), this.$getBaseRange())
      .spread(function (booleans, range) {
        var ordinals = [], tags = [];
        _.each(booleans, function (bool, i) {
          if (bool) {
            var o   = parseInt(kids[i].jq().data('ordinal'), 10),
                tag = range.getTag(o);
            if (!range.isOrdinal(o)) { //we just got the ordinal back from getTag
              tag = baja.SlotPath.escape(tag);
            }
            ordinals.push(o);
            tags.push(tag);
          }
        });
        return baja.EnumRange.make({ ordinals: ordinals, tags: tags });
      });
  };

  /**
   * Destroy all child editors.
   *
   * @returns {Promise}
   */
  EnumAlarmRangeEditor.prototype.doDestroy = function () {
    this.jq().removeClass('EnumAlarmRangeEditor');
    return this.getChildWidgets().destroyAll();
  };

  /**
   * Set all child editors readonly/writable.
   *
   * @param {Boolean} readonly
   * @returns {Promise}
   */
  EnumAlarmRangeEditor.prototype.doReadonly = function (readonly) {
    return this.getChildWidgets().setAllReadonly(readonly);
  };

  /**
   * Set all child editors enabled/disabled.
   *
   * @param {Boolean} enabled
   * @returns {Promise}
   */
  EnumAlarmRangeEditor.prototype.doEnabled = function (enabled) {
    return this.getChildWidgets().setAllEnabled(enabled);
  };

  return (EnumAlarmRangeEditor);
});
